/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** ===========================================================================
 *   @file  csl_aifHwControl.c
 *
 *   @path  $(CSLPATH)\src\aif
 *
 *   @desc Antenna Interface
 *
 */

/* =============================================================================
 * Revision History
 * ===============
 *  26-Oct-2004 Pd File Created.
 *  21-Feb-2006 Naveen Shetti updated the file for DOXYGEN compatibiliy
 *
 * =============================================================================
 */

#include <csl_aif.h>
#include <csl_aifHwControlAux.h>

/** ===========================================================================
 *   @n@b CSL_aifHwControl
 *
 *   @b Description
 *   @n This function performs various control operations on aif link,
 *      based on the command passed.
 *
 *   @b Arguments
 *   @verbatim
            haif        Handle to the aif instance
 
            cmd         Operation to be performed on the aif
 
            arg         Argument specific to the command 
 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK            - Command execution successful.
 *   @li                    CSL_ESYS_BADHANDLE - Invalid handle
 *   @li                    CSL_ESYS_INVCMD    - Invalid command
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Registers of aif instance are configured according to the command
 *       and the command arguments. The command determines which registers are
 *       modified.
 *
 *   @b Modifies
 *   @n Registers determined by the command
 *
 *   @b Example
 *   @verbatim
        // handle for link 0
        CSL_AifHandle handleAifLink0;
        // other link related declarations
        ...
        // ctrl argument for hw command
        Bool ctrlArg;

        // Open handle for link 0 - for use 
        handleAifLink0 = CSL_aifOpen(&AifLinkObj0, CSL_AIF, &aifParam, &status);

        if ((handleAifLink0 == NULL) || (status != CSL_SOK)) 
        {
           printf ("\nError opening CSL_AIF");
           exit(1);
        }

        // Do config for link 0 
        ConfigLink0.globalSetup = &gblCfg;
        ...
        //Do setup for link - 0
        CSL_aifHwSetup(handleAifLink0, &ConfigLink0);

        ctrlArg = CSL_AIF_CTRL_RX_LINK_ENABLE;

        // Send hw control command to enable Tx/Rx of link 0 
        CSL_aifHwControl(handleAifLink0, CSL_AIF_CMD_ENABLE_DISABLE_RX_LINK, (void *)&ctrlArg);
     @endverbatim
 * ============================================================================
 */
#pragma CODE_SECTION (CSL_aifHwControl, ".text:csl_section:aif");
CSL_Status  CSL_aifHwControl(
    CSL_AifHandle          hAifLink,
    CSL_AifHwControlCmd    cmd,
    void                   *arg
)
{
    CSL_Status status = CSL_SOK ;      
    
    if (hAifLink == NULL)
        return CSL_ESYS_BADHANDLE;
    
    switch (cmd) {
        
    /** Starts Rx link */
    case CSL_AIF_CMD_ENABLE_DISABLE_RX_LINK:
        CSL_aifEnDisRxLink (hAifLink, *(Bool *)arg);
        break;
    /** Starts Tx link */    
    case CSL_AIF_CMD_ENABLE_DISABLE_TX_LINK:
        CSL_aifEnDisTxLink(hAifLink, *(Bool *)arg);
        break;
    /** Enable SD0 PLL */   
    case CSL_AIF_CMD_ENABLE_DISABLE_SD0_PLL:
        CSL_aifEnDisSd0Pll(hAifLink, *(Bool *)arg);
        break;
    /** Select SD0 PLL multiply factor */
    case CSL_AIF_CMD_MOD_SD0_PLL_MUL_FACT:
        CSL_aifSd0PllMulFact(hAifLink, *(Uint8 *)arg);
        break;
    /** Enable SD1 PLL */   
    case CSL_AIF_CMD_ENABLE_DISABLE_SD1_PLL:
        CSL_aifEnDisSd1Pll(hAifLink, *(Bool *)arg);
        break;
    /** Select SD1 PLL multiply factor */
    case CSL_AIF_CMD_MOD_SD1_PLL_MUL_FACT:
        CSL_aifSd1PllMulFact(hAifLink, *(Uint8 *)arg);
        break;
    /** Force receiver state machine state */  
    case CSL_AIF_CMD_MOD_RM_LINK_RX_STATE:
        CSL_aifRmLinkRxState(hAifLink, *(CSL_AifRxSetSyncState *)arg);
        break;
    /** Modify measurement window for valid Master Frame Offset (argument type: Int16 * ) */
    case CSL_AIF_CMD_MOD_RM_LINK_VALID_MSTR_FRAME_WIND:
        CSL_aifRmLinkValidMstrFrmWind(hAifLink, *(Int16 *)arg);
        break;
    /** Modify the measurement window for received Master Frame Offset (argument type: Int16 * )*/
    case CSL_AIF_CMD_MOD_RM_LINK_RCVD_MSTR_FRAME_WIND:
        CSL_aifRmLinkRcvdMstrFrmWind(hAifLink, *(Int16 *)arg);
        break;
    /** Modify the Pi Offset of Rx link (argument type: Int16 * ) */
    case CSL_AIF_CMD_MOD_RM_LINK_PI_OFFSET:
        CSL_aifRmLinkPiOffset(hAifLink, *(Int16 *)arg);
        break;
    /** Sets 8b10b los detect threshold value (argument type: Uint32 * ) */
    case CSL_AIF_CMD_SET_RM_LINK_LOS_DET_THOLD:
        CSL_aifRmLinkLosDetThold(hAifLink, *(Uint32 *)arg);
        break;
    /** Suppress error reporting when the receiver state machine is not in state ST3 (argument type: Uint8 * ) */
    case CSL_AIF_CMD_ENABLE_DISABLE_RM_LINK_ERR_SUPPRESS:
        CSL_aifEnDisRmLinkErrSuppress(hAifLink, *(Bool *)arg);
        break;
    /** Suppress error reporting  of "extra" K28.7 characters detected in the data stream (argument type: Uint8 * ) */
    case CSL_AIF_CMD_ENABLE_DISABLE_RM_LINK_EXTRA_K28_7_ERR_SUPRESS:
        CSL_aifEnDisRmLinkExtraErrSuppress(hAifLink, *(Bool *)arg);
        break;
    /** Modify Threshold value for consecutive valid blocks of bytes which result in state ST1 (argument type: Uint16 * ) */
    case CSL_AIF_CMD_MOD_RM_SYNC_THRESH:
        CSL_aifRmSyncThresh(hAifLink, *(Uint16 *)arg);
        break;
    /** Modify Threshold value for consecutive valid message groups which result in state ST3 (argument type: Uint16 * ) */
    case CSL_AIF_CMD_MOD_RM_FRAME_SYNC_THRESH:
        CSL_aifRmFrameSyncThresh(hAifLink, *(Uint16 *)arg);
        break;
    /** Modify Threshold value for consecutive invalid blocks of bytes which result in state ST0 (argument type: Uint16 * )*/
    case CSL_AIF_CMD_MOD_RM_UNSYNC_THRESH:
        CSL_aifRmUnSyncThresh(hAifLink, *(Uint16 *)arg);
        break;
    /** Modify Threshold value for consecutive invalid message groups which result in state ST1 (argument type: Uint16 * )*/
    case CSL_AIF_CMD_MOD_RM_FRAME_UNSYNC_THRESH:
        CSL_aifRmFrameUnSyncThresh(hAifLink, *(Uint16 *)arg);
        break;
    /** Modify transmitter state machine state */
    case CSL_AIF_CMD_MOD_TM_LINK_TX_STATE:
        CSL_aifTmLinkTxState(hAifLink, *(CSL_AifTxSetSyncState *)arg);
        break;
    /** Enable allowing the TX FSM from entering state ST2 (argument type: Uint8 * ) */
    case CSL_AIF_CMD_ENABLE_DISABLE_TM_LINK_XMIT:
        CSL_aifEnDisTmLinkXmit(hAifLink, *(Bool *)arg);
        break;
    /** Enable  impact of the rm_loss_of_signal to the transmitter state machine (argument type: Uint8 * ) */
    case CSL_AIF_CMD_ENABLE_DISABLE_TM_LINK_LOS:
        CSL_aifEnDisTmLinkLos(hAifLink, *(Bool *)arg);
        break;
    /** Modify the threshold for the full indication flag for the TX MAC  FIFO (argument type: Uint8 * ) */
    case CSL_AIF_CMD_MOD_TM_LINK_FIFO_FULL_THOLD:
        CSL_aifTmLinkFifoFullThold(hAifLink, *(Uint8 *)arg);
        break;
    /** Modify the Delta Offset of Tx link(argument type: Uint16 *) */
    case CSL_AIF_CMD_MOD_TM_LINK_DELTA_OFFSET:
        CSL_aifTmLinkDeltaOffset(hAifLink, *(Uint16 *)arg);
        break;
    /** Transmit Node B Frame number low byte  (Z.128.0) - loads u_cntr low byte (argument type: Uint8 * )*/
    case CSL_AIF_CMD_SET_TM_LINK_BFN_LOW:
        CSL_aifSetTmLinkBfnLow(hAifLink, *(Uint8 *)arg);
        break;
    /** Transmit Node B Frame number high byte  (Z.130.0) - loads u_cntr high byte (argument type: Uint8 * )*/
    case CSL_AIF_CMD_SET_TM_LINK_BFN_HIGH:
        CSL_aifSetTmLinkBfnHigh(hAifLink, *(Uint8 *)arg);
        break;
    /** Enable Combiner0 (argument type: Uint8 *) */
    case CSL_AIF_CMD_ENABLE_DISABLE_CB0:
        CSL_aifEnDisCb0(hAifLink, *(Bool *)arg);
        break;
    /** Enable Combiner1 (argument type: Uint8 *) */
    case CSL_AIF_CMD_ENABLE_DISABLE_CB1:
        CSL_aifEnDisCb1(hAifLink, *(Bool *)arg);
        break;
    /** Enable Decombiner0 (argument type: Uint8 *) */
    case CSL_AIF_CMD_ENABLE_DISABLE_DC0:
        CSL_aifEnDisDc0(hAifLink, *(Bool *)arg);
        break;
    /** Enable Decombiner1 (argument type: Uint8 *) */
    case CSL_AIF_CMD_ENABLE_DISABLE_DC1:
        CSL_aifEnDisDc1(hAifLink, *(Bool *)arg);
        break;
    /** Modify Combiner0 offset from frame sync (argument type: Int16 *) */
    case CSL_AIF_CMD_MOD_CB0_FS_OFFSET:
        CSL_aifCb0FsOffset(hAifLink, *(Uint16 *)arg);
        break;
    /** Modify Combiner1 offset from frame sync (argument type: Int16 *) */
    case CSL_AIF_CMD_MOD_CB1_FS_OFFSET:
        CSL_aifCb1FsOffset(hAifLink, *(Uint16 *)arg);
        break;
    /** Modify Combiner0 valid data window for Master Frame Offset (argument type: Uint8 *) */
    case CSL_AIF_CMD_MOD_CB0_VALID_DATA_WIND:
        CSL_aifCb0ValidDataWind(hAifLink, *(Uint8 *)arg);
        break;
    /** Modify Combiner1 valid data window for Master Frame Offset (argument type: Uint8 *) */
    case CSL_AIF_CMD_MOD_CB1_VALID_DATA_WIND:
        CSL_aifCb1ValidDataWind(hAifLink, *(Uint8 *)arg);
        break;
    /** Modify the Aggregator Mode (argument type: Uint8 *)*/
    case CSL_AIF_CMD_MOD_AG_LINK_MODE:
        CSL_aifAgLinkMode(hAifLink, *(Uint8 *)arg);
        break;
    /** Aggregator error checking enable (argument type: Uint8 *)*/
    case CSL_AIF_CMD_ENABLE_AG_LINK_ERR_CHECK:
        CSL_aifEnableAgLinkErrCheck(hAifLink, *(Uint8 *)arg);
        break;
    /** Controls the data source selected by the Aggregator when there is an OBSAI header 
     *  mismatch and the PE is requesting aggregation (argument type: Uint8 *)
     */
    case CSL_AIF_CMD_SET_AG_LINK_HDR_ERR_SEL:
        CSL_aifSetAgLinkHdrErrSel(hAifLink, *(Uint8 *)arg);
        break;
    /** Enable PD link */
    case CSL_AIF_CMD_PD_ENABLE_DISABLE_LINK:
        CSL_aifPdEnDisLink(hAifLink, *(Bool *)arg);
        break;
    /** Enables CPRI control word capture (argument type: Uint8 *) */
    case CSL_AIF_CMD_PD_ENABLE_DISABLE_LINK_CPRI_CTRL_CAPT:
        CSL_aifPdEnDisLinkCpriCtrlCapt(hAifLink, *(Bool *)arg);
        break;
    /** Modify Time Stamp to increment, Incremented value is used for next message (argument type: Uint32 *) */
    case CSL_AIF_CMD_PD_MOD_LINK_84CNT_TS_INCR_LUT0:
        CSL_aifPdLink84CntTsIncrLut0(hAifLink, *(Uint32 *)arg);
        break;
    /** Modify Time Stamp to increment, Incremented value is used for next message (argument type: Uint32 *) */
    case CSL_AIF_CMD_PD_MOD_LINK_84CNT_TS_INCR_LUT1:
        CSL_aifPdLink84CntTsIncrLut1(hAifLink, *(Uint32 *)arg);
        break;
    /** Modify Time Stamp to increment, Incremented value is used for next message (argument type: Uint32 *) */
    case CSL_AIF_CMD_PD_MOD_LINK_84CNT_TS_INCR_LUT2:
        CSL_aifPdLink84CntTsIncrLut2(hAifLink, *(Uint32 *)arg);
        break;
    /** Reset the cir,pkt and err PD type lut*/
    case CSL_AIF_CMD_PD_RESET_TYPE_LUT:
        CSL_aifPdResetTypeLut(hAifLink);
        break;
    /** Reset PD addr lut */
    case CSL_AIF_CMD_PD_RESET_ADR_LUT:
        CSL_aifPdResetAdrLut(hAifLink);
        break;
    /** Enable the PE link (argument type: Bool *)*/
    case CSL_AIF_CMD_PE_ENABLE_DISABLE_LINK:
        CSL_aifPeEnDisLink(hAifLink, *(Bool *)arg);
    /** Enable of a particular message slot (arg is pointer to array of 84)*/
    case CSL_AIF_CMD_PE_MOD_LINK_84CNT_LUT:
        CSL_aifPeLink84CntLut(hAifLink,  (Uint32*) arg);
        break;
    /** Disable all message slot (arg is pointer to array of 84)*/
    case CSL_AIF_CMD_PE_RESET_LINK_84CNT_LUT:
        CSL_aifPeResetLink84CntLut(hAifLink);
        break;
    /** Modify PE Link 21 Count Identity Look Up Table0 */
    case CSL_AIF_CMD_PE_MOD_LINK_21CNT_ID_LUT0:
        CSL_aifPeLink21CntIdLut0(hAifLink,  (Uint32*) arg);
        break;
    /** Reset PE Link 21 Count Identity Look Up Table0 */
    case CSL_AIF_CMD_PE_RESET_LINK_21CNT_ID_LUT0:
        CSL_aifPeResetLink21CntIdLut0(hAifLink);
        break;
    /** Modify PE Link 21 Count Identity Look Up Table0 */
    case CSL_AIF_CMD_PE_MOD_LINK_21CNT_ID_LUT1:
        CSL_aifPeLink21CntIdLut1(hAifLink,  (Uint32*) arg);
        break;
    /** Reset PE Link 21 Count Identity Look Up Table0 */
    case CSL_AIF_CMD_PE_RESET_LINK_21CNT_ID_LUT1:
        CSL_aifPeResetLink21CntIdLut1(hAifLink);
        break;
    /** Flushes all outbound pktsw fifo's if a memory leak is detected (argument type: Bool *) */
    case CSL_AIF_CMD_DB_ENABLE_MEM_LEAK_FLUSH:
        CSL_aifDbEnMemLeakFlsh(hAifLink, *(Bool*)arg);
        break;
    /** Enables "Logic Analyzer" capture (argument type: Uint8 *) */ 
    case CSL_AIF_CMD_DB_ENABLE_LINK_CAPTURE:
        CSL_aifDbEnLinkCapture(hAifLink, *(Bool*)arg);
        break;
    /** Enables Packet Switched Outbound FIFOs per link (argument type: Uint8 *) 
     */
    case CSL_AIF_CMD_DB_OUT_ENABLE_PKTSW_FIFO:
        CSL_aifDbOutEnPktswFifo(hAifLink, *(Uint8*)arg);
        break;
    /** Flush Fifo, Fifo Pointers are cleared, FIFO will still be enabled (argument type: Uint8 *) */
    case CSL_AIF_CMD_DB_OUT_ENABLE_PKTSW_FIFO_FLUSH:
        CSL_aifDbOutEnPktswFifoFlsh(hAifLink, *(Uint8*)arg);
        break;
    /** Enables the Exception Event Link mask (argument type: CSL_AifExcEventLinkMaskObj *) */
    case CSL_AIF_CMD_EE_ENABLE_LINK_MASK:
        CSL_aifEeEnLinkMask(hAifLink, *(CSL_AifExcEventLinkMaskObj*)arg);
        break;
    /** Disables the Exception Event Link mask (argument type: CSL_AifExcEventLinkMaskObj *) */
    case CSL_AIF_CMD_EE_DISABLE_LINK_MASK:
        CSL_aifEeDisLinkMask(hAifLink, *(CSL_AifExcEventLinkMaskObj*)arg);
        break;
    /** Clears the Exception Link Event  (argument type: CSL_AifExcEventLinkClearObj *) */
    case CSL_AIF_CMD_EE_CLEAR_LINK_EVENT:
        CSL_aifEeClrLinkEvt(hAifLink, *(CSL_AifExcEventLinkClearObj*)arg);
        break;
    /** Enables the Exception Event Common mask (argument type: CSL_AifExcEventCmnMaskObj *) */
    case CSL_AIF_CMD_EE_ENABLE_COMMON_MASK:
        CSL_aifEeEnCommonMask(hAifLink, *(CSL_AifExcEventCmnMaskObj*)arg);
        break;
    /** Disables the Exception Event Common mask (argument type: CSL_AifExcEventCmnMaskObj *) */
    case CSL_AIF_CMD_EE_DISABLE_COMMON_MASK:
        CSL_aifEeDisCommonMask(hAifLink, *(CSL_AifExcEventCmnMaskObj*)arg);
        break;
    /** Clears the Exception Common Event (argument type: CSL_AifExcEventCmnMaskObj *) */
    case CSL_AIF_CMD_EE_CLEAR_COMMON_EVENT:
        CSL_aifEeClrCommonEvt(hAifLink, *(CSL_AifExcEventCmnMaskObj*)arg);
        break;
    /** Whenever this register is written, the ai_event[3:0] signal addressed by 
    *  this field is allowed to generate another event if the any aggregated 
    *  bits in the associated Interrupt Source Masked Status Registers are set 
    *  to a 1 (argument type: CSL_AifExcEventIndex *)
    */
    case CSL_AIF_CMD_EE_REEVALUATE_INTERRUT_LINE:
        CSL_aifEeRevalIntLine(hAifLink, *(CSL_AifExcEventIndex*)arg);
        break;
    /** Set the Event mode error alarm/trigger (argument type: CSL_AifExcEventMode *) */
    case CSL_AIF_CMD_EE_SET_EVENT_MODE:
        CSL_aifEeSetEvtMode(hAifLink, *(CSL_AifExcEventMode*)arg);
        break;
    /** Sets the AIF run status bit (argument type: NULL) */
    case CSL_AIF_CMD_EE_SET_AIF_RUN_STATUS_BIT:
        CSL_aifEeSetAifRunStatBit(hAifLink);
        break;
    /** Clears the AIF run status bit (argument type: NULL) */
    case CSL_AIF_CMD_EE_CLEAR_AIF_RUN_STATUS_BIT:
        CSL_aifEeClrAifRunStatBit(hAifLink);
        break;
    /** Sets Link select bits for the AIF modules (argument type: CSL_AifExcLinkSelect *)  */
    case CSL_AIF_CMD_EE_SET_LINK_SELECT:
        CSL_aifEeSetLinkSel(hAifLink,*(CSL_AifExcLinkSelect *)arg);
        break;
    /** Enables/disables auto-resync of PE-DMA and PD-FS tracker for a link (argument type: Bool *)  */
    case CSL_AIF_CMD_DB_OUT_ENABLE_DISABLE_LINK_RESYNC:
        CSL_aifDbOutEnDisLinkResync(hAifLink, *(Bool *)arg);

    /** Sets sync value for PE-DMA and PD-FS tracker for a link (argument type: Uint8 *)  */
    case CSL_AIF_CMD_DB_OUT_SET_LINK_SYNC_VALUE:
        CSL_aifDbOutSetLinkSyncVal(hAifLink, *(Uint8 *)arg);
    
    /** enables/disables auto-resync of PD-DMA trackers (argument type: Bool *)  */
    case CSL_AIF_CMD_DB_IN_ENABLE_DISABLE_LINK_RESYNC:
        CSL_aifDbInEnDisLinkResync(hAifLink, *(Bool *)arg);

    /** Sets sync value for PD-DMA tracker for a link (argument type: Uint8 *)  */
    case CSL_AIF_CMD_DB_IN_SET_LINK_SYNC_VALUE:
        CSL_aifDbInSetLinkSyncVal(hAifLink, *(Uint8 *)arg);

    default:
            status = CSL_ESYS_INVCMD;
    }
    
    return status;
}

