/*  ===========================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in
 *   the license agreement under which this software has been supplied.
 *   ==========================================================================
 */

/** ===========================================================================
 *   @file  csl_aifHwSetup.c
 *
 *   @path  $(CSLPATH)\src\aif
 *
 *   @desc  AIF CSL implementation
 *
 */
 
/* =============================================================================
 * Revision History
 * ===============
 *  26-Oct-2004 Pd File Created.
 *  21-Feb-2006 Naveen Shetti updated the file for DOXYGEN compatibiliy
 *
 * =============================================================================
 */

#include <csl_aif.h>
#include <csl_aifHwSetupAux.h>

/** ============================================================================
 *   @n@b CSL_aifHwSetup
 *
 *   @b Description       
 *   @n It configures the AIF instance registers as per the values passed
 *      in the hardware setup structure.
 *
 *   @b Arguments
 *   @verbatim
            hAifLink        Handle to the AIF instance

            linkSetup       Pointer to hardware setup structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Hardware setup successful.
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Hardware structure is not
 *                                                properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  Both @a CSL_aifInit() and @a CSL_aifOpen() must be called successfully
 *   in that order before CSL_aifHwSetup() can be called. The user has to
 *   allocate space for & fill in the main setup structure appropriately before
 *   calling this function.
 *
 *   <b> Post Condition </b>
 *   @n  The specified instance will be setup according to value passed.
 *
 *   @b Modifies
 *   @n The hardware registers of AIF.
 *
 *   @b Example                  
 *   @verbatim
          // handle for link 0
          CSL_AifHandle handleAifLink0;
          //  link object for link 0
          CSL_AifLinkObj AifLinkObj0;
          //AIF module specific parameters
          CSL_AifParam  aifParam;
          // CSL status
          CSL_Status status;
          // global config for AIF 
          CSL_AifGlobalSetup gblCfg = {};

          // Setup objects for configuring link 0 
          CSL_AifLinkSetup ConfigLink0;
          // Setup for common params for link 0 
          CSL_AifCommonLinkSetup commoncfg0= {CSL_AIF_LINK_0};
          // Setup for inbound for link 0
          CSL_AifInboundLinkSetup inboundCfg0 = {};

          aifParam.LinkIndex = CSL_AIF_LINK_0; // set AIF module specific params
   
          // Open handle for link 0 - for use 
          handleAifLink0 = CSL_aifOpen(&AifLinkObj0, CSL_AIF, &aifParam, &status);

          if ((handleAifLink0 == NULL) || (status != CSL_SOK)) 
          {
             printf ("\nError opening CSL_AIF");
             exit(1);
          }

          // Do config for link 0 
          ConfigLink0.globalSetup = &gblCfg;
          ConfigLink0.commonlinkSetup = &commoncfg0;
          ConfigLink0.outboundlinkSetup = NULL; // no outbound link setup for now
          ConfigLink0.inboundlinkSetup = &inboundCfg0;
       
          //Do setup for link - 0
          CSL_aifHwSetup(handleAifLink0, &ConfigLink0);
     @endverbatim
 * =============================================================================
 */ 
#pragma CODE_SECTION (CSL_aifHwSetup, ".text:csl_section:aif");
CSL_Status  CSL_aifHwSetup(
    /** Pointer to the object that holds reference to the
     *  instance of AIF link requested after the call
    */
    CSL_AifHandle             hAifLink,
    /** Pointer to setup structure which contains the
     *  information to program AIF to a useful state
    */
    CSL_AifLinkSetup          *linkSetup
)

{
    CSL_Status status = CSL_SOK;

    if (hAifLink == NULL)
        return CSL_ESYS_BADHANDLE;
    
    if (linkSetup == NULL)
        return CSL_ESYS_INVPARAMS;

    /* Global setup */
    if(linkSetup->globalSetup !=NULL) {
        aifSetupGlobalRegs(linkSetup, hAifLink);                                                       
    }

    // SERDES setup
    if (linkSetup->commonlinkSetup->pSerdesSetup != NULL) {
        aifSetupSdRegs(linkSetup, hAifLink);
    }

    // link regs setup
    if (linkSetup->commonlinkSetup != NULL) {
        aifSetupLinkRegs(linkSetup, hAifLink);
    }
    
    /* Outbound link setup*/
    
    /** TxMac setup*/
    if (linkSetup->outboundlinkSetup->pTxMacSetup != NULL) {
        aifSetupTxMacRegs(linkSetup, hAifLink);
    }

    if (linkSetup->outboundlinkSetup->pPeSetup != NULL) {
        aifSetupPeRegs(linkSetup, hAifLink);
    }

    /* Inbound link setup*/
    
    /** Rx MAC config */ 
    if (linkSetup->inboundlinkSetup->pRxMacSetup != NULL) {
        aifSetupRxMacRegs(linkSetup, hAifLink);
    }
    
    /* Protocol Decoder Setup */
    if (linkSetup->inboundlinkSetup->pPdSetup != NULL) {
        aifSetupPdRegs(linkSetup, hAifLink);
    }

	/* CPRI INPUT Data Format Converter Setup */
	if(linkSetup->inboundlinkSetup != NULL)
	{
		aifSetupCiRegs(linkSetup, hAifLink);
	}

	/* CPRI OUTPUT Data Format Converter Setup */
	if(linkSetup->outboundlinkSetup != NULL)
	{
		aifSetupCoRegs(linkSetup, hAifLink);
	}
    if (linkSetup->globalSetup->pCdSetup != NULL) {
        aifSetupCdRegs(linkSetup, hAifLink);
    }
    
    /* Aggregator Configuration */
    if (linkSetup->outboundlinkSetup->pAggrSetup != NULL) {
        aifSetupAgRegs(linkSetup, hAifLink);
    }
    
    /* Inbound FIFO setup */
    if (linkSetup->globalSetup->pInboundPsFifoSetup != NULL)
    {
        aifSetupInDbRegs(linkSetup, hAifLink);
    }

    /* Inbound FIFO setup */
    if (linkSetup->outboundlinkSetup != NULL)
    {
        aifSetupOutDbRegs(linkSetup, hAifLink);
    }
    
    return status;
}
