/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** ===========================================================================
 *   @file  csl_aifOpen.c
 *
 *   @path  $(CSLPATH)\src\aif
 *
 *   @desc  Antenna Interface.
 */

/* =============================================================================
 * Revision History
 * ===============
 *  26-Oct-2004 Pd File Created.
 *  21-Feb-2006 Naveen Shetti updated the file for DOXYGEN compatibiliy
 *
 * =============================================================================
 */

#include <csl_aif.h>

/** ============================================================================
 *   @n@b CSL_aifOpen 
 *   @brief Opens the instance of AIF requested.
 *
 *   @b Description
 *   @n The open call sets up the data structures for the particular instance of
 *   AIF device. The device can be re-opened anytime after it has been normally
 *   closed if so required. The handle returned by this call is input as an
 *   essential argument for rest of the APIs described for this module.
 *
 *   @b Arguments
 *   @verbatim            
           pAifLinkObj    Pointer to the object that holds reference to the
                          instance of AIF requested after the call
                           
           aifNum         Instance of AIF to which a handle is requested
           
           paifParam      Module specific parameters
 
           pStatus        pointer for returning status of the function call

     @endverbatim
 *
 *   <b> Return Value </b>  
 *      CSL_AifHandle
 *        Valid AIF instance handle will be returned if status value is
 *        equal to CSL_SOK.
 *
 *   <b> Pre Condition </b>
 *   @n  AIF must be successfully initialized via @a CSL_AIFInit() before 
 *   calling this function. Memory for the @a CSL_AifObj must be allocated 
 *   outside this call. This object must be retained while usage of this peripheral.
 *
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned in the status variable. If status
 *              returned is
 *   @li            CSL_SOK             Valid AIF handle is returned
 *   @li            CSL_ESYS_FAIL       The AIF instance is invalid
 *
 *        2.    AIF object structure is populated
 * 
 *   @b  Modifies
 *   @n    1. The status variable
 *
 *         2. AIF object structure
 *
 *   @b Example:
 *   @verbatim
          // handle for link 0
          CSL_AifHandle handleAifLink0;
          //  link object for link 0
          CSL_AifLinkObj AifLinkObj0;
          //AIF module specific parameters
          CSL_AifParam  aifParam;
          // CSL status
          CSL_Status status;

          aifParam.LinkIndex = CSL_AIF_LINK_0;
   
          // Open handle for link 0 - for use 
          handleAifLink0 = CSL_aifOpen(&AifLinkObj0, CSL_AIF, &aifParam, &status);

          if ((handleAifLink0 == NULL) || (status != CSL_SOK)) 
          {
             printf ("\nError opening CSL_AIF");
             exit(1);
          }
       @endverbatim
 *
 * ===========================================================================
 */
#pragma CODE_SECTION (CSL_aifOpen, ".text:csl_section:aif");
CSL_AifHandle CSL_aifOpen (
   /** Pointer to the object that holds reference to the
    *  instance of AIF requested after the call
    */
   CSL_AifLinkObj         *pAifLinkObj,
   /** Instance of AIF to which a handle is requested
    */
   CSL_InstNum            aifNum,
   /** Module specific parameters;
    */
   CSL_AifParam           *paifParam,
   /** This returns the status (success/errors) of the call.
    * Could be 'NULL' if the user does not want status information.
    */
   CSL_Status             *pStatus
   )
{
    CSL_Status           st;
    CSL_AifHandle       hAif;
    CSL_AifBaseAddress  baseAddress;

    if (pStatus == NULL) {
        return NULL;
    }
    if (pAifLinkObj == NULL) {
        *pStatus = CSL_ESYS_INVPARAMS;
        return NULL;
    }

    if (CSL_aifGetBaseAddress(aifNum, paifParam, &baseAddress) == CSL_SOK) {
        pAifLinkObj->regs = baseAddress.regs;
        pAifLinkObj->perNum = (CSL_InstNum)aifNum;
        pAifLinkObj->linkIndex = (CSL_AifLinkIndex)paifParam->linkIndex;
        hAif = (CSL_AifHandle)pAifLinkObj;
        st = CSL_SOK;
    } else {
        pAifLinkObj->regs = (CSL_AifRegsOvly)NULL;
        pAifLinkObj->perNum = (CSL_InstNum)-1;
        pAifLinkObj->linkIndex = (CSL_AifLinkIndex)NULL;
        hAif = (CSL_AifHandle)NULL;
        st = CSL_ESYS_FAIL;
    }

    *pStatus = st;

    return hAif;
}
