/* ============================================================================
 * Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005                 
 *                                                                              
 * Use of this software is controlled by the terms and conditions found in the
 * license agreement under which this software has been supplied.             
 * ============================================================================
 */ 


/** ===========================================================================  
 * @file csl_cfgHwControl.c
 *
 * @brief File for functional layer of CSL API @a CSL_cfgHwControl()
 *
 * @path $(CSLPATH)\cfg\src
 * 
 * Description
 *   - The @a CSL_cfgHwControl() function definition & it's associated
 *     functions
 * ============================================================================
 */

/* =============================================================================
 * Revision History
 * ===============
 * 14-Apr-2005 Brn File Created
 * =============================================================================
 */

#include <csl_cfg.h>
#include <csl_cfgAux.h>

/** @brief Takes a command with an optional argument & implements it.
 */

/** ============================================================================
 *   @n@b CSL_cfgHwControl
 *
 *   @b Description
 *   @n Takes a command of CFG with an optional argument & implements it.
 *
 *   @b Arguments
 *   @verbatim
            hCfg            Handle to the CFG instance

            cmd             The command to this API indicates the action to be
                            taken on CFG.

            arg             An optional argument.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li         CSL_SOK               - Status info return successful.
 *   @li         CSL_ESYS_INVCMD       - Invalid command
 *   @li         CSL_ESYS_INVPARAMS    - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n The registers of CFG.
 *
 *   @b Example
 *   @verbatim
        CSL_CfgHandle         hCfg;
        CSL_CfgHwControlCmd   cmd;
        void                  arg;

        status = CSL_cfgHwControl(hCfg, cmd, &arg);
     @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_cfgHwControl, ".text:csl_section:cfg");
CSL_Status CSL_cfgHwControl(
    /* Handle to the CFG instance */
    CSL_CfgHandle        hCfg,
    
    /* The command to this API indicates the action to be taken */
    CSL_CfgHwControlCmd  cmd,
    
    /* An optional argument @a void* casted */
    void                 *arg
)
{
    CSL_Status status = CSL_SOK;

    if (hCfg == NULL)
        return CSL_ESYS_BADHANDLE;

    switch(cmd){
        case CSL_CFG_CMD_CLEAR:
            CSL_cfgCmdClear(hCfg);
            break;

        default:
            status = CSL_ESYS_INVCMD ;
            break;
    }

  return status;
}
