/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005                 
 *                                                                              
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.             
 *   ===========================================================================
 */

/** =============================================================================
 *  @file  csl_cicClose.c
 *
 *  @PATH  $(CSLPATH)\src\cic
 *
 *  @brief File for functional layer of CSL API CSL_cicClose ()
 */
 
/* =============================================================================
 *  Revision History
 *  ===============
 *  16-Jan-2005 pd File Created
 * =============================================================================
 */
 
#include <csl_cic.h>
#include <csl_cicAux.h>

/** ============================================================================
 *   @n@b CSL_cicClose
 *
 *   @b Description
 *   @n This cic Handle can no longer be used to access the event. The event is 
 *      de-allocated and further access to the event resources are possible only
 *      after opening the event object again. 
 *
 *   @b Arguments
 *   @verbatim
            hCic            Handle identifying the event
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Close successful
 *   @li                    CSL_CIC_BADHANDLE  - The handle passed is invalid
 *
 *   <b> Pre Condition </b>
 *   @n  Functions CSL_cicInit() and CSL_cicOpen() have to be called in that 
 *       order successfully before calling this function.
 *
 *   <b> Post Condition </b>
 *   @n  The cic CSL APIs can not be called until the cic CSL is reopened 
 *       again using CSL_cicOpen() 
 *
 *   @b Modifies
 *   @n  None
 *            
 *   @b Example
 *   @verbatim
     CSL_Status                 intStat;
     CSL_CicParam               param;
     CSL_CicObj                 cicObj20;
     CSL_CicHandle              hCic20;
     CSL_CicContext             context;             
     // Init Module
     ...
     if (CSL_cicInit(&context) != CSL_SOK) {
     exit;
     
     //Opening a handle for the CSL_CIC_EVENTID_FSEVT18 at Cic Output Event ID 4
     param.eventId = CSL_CIC_EVENTID_FSEVT18;
     param.ectlEvtId = CSL_CIC_ECTL_EVT4;

     hCic20 = CSL_cicOpen(&cicObj20, CSL_CIC_0, &param, &intStat);
     
     // Close handle
     CSL_CicClose(hCic20);    
   
    @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_cicClose, ".text:csl_section:cic");
CSL_Status  CSL_cicClose (
    CSL_CicHandle      hCic
)
{
    Uint32      evt;
    Uint32      x;
    Uint32      y;
    CSL_Status  closeStatus = CSL_ESYS_BADHANDLE;

    if (hCic != CSL_CIC_BADHANDLE) {
        evt = (Uint32)(hCic->eventId);
        y = evt >> 5;
        x = 1 << (evt & 0x1f);
        
        asm(" dint");
        _CSL_cicAllocMask[hCic->cicNum][y] &= ~x; /* clear bit -> unused */
        hCic = (CSL_CicHandle) CSL_CIC_BADHANDLE;
        asm(" rint");
        
        closeStatus = CSL_SOK;
    }
        
    return closeStatus;
}

