/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006                 
 *                                                                              
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.             
 *   ===========================================================================
 */

/** =============================================================================
 *  @file  csl_cicGetHwStatus.c
 *
 *  @brief File for functional layer of CSL API @a CSL_cicGetHwStatus()
 *
 *  @PATH  $(CSLPATH)\src\cic
 */
 
/* =============================================================================
 *  Revision History
 *  ===============
 *  31-Jan-2006 pd File Created
 * =============================================================================
 */
 
#include <csl_cic.h>
#include <csl_cicAux.h>

/** ============================================================================
 *   @n@b CSL_cicGetHwStatus
 *
 *   @b Description
 *   @n Queries the peripheral for status. The CSL_cicGetHwStatus(..) API
 *      could be used to retrieve status or configuration information from
 *      the peripheral. The user must allocate an object that would hold
 *      the retrieved information and pass a pointer to it to the function.
 *      The type of the object is specific to the query-command.
 *
 *   @b Arguments
 *   @verbatim
            hCic           Handle identifying the event

            query           The query to this API of CIC which indicates the
                            status to be returned.

            answer          Placeholder to return the status.
     @endverbatim
 *
 *   <b> Return Value </b> CSL_Status
 *   @li                   CSL_SOK               - Status info return successful
 *   @li                   CSL_ESYS_INVQUERY     - Invalid query
 *   @li                   CSL_ESYS_INVPARAMS    - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 * @b Example:
 *   @verbatim
    CSL_CicContext             context; 
    CSL_Status                 intStat;
    CSL_CicParam               evtEctlId;
    CSL_CicObj                 cicObj20;
    CSL_CicHandle              hCic20;
    CSL_CicGlobalEnableState   state;
    Uint32                     intrStat;
    
    // Init Module
    ...
    if (CSL_cicInit(&context) != CSL_SOK)
       exit;
    // Opening a handle for the CSL_CIC_EVENTID_FSEVT18 at Cic Output Event ID 4
     param.eventId = CSL_CIC_EVENTID_FSEVT18;
     param.ectlEvtId = CSL_CIC_ECTL_EVT4;
     hCic20 = CSL_cicOpen (&cicObj20, CSL_CIC_0, &param, &intStat);
    
    do {
        CSL_cicGetHwStatus(hCic20,CSL_CIC_QUERY_PENDSTATUS, \
                            (void*)&intrStat);
    } while (!stat);
    
    // Close handle
    CSL_CicClose(hCic20);

    @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_cicGetHwStatus, ".text:csl_section:cic");
CSL_Status  CSL_cicGetHwStatus (
    CSL_CicHandle            hCic,
    CSL_CicHwStatusQuery     myQuery,
    void                      *answer
)
{


    CSL_Status status = CSL_SOK;
    
    if (hCic && answer) {
        switch (myQuery) {
            case CSL_CIC_QUERY_PENDSTATUS:
                if (hCic->eventId < 2)
                    *((Uint32*)answer) = \
                             (Uint32)(CSL_cicCombinedEventGet(hCic));
                else    
                    *((Bool*)answer) = \
                               (Bool)(CSL_cicQueryEventStatus(hCic));
                break;  
            default:
                status = CSL_ESYS_INVQUERY;
                break;
        }       
    } 
    else
        status = CSL_ESYS_INVPARAMS;
        
    return status;
}

