/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006                 
 *                                                                              
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.             
 *   ===========================================================================
 */

/** =============================================================================
 *  @file  csl_cicHwControl.c
 *
 *  @brief File for functional layer of CSL API CSL_cicHwControl()
 *
 *  @PATH  $(CSLPATH)\src\cic
 */
 
/* =============================================================================
 *  Revision History
 *  ===============
 *  1-Feb-2006 pd File Created
 * =============================================================================
 */
 
#include <csl_cic.h>
#include <csl_cicAux.h>

/** ============================================================================
 *   @n@b CSL_cicHwControl
 *
 *   @b Description
 *   @n Perform a control-operation. This API is used to invoke any of the
 *      supported control-operations supported by the module.
 *
 *   @b Arguments
 *   @verbatim
            hCic           Handle identifying the event

            command         The command to this API indicates the action to be
                            taken on CIC.
    
            commandArg      An optional argument.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - HwControl successful.
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVCMD     - Invalid command
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_cicOpen() must be invoked before this call.
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n The hardware registers of CIC.
 *
 *   @b Example
 *   @verbatim
    CSL_CicObj cicObj20;
    CSL_CicGlobalEnableState state;
    CSL_CicContext context; 
    CSL_Status intStat;
    CSL_CicParam evtEctlId;
    
    // Init Module
    CSL_cicInit(&context);
    
    // Opening a handle for the CSL_CIC_EVENTID_FSEVT18 at Cic Output Event ID 4
     param.eventId = CSL_CIC_EVENTID_FSEVT18;
     param.ectlEvtId = CSL_CIC_ECTL_EVT4;
     
     Cic20 = CSL_cicOpen (&cicObj20, CSL_CIC_0, &param, NULL);
            
    CSL_cicHwControl(hCic20, CSL_CIC_CMD_EVTENABLE, NULL);
     @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_cicHwControl, ".text:csl_section:cic");
CSL_Status  CSL_cicHwControl(
    CSL_CicHandle            hCic,
    CSL_CicHwControlCmd      controlCommand,
    void*                    commandArg
)

{
    CSL_Status status = CSL_SOK;
    
    if (hCic == NULL)
        status = CSL_ESYS_BADHANDLE;
        
    switch(controlCommand) {
        case CSL_CIC_CMD_EVTENABLE: 
            if (hCic->eventId < 2) {
            	if (commandArg)
                	CSL_cicCombinedEventEnable(hCic, \
                								*(CSL_BitMask32*)commandArg);
            }
            else 
                CSL_cicEventEnable(hCic);                   
            break;
        
        case CSL_CIC_CMD_EVTDISABLE:
            if (hCic->eventId < 2) {
                if (commandArg)
                    CSL_cicCombinedEventDisable(hCic, \
                                                 *(CSL_BitMask32*)commandArg);
            }       
            else 
                CSL_cicEventDisable(hCic);
           
           break;
            
        case CSL_CIC_CMD_EVTSET:
            CSL_cicEventSet(hCic);               
            break;
                
        case CSL_CIC_CMD_EVTCLEAR:  
            if (hCic->eventId < 2) {
                    if (commandArg)
                        CSL_cicCombinedEventClear(hCic, \
                                                  *(CSL_BitMask32*)commandArg);
            } 
            else                    
            	CSL_cicEventClear(hCic);            
            break;
        
        default:
            status = CSL_ESYS_INVCMD;
            break;

    }
    return status;
}
