/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006                 
 *                                                                              
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.             
 *   ===========================================================================
 */

/** ===========================================================================
 *  @file  csl_cicOpen.c
 *
 *  @brief File for functional layer of CSL API CSL_cicOpen()
 *
 *  @PATH  $(CSLPATH)\src\cic
 */
 
/* =============================================================================
 *  Revision History
 *  ===============
 *  1-Feb-2006 pd File Created
 *
 * =============================================================================
 */

#include <csl_cic.h>
#include <csl_cicAux.h>

/** ============================================================================
 *   @n@b CSL_cicOpen
 *
 *   @b Description
 *   @n The API would reserve an interrupt-event for use. It returns
 *      a valid handle to the event only if the event is not currently
 *      allocated. The user could release the event after use by calling
 *      CSL_cicClose(..)
 *
 *   @b Arguments
 *   @verbatim
              pCicObj     Pointer to the CSL-object allocated by the user

              eventId      The event-id of the interrupt

              param        Pointer to the Cic specific parameter
              
              pStatus      Pointer for returning status of the function call
                           
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_CicHandle
 *   @n                     Valid CIC handle identifying the event
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n   1.    CIC object structure is populated
 *   @n   2.    The status is returned in the status variable. If status
 *              returned is
 *   @li            CSL_SOK             Valid cic handle is returned
 *   @li            CSL_ESYS_FAIL       The open command failed
 *
 *   @b Modifies
 *   @n    1. The status variable
 *   @n    2. CIC object structure
 *
 * @b Example:
 * @verbatim

    CSL_CicObj cicObj20;
    CSL_CicContext context; 
    CSL_Status intStat;
    CSL_CicParam param;

    // Init Module
    CSL_cicInit(&context);

    // Opening a handle for the CSL_CIC_EVENTID_FSEVT18 at Cic Output Event ID 4
    param.eventId = CSL_CIC_EVENTID_FSEVT18;
    param.ectlEvtId = CSL_CIC_ECTL_EVT4;
    
    hCic20 = CSL_cicOpen(&cicObj20, CSL_CIC_0, &param , &intStat);

   @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_cicOpen, ".text:csl_section:cic");
CSL_CicHandle  CSL_cicOpen (
    CSL_CicObj          *pCicObj,
    CSL_InstNum         cicNum,
    CSL_CicParam        *pCicParam,
    CSL_Status          *pStatus
)
{
    CSL_Status status;
    CSL_CicHandle hCic = (CSL_CicHandle)NULL;
    CSL_CicBaseAddress baseAddress;
    Uint32          evtId;      
    Uint32          _yCo;
    Uint32          _xCo;

    /* Added according to review comment 1. */
    if (pStatus == NULL) {
        return NULL;
    }

    if (pCicObj == NULL) {
        *pStatus = CSL_ESYS_INVPARAMS;
        return NULL;
    }

    status = CSL_cicGetBaseAddress(cicNum, pCicParam, &baseAddress);
    if (status == CSL_SOK) {
        evtId = (Uint32)(pCicParam->eventId);
        _yCo = evtId >> 5;          /* _yCo co-ordinate */
        _xCo = 1 << (evtId & 0x1f); /* _xCo co-ordinate */

        asm(" dint");
        if (!(_CSL_cicAllocMask[cicNum][_yCo] & _xCo)) {
            _CSL_cicAllocMask[cicNum][_yCo] |= _xCo; /* set bit -> used */
            hCic = pCicObj;
        }
        asm(" rint");

        if (hCic != (CSL_CicHandle)NULL) {
            pCicObj->regs = baseAddress.regs;
            pCicObj->eventId = pCicParam->eventId;
            pCicObj->ectlEvtId = pCicParam->ectlEvtId;
            pCicObj->cicNum = (CSL_InstNum)cicNum;
            hCic = (CSL_CicHandle)pCicObj;
			CSL_cicMapEventVector(hCic); 
        }
        else {
            status = CSL_ESYS_FAIL;
        }
    }
    else {
        pCicObj->regs = (CSL_CicRegsOvly)NULL;
        pCicObj->cicNum = (CSL_InstNum)-1;
        pCicObj->eventId = pCicParam->eventId;
        pCicObj->ectlEvtId = pCicParam->ectlEvtId;
    }

    *pStatus = status;
     
    return hCic;
} 

