/*  ============================================================================
 *  Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *  Use of this software is controlled by the terms and conditions found in the
 *  license agreement under which this software has been supplied.
 *  ===========================================================================
 */

/** @file csl_cicGetBaseAddress.c
 *
 *    @brief File for functional layer of CSL API @a CSL_cicGetBaseAddress()
 *
 *  Description
 *    - The @a CSL_cicGetBaseAddress() function definition & it's associated functions
 *
 *  PATH  \\(CSLPATH)\\src\\common
 *
 *  Modification 1
 *    - Modified on: 28/5/2004
 *    - Reason: created the sources
 *
 *  @date 28th May, 2004
 *    @author PD
 *
 */

/*  ============================================================================
 *  Revision History
 *  ===============
 *  31-aug-2004 PSK Modified according to review comments.
 *  ============================================================================
 */

#include <csl_cic.h>

/** ============================================================================
 *   @n@b CSL_cicGetBaseAddress
 *
 *   @b Description
 *   @n  Function to get the base address of the peripheral instance.
 *       This function is used for getting the base address of the peripheral
 *       instance. This function will be called inside the CSL_cicOpen()
 *       function call. This function is open for re-implementing if the user
 *       wants to modify the base address of the peripheral object to point to
 *       a different location and there by allow CSL initiated write/reads into
 *       peripheral MMR's go to an alternate location.
 *
 *   @b Arguments
 *   @verbatim
            cicNum          Specifies the instance of CIC to be opened.

            pCicParam       Module specific parameters.

            pBaseAddress    Pointer to baseaddress structure containing base
                            address details.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             Successful in getting the base address
 *   @li                    CSL_ESYS_FAIL      The instance number is invalid.
 *   @li                    CSL_ESYS_INVPARAMS Invalid parameters
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Base Address structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. Base address structure is modified.
 *
 *   @b Example
 *   @verbatim
        CSL_Status              status;
        CSL_CicBaseAddress  baseAddress;

       ...
      status = CSL_cicGetBaseAddress(CSL_CIC_0, NULL, &baseAddress);

    @endverbatim
 * ===========================================================================
 */
#pragma CODE_SECTION (CSL_cicGetBaseAddress, ".text:csl_section:cic");
CSL_Status CSL_cicGetBaseAddress (
        CSL_InstNum            cicNum,
        CSL_CicParam *         pCicParam,
        CSL_CicBaseAddress *   pBaseAddress
)
{
    CSL_Status status = CSL_SOK;

    pCicParam = pCicParam; /* To avoid compiler warning/remark */

    /* Added according to review comment 2. */
    if(pBaseAddress == NULL) {
        status = CSL_ESYS_INVPARAMS;
    }

    switch (cicNum) {
        case CSL_CIC_0:
            pBaseAddress->regs = (CSL_CicRegsOvly)CSL_CIC_0_REGS;
            break;
        case CSL_CIC_1:
            pBaseAddress->regs = (CSL_CicRegsOvly)CSL_CIC_1_REGS;
            break;
        case CSL_CIC_2:
            pBaseAddress->regs = (CSL_CicRegsOvly)CSL_CIC_2_REGS;
            break;
        case CSL_CIC_3:
            pBaseAddress->regs = (CSL_CicRegsOvly)CSL_CIC_3_REGS;
            break;
        default:
            pBaseAddress->regs = NULL;
            status = CSL_ESYS_FAIL;
            break;
    }

    return status;
}

