/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */
/** ============================================================================
 *   @file  csl_etbGetBaseAddress.c
 *
 *   @path  $(CSLPATH)\src\common
 *
 *   @desc  CSL Implementation of CSL_etbGetBaseAddress
 *
 */
/*  ============================================================================
 *  Revision History
 *  ===============
 *  10-aPR-2006 NG  Author File Created
 *
 *  ============================================================================
 */
#include <soc.h>
#include <csl_etb.h>

/** ============================================================================
 *   @n@b CSL_etbGetBaseAddress
 *
 *   @b Description
 *   @n  Function to get the base address of the peripheral instance.
 *       This function is used for getting the base address of the peripheral
 *       instance. This function will be called inside the CSL_etbOpen()
 *       function call. This function is open for re-implementing if the user
 *       wants to modify the base address of the peripheral object to point to
 *       a different location and there by allow CSL initiated write/reads into
 *       peripheral. MMR's go to an alternate location.
 *
 *   @b Arguments
 *   @verbatim
            etbNum          Specifies the instance of the etb to be opened.

            pEtbParam       Module specific parameters.

            pBaseAddress    Pointer to baseaddress structure containing base
                            address details.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK            Successful on getting the base
 *                                             address of etb
 *   @li                    CSL_ESYS_FAIL      The instance number is invalid.
 *   @li                    CSL_ESYS_INVPARAMS Inavlid parameters
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Base Address structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. Base address structure is modified.
 *
 *   @b Example
 *   @verbatim
        CSL_Status              status;
        CSL_EtbBaseAddress  baseAddress;

       ...
      status = CSL_etbGetBaseAddress(CSL_ETB_0, NULL, &baseAddress);

     @endverbatim
 * ===========================================================================
 */

#pragma CODE_SECTION (CSL_etbGetBaseAddress, ".text:csl_section:etb");
CSL_Status CSL_etbGetBaseAddress (
    CSL_InstNum           etbNum,
    CSL_EtbParam          *pEtbParam,
    CSL_EtbBaseAddress    *pBaseAddress
)
{
    CSL_Status status = CSL_SOK;

    if (pBaseAddress == NULL) {
        return CSL_ESYS_INVPARAMS;
    }

    switch (etbNum) {
        case CSL_ETB_0:
            pBaseAddress->regs = (CSL_EtbRegsOvly)CSL_ETB_0_REGS;
            break;

        case CSL_ETB_1:
            pBaseAddress->regs = (CSL_EtbRegsOvly)CSL_ETB_1_REGS;
            break;

        case CSL_ETB_2:
            pBaseAddress->regs = (CSL_EtbRegsOvly)CSL_ETB_2_REGS;
            break;
        
        default:
            pBaseAddress->regs = (CSL_EtbRegsOvly)NULL;
            status = CSL_ESYS_FAIL;
            break;
    }

    return status;
}

