/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *  ============================================================================
 */

/** 
 *  @file    csl_ddr2Open.c
 *  
 *  @brief   File for functional layer of CSL API @a CSL_ddr2Open()
 *
 *  @path:   $(CSLPATH)\src\ddr2
 *
 *  @desc
 *    - The @a CSL_ddr2Open() function definition & it's associated functions
 *
 *  Modification 1
 *    - Created on: 10/04/2005
 *    - Reason:       Created the sources.
 *
 *  @date 10th April, 2005
 *  @author Ramitha Mathew.
 *
 */
#include <csl_ddr2.h>

/** ============================================================================
 *   @n@b   CSL_ddr2Open
 *
 *   @b Description
 *   @n This function returns the handle to the DDR2 instance. This
 *      handle is passed to all other CSL APIs.
 *
 *   @b Arguments
 *   @verbatim

            pDdr2Obj        Pointer to the object that holds reference to the
                            instance of DDR2 requested after the call
            
            ddr2Num         Instance of DDR2 to which a handle is requested
 
            pDdr2Param      Pointer to module specific parameters
 
            pStatus         pointer for returning status of the function call

     @endverbatim
 *
 *   <b> Return Value </b>  
 *      CSL_Ddr2Handle - Valid DDR2 instance handle will be returned if status
 *    @li                value is equal to CSL_SOK.
 * 
 *   <b> Pre Condition </b>
 *   @n  @a The DDR2 must be successfully initialized via CSL_ ddr2Init() 
 *          before calling this function.
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b  Modifies
 *   @n  None
 *
 *   @b Example:
 *   @verbatim
         CSL_Status         status;
         CSL_Ddr2Obj        ddr2Obj;
         CSL_Ddr2Handle     hDdr2;
 
         hDdr2 = CSL_Ddr2Open(&ddr2Obj,
                              CSL_DDR2,
                              NULL,
                              &status
                             );
     @endverbatim
 *
 * =============================================================================
 */

#pragma CODE_SECTION (CSL_ddr2Open, ".text:csl_section:ddr2");
CSL_Ddr2Handle CSL_ddr2Open (
    CSL_Ddr2Obj      *pDdr2Obj,
    CSL_InstNum      ddr2Num,
    CSL_Ddr2Param    *pDdr2Param,
    CSL_Status       *pStatus
)
{    
    CSL_Status             status = CSL_SOK;
    CSL_Ddr2Handle         hDdr2;
    CSL_Ddr2BaseAddress    baseAddress;
    
    *pDdr2Param          = *pDdr2Param;
    
    if (pDdr2Obj == NULL) {
        return NULL;
    }
        
    status = CSL_ddr2GetBaseAddress(ddr2Num, pDdr2Param, &baseAddress);

    if (status == CSL_SOK) {
        pDdr2Obj->regs = baseAddress.regs;
        pDdr2Obj->perNum = (CSL_InstNum)ddr2Num;
        hDdr2 = (CSL_Ddr2Handle)pDdr2Obj;
    } 
    else {
        pDdr2Obj->regs = (CSL_Ddr2RegsOvly)NULL;
        pDdr2Obj->perNum = (CSL_InstNum)-1;
        hDdr2 = (CSL_Ddr2Handle)NULL;
    }

    if (pStatus != NULL) {
        *pStatus = status;
    }
    return hDdr2;
}

