/* =============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 * =============================================================================
 */ 
/** ============================================================================
 *   @file  csl_dtfGetHwStatus.c
 *
 *   @path  $(CSLPATH)\dtf\src
 *
 *   @desc  
 *
 */
/* =============================================================================
 *  Revision History
 *  ===============
 *  12-Apr-2006 NG  Author File Created 
 *
 * =============================================================================
 */
#include <csl_dtf.h>

/** ============================================================================
 *   @n@b CSL_dtfGetHwStatus
 *
 *   @b Description
 *   @n Gets the status of different operations or some setup-parameters of 
 *      DTF. The status is returned through the third parameter.
 *
 *   @b Arguments
 *   @verbatim
            hDtf     DTF handle returned by successful 'open'

            query     The query to this API of DTF which indicates the
                      status to be returned. 

                      Query command, refer @a CSL_DtfHwStatusQuery for the list 
                      of query commands supported 

            response  Response from the query. 

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Status info return successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVQUERY   - Invalid query command
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_dtfInit() and CSL_dtfOpen() must be called successfully 
 *       in order before calling CSL_dtfGetHwStatus(). 
 *       Refer to @a CSL_DtfHwStatusQuery for the argument to be passed along 
 *       with the corresponding query command.
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_DtfHandle hDtf;
        CSL_Status status;
        Uint32 response;
        ...
        status = CSL_dtfGetHwStatus(hDtf,
                                     CSL_DTF_QUERY_DTFOWN_STATUS,
                                     &response);


     @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_dtfGetHwStatus, ".text:csl_section:dtf");

CSL_Status  CSL_dtfGetHwStatus (
    CSL_DtfHandle           hDtf,
    CSL_DtfHwStatusQuery    query,
    void                    *response
)
{
    CSL_Status status = CSL_SOK;

    if (hDtf == NULL) {
        return CSL_ESYS_BADHANDLE;
    }

    switch (query) {
        case CSL_DTF_QUERY_AOWNERSHIP:
            /* Queries the DTF ownership */
            *(Uint32 *)response = (CSL_FEXT(hDtf->regs->DTFCR, DTF_DTFCR_AOWN));
            break;

        case CSL_DTF_QUERY_DTFOWN_STATUS:
            /* Queries DTF ownership state control */
            *(Uint32 *)response = 
                (CSL_FEXT( hDtf->regs->DTFCR, DTF_DTFCR_DTFOWN_ST));
            break;

        case CSL_DTF_QUERY_DTFENABLE:
            /* Queries DTFENABLE */
            *(Uint32 *)response = 
                (CSL_FEXT( hDtf->regs->DTFCR, DTF_DTFCR_DTFENABLE));
            break;

        case CSL_DTF_QUERY_SPMDISABLE:
            /* Queries Static-Pin-Merger */
            *(Uint32 *)response = 
                (CSL_FEXT( hDtf->regs->DTFCR, DTF_DTFCR_SPMDISABLE));
            break;

        default:
            status = CSL_ESYS_INVQUERY;
            break;
    }
    return(status);
}

