/* =============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 * =============================================================================
 */ 
/** ============================================================================
 *   @file  csl_dtfHwControl.c
 *
 *   @path  $(CSLPATH)\dtf\src
 *
 *   @desc  
 *    - The CSL_dtfHwControl() function definition & it's associated functions
 *
 */
/* =============================================================================
 *  Revision History
 *  ===============
 *  12-Apr-2006 NG  Author File Created 
 *
 * =============================================================================
 */
#include <csl_dtf.h>

/** ============================================================================
 *   @n@b CSL_dtfHwControl
 *
 *   @b Description
 *   @n This function takes an input control command with an optional argument
 *      and accordingly controls the operation/configuration of DTF.
 *
 *   @b Arguments
 *   @verbatim
            hDtf       DTF handle returned by successful 'open'

            cmd         The command to this API indicates the action to be
                        taken on DTF.
                        Control command, refer @a CSL_DtfControlCmd for the 
                        list of commands supported

            arg         An optional argument.
                        Optional argument as per the control command, 
                        @a void * casted

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li         CSL_SOK               - Status info return successful.
 *   @li         CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li         CSL_ESYS_INVCMD       - Invalid command
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_dtfInit() and CSL_dtfOpen() must be called successfully 
 *       in order before calling CSL_dtfHwControl(). 
 *
 *       Refer to @a CSL_DtfHwControlCmd for the argument type (@a void*) that 
 *       needs to be passed with the control command
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n The hardware registers of DTF.
 *
 *   @b Example
 *   @verbatim
      CSL_Status status;
      Uint32  arg;
      CSL_DtfHandle hDtf;
      ...
      // DTF object defined and HwSetup structure defined and initialized
      ...
     
      // Init successfully done
      ...
      // Open successfully done
      ...
      
      arg = 1;
      status = CSL_dtfHwControl(hDtf,
                                CSL_DTF_CMD_SET_DTFENABLE,
                                &arg);

     @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_dtfHwControl, ".text:csl_section:dtf");

CSL_Status  CSL_dtfHwControl (
    CSL_DtfHandle        hDtf,
    CSL_DtfControlCmd    cmd,
    void*                arg
)
{
    CSL_Status          status  = CSL_SOK;
    Uint32              argv = *(Uint32*)arg;
    
    if (hDtf == NULL) {
        status = CSL_ESYS_BADHANDLE;
        return status;
    }

    switch (cmd) {

        case CSL_DTF_CMD_ENA_AOWN_APP:
            /* Setup the DTF ownership to Application */
            CSL_FINST(hDtf->regs->DTFCR, DTF_DTFCR_AOWN, APP);
            break;

        case CSL_DTF_CMD_ENA_AOWN_EMU:
            /* Setup the DTF ownership to Emulation */
            CSL_FINST(hDtf->regs->DTFCR, DTF_DTFCR_AOWN, EMU);
            break;

        case CSL_DTF_CMD_FLUSH_DTFFLUSH:
            /* DTF Flush */
            CSL_FINS(hDtf->regs->DTFCR, DTF_DTFCR_DTFFLUSH, 1);
            break;

        case CSL_DTF_CMD_SET_DTFENABLE:
            /* Asserting the DTF Enable */
            CSL_FINS(hDtf->regs->DTFCR, DTF_DTFCR_DTFENABLE, argv);
            break;

        case CSL_DTF_CMD_DIS_SPMDISABLE:
            /* Asserting the DTF Enable */
            CSL_FINST(hDtf->regs->DTFCR, DTF_DTFCR_SPMDISABLE, DISABLE);
            break;

        case CSL_DTF_CMD_ENA_SPMDISABLE:
            /* Asserting the DTF Enable */
            CSL_FINST(hDtf->regs->DTFCR, DTF_DTFCR_SPMDISABLE, ENABLE);
            break;

        case CSL_DTF_CMD_RELEASE_OWNERSHIP:
            /* Releasing the DTF ownership */
            CSL_FINST(hDtf->regs->DTFCR, DTF_DTFCR_DTFOWN_ST, RELEASE);
            break;

        case CSL_DTF_CMD_CLAIM_OWNERSHIP:
            /* Claiming the DTF ownership */
            CSL_FINST(hDtf->regs->DTFCR, DTF_DTFCR_DTFOWN_ST, CLAIM);
            break;

        case CSL_DTF_CMD_ENA_OWNERSHIP:
            /* Enabling the DTF ownership */
            CSL_FINST(hDtf->regs->DTFCR, DTF_DTFCR_DTFOWN_ST, ENABLE);
            break;

        default:
            status = CSL_ESYS_INVCMD;
            break;
    }
    return (status);
}

