/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *   ===========================================================================
 */
/** @file  csl_dtfOpen.c
 *
 *  @path $(CSLPATH)\dtf\src
 *
 *  @brief File for functional layer of CSL API @a CSL_dtfOpen()
 *
 */
/* =============================================================================
 * Revision History
 * ===============
 *
 *  12-Apr-2006 NG  Author File created
 * =============================================================================
 */
#include <csl_dtf.h>

/** ============================================================================
 *   @n@b CSL_dtfOpen
 *
 *   @b Description
 *   @n This function returns the handle to the DTF controller
 *      instance. This handle is passed to all other CSL APIs.
 *
 *   @b Arguments
 *   @verbatim
 *          pDtfObj     Pointer to dtf object.
 *
 *          dtfNum      Instance of  DSP DTF to be opened. 
 *                      There are three instance of the dtf available.
 *
 *          pDtfParam   Module specific parameters.
 *
 *          pStatus     Status of the function call
 *    @endverbatim
 *
 *   <b> Return Value </b>  CSL_DtfHandle
 *   @n                         Valid dtf handle will be returned if
 *                              status value is equal to CSL_SOK.
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned in the status variable. If status
 *              returned is
 *   @li            CSL_SOK             Valid dtf handle is returned
 *   @li            CSL_ESYS_FAIL       The dtf instance is invalid
 *   @li            CSL_ESYS_INVPARAMS  Invalid parameter
 *
 *        2.    DTF object structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. DTF object structure
 *
 *   @b Example
 *   @verbatim
 *          CSL_Status         status;
 *          CSL_DtfObj         dtfObj;
 *          CSL_DtfHandle      hDtf;
 *
 *          ...
 *               
 *          hDtf = CSL_dtfOpen(&dtfObj, CSL_DTF_0, NULL, &status);
 *           
 *          ...
 *  @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_dtfOpen, ".text:csl_section:dtf");

CSL_DtfHandle CSL_dtfOpen (
    CSL_DtfObj 	    *pDtfObj,
    CSL_InstNum 	dtfNum,
    CSL_DtfParam 	*pDtfParam,
    CSL_Status 		*pStatus
)
{
    CSL_Status              status;
    CSL_DtfHandle           hDtf;
    CSL_DtfBaseAddress      baseAddress;

    if (pStatus == NULL) {
        *pStatus = CSL_ESYS_INVPARAMS;
        hDtf = (CSL_DtfHandle) NULL;
        return hDtf;
    }

    if (pDtfObj == NULL) {
        *pStatus = CSL_ESYS_INVPARAMS;
        hDtf = (CSL_DtfHandle) NULL;
        return hDtf;
    }

    status = CSL_dtfGetBaseAddress(dtfNum, pDtfParam, &baseAddress);

    if (status == CSL_SOK) {
        pDtfObj->regs = baseAddress.regs;
        pDtfObj->dtfNum = (CSL_InstNum)dtfNum;
        hDtf = (CSL_DtfHandle)pDtfObj;
    }
    else {
        pDtfObj->regs = (CSL_DtfRegsOvly) NULL;
        pDtfObj->dtfNum = (CSL_InstNum) -1;
        hDtf = (CSL_DtfHandle) NULL;
    }

    *pStatus = status;

    return hDtf;
}

