/*  ===========================================================================
 *  Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *  Use of this software is controlled by the terms and conditions found
 *  in the license agreement under which this software has been supplied
 *  provided
 *
 *  ===========================================================================
 */

/** ============================================================================
 *   @file  csl_edcClear.c
 *
 *   @path  $(CSLPATH)\src\edc
 *
 *   @desc  File for functional layer of CSL API CSL_edcClear()
 *
 */
 
/* =============================================================================
 *  Revision History
 *  ===============
 *  08-Jul-2004 Chad Courtney File created.
 *
 *  02-Sept-2005 ds modified.
 *
 * =============================================================================
 */ 
#include <csl_edc.h>

/** ============================================================================
 *   @n@b CSL_edcClear
 *
 *   @b Description
 *   @n Clears the Address of the parity error for the specified memory 
 *      along with the access type parity error bit.
 *
 *   @b Arguments
 *   @verbatim
            edcMem          Specificies what memory EDC error address is 
                            to be cleared
 
            edcAccessType   Specificies what fetch type parity error bit or 
                            parity error count type is to be cleared.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status 
 *   @li                    CSL_SOK            - Address of parity error clear 
 *                                               is successful
 *   @li                    CSL_ESYS_INVPARAMS - The parameter passed is invalid
 *                                                
 *   <b> Pre Condition </b>
 *   @n  None
 *       
 *
 *   <b> Post Condition </b>
 *   @n  None
 *       
 *   @b Modifies
 *   @n EDC registers
 *
 *   @b Example
 *   @verbatim
            CSL_Status          status;
            ...
            status = CSL_edcClear (CSL_EDC_L1P, CSL_EDC_DCLR);
            ...
    @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_edcClear, ".text:csl_section:edc");
CSL_Status  CSL_edcClear (
    CSL_EdcMem              edcMem,
    CSL_EdcClrAccessType    edcAccessType   
)
{
    CSL_Status status = CSL_SOK;
    
    switch(edcAccessType) {
        /* Clear Data Fetch Parity Error */
        case CSL_EDC_DCLR: 
            /* Clear Parity Error for L1P */
            if(edcMem == CSL_EDC_L1P) { 
                /* No Data Parity Fetch Error for L1P */
                status = CSL_ESYS_INVPARAMS; 
            }
        
            /* Clear Parity Error for L2 */
            else {
                if(edcMem == CSL_EDC_L2) { 
                    ((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDCMD = 
                                                CSL_FMKT(EDC_L2EDCMD_DCLR, CLR);
                }
                else {  /* Incorrect Memory Type given */
                    status = CSL_ESYS_INVPARAMS;
                }
            }
        break;
        
        /* Clear Program Fetch Parity Error */
        case CSL_EDC_PCLR: 
            /* Clear Parity Error for L1P */
            if(edcMem == CSL_EDC_L1P) { 
                ((CSL_EdcRegs*)CSL_EDC_0_REGS)->L1PEDCMD = 
                                              CSL_FMKT(EDC_L1PEDCMD_ICLR, CLR);
            }
            /* Clear Parity Error for L2 */
            else {
                if(edcMem == CSL_EDC_L2) {
                    ((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDCMD = 
                                                CSL_FMKT(EDC_L2EDCMD_ICLR, CLR);
                }
                else {  
                     /* Incorrect Memory Type given */
                    status = CSL_ESYS_INVPARAMS;
                }
            }
        break;
        
        /* Clear DMA Read Parity Error */
        case CSL_EDC_DMACLR: 
            /* Clear Parity Error for L1P */
            if(edcMem == CSL_EDC_L1P) {
                ((CSL_EdcRegs*)CSL_EDC_0_REGS)->L1PEDCMD =  
                                             CSL_FMKT(EDC_L1PEDCMD_DMACLR, CLR);
            }
            /* Clear Parity Error for L2 */
            else {
                if(edcMem == CSL_EDC_L2) {
                    ((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDCMD = 
                                              CSL_FMKT(EDC_L2EDCMD_DMACLR, CLR);
                }
                else {
                    /* Incorrect Memory Type given */
                    status = CSL_ESYS_INVPARAMS;
                }
            }
        break;
        
        /* Clear Correctable Parity Error Count */
        case CSL_EDC_CECNTCLR: 
            if(edcMem == CSL_EDC_L1P) {
                /* No Parity Error Count for L1P */
                status = CSL_ESYS_INVPARAMS; 
            }
            /* Clear Correctable Parity Error Count for L2 */
            else {
                if(edcMem == CSL_EDC_L2) { 
                    CSL_FINST(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDCPEC, \
                            EDC_L2EDCPEC_CNT, RESETVAL);      
                }
                else { 
                    /* Incorrect Memory Type given */
                    status = CSL_ESYS_INVPARAMS;
                }
            }
        break;
        
        /* Clear Correctable Parity Error Count */
        case CSL_EDC_NCECNTCLR: 
            /* No Parity Error Count for L1P */
            if(edcMem == CSL_EDC_L1P) {
                /* Clear Correctable Parity Error Count for L2 */
                status = CSL_ESYS_INVPARAMS; 
            }
            else {
                if(edcMem == CSL_EDC_L2) { 
                    CSL_FINST(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDNPEC, \
                              EDC_L2EDNPEC_CNT, RESETVAL);      
                }
                else {  
                    /* Incorrect Memory Type given */
                    status = CSL_ESYS_INVPARAMS;
                }
            }    
        break;
        
        default:
            status = CSL_ESYS_INVPARAMS;         
    }
    
    return status;    
}
