/*  ===========================================================================
 *  Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *  Use of this software is controlled by the terms and conditions found
 *  in the license agreement under which this software has been supplied
 *  provided
 *
 *  ===========================================================================
 */
 
 /** ============================================================================
 *   @file  csl_edcGetErrorAddress.c
 *
 *   @path  $(CSLPATH)\src\edc
 *
 *   @desc  File for functional layer of CSL API CSL_edcGetErrorAddress ()
 *
 */
 
/* =============================================================================
 *  Revision History
 *  ===============
 *  08-Jul-2004 Chad Courtney File created.
 *
 *  02-Sept-2005 ds modified.
 *
 * =============================================================================
 */

#include <csl_edc.h>

/** ============================================================================
 *   @n@b CSL_edcGetErrorAddress
 *
 *   @b Description
 *   @n Gets the Address location of the parity error
 *
 *   @b Arguments
 *   @verbatim
            edcMem          Specificies what memory EDC Address Info is to be
                            aquired for.
 
            edcAddr         Structure for returning Address, L2 Way, SRAM/Cache
                            Info for error.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status 
 *   @li                    CSL_SOK            - EDC get error address is 
 *                                               successful
 *   @li                    CSL_ESYS_INVPARAMS - The parameter passed is invalid
 *                                                
 *   <b> Pre Condition </b>
 *   @n  None
 *       
 *
 *   <b> Post Condition </b>
 *   @n  None
 *       
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
            CSL_Status          status;
            CSL_EdcAddrInfo    edcAddr
            ...
            status = CSL_edcGetErrorAddress (CSL_EDC_L1P, &edcAddr);
            ...
    @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_edcGetErrorAddress, ".text:csl_section:edc");
CSL_Status  CSL_edcGetErrorAddress (
    CSL_EdcMem          edcMem,
    CSL_EdcAddrInfo     *edcAddr
)
{
    CSL_Status status = CSL_SOK;
    Uint32     addr;
    
    switch (edcMem) {
        /* Get Address Information for L2 Error */
        case CSL_EDC_L2:       
            addr = (Uint32) CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDADDR, \
                                  EDC_L2EDADDR_ADDR); 
            edcAddr->addr = addr << 5; /* 5 LSBs assumed to be 0 */
                                        
            edcAddr->l2way = (CSL_EdcAddrL2way) \
                CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDADDR, \
                        EDC_L2EDADDR_L2WAY);
            edcAddr->sram  = (CSL_EdcAddrSram) \
                CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDADDR, \
                        EDC_L2EDADDR_SRAM);
            edcAddr->bitPos = (Uint8) 
                CSL_FEXTR(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 20,16);
            edcAddr->word32Bit = (Uint8) 
                CSL_FEXTR(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 23, 21);
        break;
        
        /* Get Address Information for L1P Error */
        case CSL_EDC_L1P:        
            addr = (Uint32)CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L1PEDADDR, \
                        EDC_L1PEDADDR_ADDR);
            edcAddr->addr = addr << 5; /* 5 LSBs assumed to be 0 */
                 
            edcAddr->sram = (CSL_EdcAddrSram) \
                CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L1PEDADDR, \
                        EDC_L1PEDADDR_SRAM);        
        break;
        
        default:
            status = CSL_ESYS_INVPARAMS;         
    }
    
    return status;    
}
