/*  ===========================================================================
 *  Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *  Use of this software is controlled by the terms and conditions found
 *  in the license agreement under which this software has been supplied
 *  provided
 *
 *  ===========================================================================
 */
 
/** ============================================================================
 *   @file  csl_edcGetHwStatus.c
 *
 *   @path  $(CSLPATH)\src\edc
 *
 *   @desc  File for functional layer of CSL API CSL_edcGetHwStatus()
 *
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  08-Jul-2004 Chad Courtney File created.
 *
 *  02-Sept-2005 ds modified.
 *
 * =============================================================================
 */
 
#include <csl_edc.h>

/** ============================================================================
 *   @n@b CSL_edcGetHwStatus
 *
 *   @b Description
 *   @n Gets the requested HW Status of the specified memory
 *
 *   @b Arguments
 *   @verbatim
            edcMem          Specificies what memory EDC status is to be 
                            obtained for.
 
            query           The query to this API which indicates the status
                            to be returned.
                            
            response        Placeholder to return the status.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status 
 *   @li                    CSL_SOK            - EDC get error address is 
 *                                               successful
 *   @li                    CSL_ESYS_INVPARAMS - The parameter passed is invalid
 *
 *   @li                    CSL_ESYS_INVQUERY  - Query command not supported                                             
 *   <b> Pre Condition </b>
 *   @n  None
 *       
 *
 *   <b> Post Condition </b>
 *   @n  None
 *       
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
            CSL_Status          status;
            Uint16              enStat;
            ...
            status = CSL_edcGetHwStatus (CSL_EDC_L1P, CSL_EDC_QUERY_ENABLESTAT,
                                        (void *)&enStat);
            ...
    @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_edcGetHwStatus, ".text:csl_section:edc")
CSL_Status  CSL_edcGetHwStatus (
    CSL_EdcMem              edcMem,
    CSL_EdcHwStatusQuery    query,
    void                    *response
)
{

    CSL_Status           status = CSL_SOK;
    CSL_EdcStatusInfo    *edcStatInfo;

    if (response == NULL) {
        return CSL_ESYS_INVPARAMS;
    }
    
    switch (query) {
        /* Query Enable/Disable status */
        case CSL_EDC_QUERY_ENABLESTAT:
            /* EDC HW Status query applies to L1P */      
            if(edcMem == CSL_EDC_L1P) {      
                *((CSL_EdcEnableStatus*)response) = (CSL_EdcEnableStatus) 
                    (CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L1PEDSTAT, 
                               EDC_L1PEDSTAT_LOGICEN)           
                     |CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L1PEDSTAT,
                               EDC_L1PEDSTAT_LOGICDIS) << 1 
                     |CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L1PEDSTAT, 
                               EDC_L1PEDSTAT_SUSP) << 2 ); 
                                
            }
            /* EDC HW Status query applies to L2 */
            else {
                if(edcMem == CSL_EDC_L2) {
                    *((CSL_EdcEnableStatus*)response) = (CSL_EdcEnableStatus)
                        (CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 
                                 EDC_L2EDSTAT_LOGICEN)
                        |CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 
                                 EDC_L2EDSTAT_LOGICDIS) << 1 
                        |CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 
                                 EDC_L1PEDSTAT_SUSP) << 2 ); 
                }
                else {
                    /* Incorrect Memory Type given for EDC Status Query */                          
                    status = CSL_ESYS_INVQUERY;
                }
            }
        break;
        
        /* Query error status */
        case CSL_EDC_QUERY_ERRORSTAT:       
            if(edcMem == CSL_EDC_L1P) {
                 /* EDC HW Status query applies to L1P */
                *((CSL_EdcErrorStatus*)response) =  (CSL_EdcErrorStatus)    
                    (CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L1PEDSTAT, 
                              EDC_L1PEDSTAT_IERR) << 1 
                    |CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L1PEDSTAT, 
                             EDC_L1PEDSTAT_DMAERR) << 2 );
            }
            else {
                /* EDC HW Status query applies to L2 */
                if(edcMem == CSL_EDC_L2) { 
                    *((CSL_EdcErrorStatus*)response) =  (CSL_EdcErrorStatus)    
                        (CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 
                                 EDC_L2EDSTAT_DERR)
                        |CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 
                                 EDC_L2EDSTAT_IERR) << 1 
                        |CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 
                                 EDC_L2EDSTAT_DMAERR) << 2 );
                }
                else {
                    /* Incorrect Memory Type given for EDC Status Query */                  
                    status = CSL_ESYS_INVQUERY;
                }
            }
        break;
        
        /* Query number of bit error status (L2 only) */
        case CSL_EDC_QUERY_NERRSTAT:        
            if(edcMem == CSL_EDC_L1P) {
                /* Invalid Query - L1P does not have NERR */      
                status = CSL_ESYS_INVQUERY; 
            }
            else {
                if(edcMem == CSL_EDC_L2) { 
                    *((CSL_EdcNumErrors*)response) = (CSL_EdcNumErrors) 
                        (CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 
                                 EDC_L2EDSTAT_NERR));
                }
                else {
                    /* Incorrect Memory Type given for EDC Status Query */
                    status = CSL_ESYS_INVQUERY;
                }
            }
        break;
        
        /* EDC HW Status query applies to L1P */
        case CSL_EDC_QUERY_BITPOS:          
            if(edcMem == CSL_EDC_L1P) {
                /* Invalid Query - L1P does not provide bit position */      
                status = CSL_ESYS_INVQUERY; 
            }
            else {
                if(edcMem == CSL_EDC_L2) { 
                    /* EDC HW Status query applies to L2 */
                    *((Uint32*)response) = 
                        CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 
                                EDC_L2EDSTAT_BITPOS);
                }
                else {
                    /* Incorrect Memory Type given for EDC Status Query */
                    status = CSL_ESYS_INVQUERY;
                }
            }
        break;
        
        /* Query all status (returns EDSTAT register) */
        case CSL_EDC_QUERY_ALLSTAT:
            /* EDC HW Status query applies to L1P */         
            if(edcMem == CSL_EDC_L1P) { 
                edcStatInfo = (CSL_EdcStatusInfo *)response;     
                edcStatInfo->enStat = 
                             CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L1PEDSTAT, 
                                     EDC_L1PEDSTAT_LOGICEN);
                edcStatInfo->disStat = 
                             CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L1PEDSTAT,
                                     EDC_L1PEDSTAT_LOGICDIS);
                edcStatInfo->suspStat = 
                             CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L1PEDSTAT, 
                                     EDC_L1PEDSTAT_SUSP);
                edcStatInfo->prgErr = 
                             CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L1PEDSTAT, 
                                     EDC_L1PEDSTAT_IERR);
                edcStatInfo->dmaErr = 
                             CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L1PEDSTAT, 
                                     EDC_L1PEDSTAT_DMAERR);
            }
            else {
                /* EDC HW Status query applies to L2 */
                if(edcMem == CSL_EDC_L2) { 
                    edcStatInfo = (CSL_EdcStatusInfo *)response;     
                    edcStatInfo->enStat = 
                              CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 
                                      EDC_L2EDSTAT_LOGICEN);
                    edcStatInfo->disStat = 
                              CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 
                                      EDC_L2EDSTAT_LOGICDIS);
                    edcStatInfo->suspStat = 
                              CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 
                                      EDC_L1PEDSTAT_SUSP);
                    edcStatInfo->prgErr = 
                              CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 
                                      EDC_L2EDSTAT_IERR);
                    edcStatInfo->dmaErr = 
                              CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 
                                 EDC_L2EDSTAT_DMAERR);
                    edcStatInfo->dataErr = 
                              CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDSTAT, 
                                      EDC_L2EDSTAT_DERR);
                }
                else {
                    /* Incorrect Memory Type given for EDC Status Query */
                    status = CSL_ESYS_INVQUERY;
                }
            }
        break;
        
        /* Query Page 0 enables (L2 only) */
        case CSL_EDC_QUERY_PAGE0:
            /* EDC HW Status query applies to L1P */           
            if(edcMem == CSL_EDC_L1P) {
                /* Invalid Query - L1P does not have Page Enables */      
                status = CSL_ESYS_INVQUERY; 
            }
            else {
                if(edcMem == CSL_EDC_L2) { 
                    /* EDC HW Status query applies to L2 */
                    *((Uint32*)response) = 
                        ((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDEN0;
                }
                else {
                    /* Incorrect Memory Type given for EDC Status Query */                          
                    status = CSL_ESYS_INVQUERY;
                }
            }
        break;
        
        /* Query Page 1 enables (L2 only) */
        case CSL_EDC_QUERY_PAGE1:           
            /* EDC HW Status query applies to L1P */
            if(edcMem == CSL_EDC_L1P) {
                /* Invalid Query - L1P does not have Page Enables */      
                status = CSL_ESYS_INVQUERY;
            }
            else {
                /* EDC HW Status query applies to L2 */
                if(edcMem == CSL_EDC_L2) { 
                    *((Uint32*)response) = 
                        ((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDEN1;
                }
                else {
                    /* Incorrect Memory Type given for EDC Status Query */                          
                    status = CSL_ESYS_INVQUERY;
                }
            }
        break;
        
        /* Query correctable parity error count (L2 only) */
        case CSL_EDC_QUERY_CE_CNT:          
            if(edcMem == CSL_EDC_L1P) {
                /* EDC HW Status query applies to L1P */
                /* Invalid Query - L1P does not provide error count */      
                status = CSL_ESYS_INVQUERY; 
            }
            else {
                /* EDC HW Status query applies to L2 */
                if(edcMem == CSL_EDC_L2) { 
                    *((Uint32*)response) = 
                        CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDCPEC, 
                                EDC_L2EDCPEC_CNT);
                }
                else {
                    /* Incorrect Memory Type given for EDC Status Query */                          
                    status = CSL_ESYS_INVQUERY;
                }
            }
        break;
        
        /* Query non-correctable parity error count (L2 only) */
        case CSL_EDC_QUERY_NCE_CNT:         
            if(edcMem == CSL_EDC_L1P) {
                /* EDC HW Status query applies to L1P */
                /* Invalid Query - L1P does not provide error count */
                status = CSL_ESYS_INVQUERY; 
            }
            else {
                /* EDC HW Status query applies to L2 */
                if(edcMem == CSL_EDC_L2) { 
                    *((Uint32*)response) = 
                        CSL_FEXT(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDNPEC, 
                                EDC_L2EDNPEC_CNT);
                }
                else {
                    /* Incorrect Memory Type given for EDC Status Query */                          
                    status = CSL_ESYS_INVQUERY;
                }
            }
        break;

        default:
            status = CSL_ESYS_INVQUERY ;

    }
    return status;
}
