/*  ===========================================================================
 *  Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *  Use of this software is controlled by the terms and conditions found
 *  in the license agreement under which this software has been supplied
 *  provided
 *
 *  ===========================================================================
 */
 
/** ============================================================================
 *   @file  csl_edcPageEnable.c
 *
 *   @path  $(CSLPATH)\src\edc
 *
 *   @desc  File for functional layer of CSL API CSL_edcPageEnable()
 *
 */
 
/* =============================================================================
 *  Revision History
 *  ===============
 *  08-Jul-2004 Chad Courtney File created.
 *
 *  02-Sept-2005 ds modified.
 *
 * =============================================================================
 */

#include <csl_edc.h>

/** ============================================================================
 *   @n@b CSL_edcPageEnable
 *
 *   @b Description
 *   @n Enables the pages for EDC specified by a 32-bit mask.
 *
 *   @b Arguments
 *   @verbatim
            mask        Specifies what pages of the given map(s) are to be 
                        enabled by setting the bit corresponding to 
                        the page to 1
                        
            umap        Specificies which map(s) to apply mask to (MAP0, MAP1, 
                        or BOTH)
 
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status 
 *   @li                    CSL_SOK            - Enable pages for EDC is 
                                                 successful 
 *   @li                    CSL_ESYS_INVPARAMS - The parameter passed is invalid
 *                                                
 *   <b> Pre Condition </b>
 *   @n  None
 *       
 *
 *   <b> Post Condition </b>
 *   @n  None
 *       
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
            CSL_Status          status;
            ...
            status = CSL_edcPageEnable (0x1, CSL_EDC_UMAP0);
            ...
    @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_edcPageEnable, ".text:csl_section:edc");
CSL_Status  CSL_edcPageEnable (
    Uint32              mask,
    CSL_EdcUmap         umap
)
{
    CSL_Status status = CSL_SOK;
    
    switch (umap) {
        /* Apply Page Enables to UMAP0 */
        case CSL_EDC_UMAP0:     
            CSL_FINS(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDEN0, 
                    EDC_L2EDEN0_EN, mask);      
        break;
        
        /* Apply Page Enables to UMAP1 */
        case CSL_EDC_UMAP1:     
            CSL_FINS(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDEN1, 
                    EDC_L2EDEN1_EN, mask);      
        break;
        
        /* Apply Page Enables to both UMAP0 and UMAP1 */
        case CSL_EDC_UMAPBOTH:  
            CSL_FINS(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDEN0, 
                    EDC_L2EDEN0_EN, mask);      
            CSL_FINS(((CSL_EdcRegs*)CSL_EDC_0_REGS)->L2EDEN1, 
                    EDC_L2EDEN1_EN, mask);      
        break;
        
        default:
            status = CSL_ESYS_INVPARAMS;         
    }
    
    return status;    
}
