/*  ===========================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *   ==========================================================================
 */
/** @file  csl_etbClose.c
 *
 *  @path $(CSLPATH)\etb\src
 *
 *  @brief File for functional layer of CSL API @a CSL_etbClose()
 *
 */
/* ============================================================================
 *  Revision History
 *  ===============
 *  
 *  10-Apr-2006 NG   File Created.                    
 * ============================================================================
 */
#include <csl_etb.h>

/** ===========================================================================
 *   @n@b CSL_etbClose
 *
 *   @b Description
 *   @n This function closes the specified instance of ETB. After successful 
 *      close, locks ETB i.e., by setting the LAR register of ETB in order to 
 *      disable accesses to any ETB registers.
 *
 *   @b Arguments
 *   @verbatim
            hEtb            Handle to the ETB

     @endverbatim
 *
 *   <b> Pre Condition </b>
 *   @n  Both @a CSL_etbInit() and @a CSL_etbOpen() must be called 
 *       successfully in order before calling @a CSL_etbClose().
 *
 *   <b> Post Condition </b>
 *   @n The ETB CSL APIs can not be called until the ETB CSL is reopened again 
 *      using CSL_etbOpen().
 *
 *   @b Modifies
 *   @n Obj structure values
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Close successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *
 *   @b Example
 *   @verbatim
            CSL_EtbHandle      hEtb;
            CSL_Status         status;

            ...
            
            status = CSL_etbClose(hEtb);
     @endverbatim
 * ===========================================================================
 */
#pragma CODE_SECTION (CSL_etbClose, ".text:csl_section:etb");

CSL_Status CSL_etbClose (
    CSL_EtbHandle	hEtb
)
{
    CSL_Status	status;

    if (hEtb != NULL) {
        hEtb->regs = (CSL_EtbRegsOvly) NULL;
        hEtb->etbNum = (CSL_InstNum) -1;
        status = CSL_SOK;
        /* lock ETB in order to disable accesses to any ETB registers */
        CSL_FINS(hEtb->regs->LAR, ETB_LAR_LAR, 0);
    }
    else {
        status = CSL_ESYS_BADHANDLE;
    }

    return status;
}

