/* =============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 * =============================================================================
 */ 
/** ============================================================================
 *   @file  csl_etbGetHwStatus.c
 *
 *   @path  $(CSLPATH)\etb\src
 *
 *   @desc  
 *
 */
/* =============================================================================
 *  Revision History
 *  ===============
 *  11-Apr-2006 NG  Author File Created 
 *  24-Apr-2006 NG  Added the query commands (CSL_ETB_QUERY_TRIGGERCOUNT, 
 *                  CSL_ETB_QUERY_ENAFORMATTING and
 *                  CSL_ETB_QUERY_ENACONTFORMATTING)
 * =============================================================================
 */
#include <csl_etb.h>
#include <csl_etbAux.h>

/** ============================================================================
 *   @n@b CSL_etbGetHwStatus
 *
 *   @b Description
 *   @n Gets the status of different operations or some setup-parameters of 
 *      ETB. The status is returned through the third parameter.
 *
 *   @b Arguments
 *   @verbatim
            hEtb     ETB handle returned by successful 'open'

            query     The query to this API of ETB which indicates the
                      status to be returned. 

                      Query command, refer @a CSL_EtbHwStatusQuery for the list 
                      of query commands supported 

            response  Response from the query. Pointer to appropriate object 
                      corresponding to the query command needs to be passed here

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Status info return successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVQUERY   - Invalid query command
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_etbInit() and CSL_etbOpen() must be called successfully 
 *       in order before calling CSL_etbGetHwStatus(). 
 *       Refer to @a CSL_EtbHwStatusQuery for the argument to be passed along 
 *       with the corresponding query command.
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n Third parameter response
 *
 *   @b Example
 *   @verbatim
        CSL_EtbHandle hEtb;
        CSL_Status    status;
        Uint32        response;
        ...
        status = CSL_etbGetHwStatus(hEtb,
                                     CSL_ETB_QUERY_ACQUISITION_COMPLETE,
                                     &response);
        ...


     @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_etbGetHwStatus, ".text:csl_section:etb");

CSL_Status  CSL_etbGetHwStatus (
    CSL_EtbHandle           hEtb,
    CSL_EtbHwStatusQuery    query,
    void                    *response
)
{
    CSL_Status status = CSL_SOK;

    if (hEtb == NULL) {
        return CSL_ESYS_BADHANDLE;
    }

    switch (query) {
        case CSL_ETB_QUERY_RAM_FULL:
            /* RAM Full (RAM write pointer has wrapped around) */
            CSL_etbGetRam(hEtb, response);
            break;

        case CSL_ETB_QUERY_TRIG_STAUS:
            /* Trigger bit set when a trigger has been observed */
            CSL_etbGetTrigSts(hEtb, response);
            break;

        case CSL_ETB_QUERY_ACQUISITION_COMPLETE:
            /* Acquisition complete */
            CSL_etbGetAcq(hEtb, response);
            break;

        case CSL_ETB_QUERY_FORMAT_PIPELINE:
            /* Formatterpipeline empty, All data stored to RAM */
            CSL_etbGetFormatPipeLine(hEtb, response);
            break;

        case CSL_ETB_QUERY_FLUSH:
            /* Flush in progress */
            CSL_etbGetFlush(hEtb, response);
            break;

        case CSL_ETB_QUERY_FORMAT_STOP:
            /* Formatter stopped */
            CSL_etbGetFormat(hEtb, response);
            break;

        case CSL_ETB_QUERY_ITATBDATA0_STATUS:
            /* Integration register Status ITATBDATA0 */
            *(Uint32 *)response = hEtb->regs->ITATBDATA0;
            break;

        case CSL_ETB_QUERY_ITATBCTR1_STATUS:
            /* Integration register Status ITATBCTR1 */
            *(Uint32 *)response = hEtb->regs->ITATBCTR1;
            break;

        case CSL_ETB_QUERY_ITATBCTR0_STATUS:
            /* Integration register Status ITATBCTR0 */
            *(Uint32 *)response = hEtb->regs->ITATBCTR0;
            break;

        case CSL_ETB_QUERY_SECURITY_LEVEL:
            /* Reports security level */
            CSL_etbGetSecurityLevel(hEtb, response);
            break;

        case CSL_ETB_QUERY_DEVICE_ID:
            /* DID - Device ID */
            CSL_etbGetDevId(hEtb, response);
            break;

        case CSL_ETB_QUERY_DEVICE_TYPE:
            /* Device Type Identification Register */
            CSL_etbGetDevType(hEtb, response);
            break;

        case CSL_ETB_QUERY_TRIGIN_VALUE:
            /* Read the value of TRIGIN */
            CSL_etbGetTrigVal(hEtb, response);
            break;

        case CSL_ETB_QUERY_FLUSHIN_VALUE:
            /* Read the value of FLUSHIN */
            CSL_etbGetFlushVal(hEtb, response);
            break;

        case CSL_ETB_QUERY_RAM_DEPTH :
            /* Read the RAM size */
            CSL_etbGetRamDepth(hEtb, response);
            break;       
            
        case CSL_ETB_QUERY_READ_POINTER:
            /* Read RAM read pointer value */
            CSL_etbGetReadPointer(hEtb, response);
            break;       

        case CSL_ETB_QUERY_WRITE_POINTER:
            /* Read RAM write pointer value */
            CSL_etbGetWritePointer(hEtb, response);
            break;       

        case CSL_ETB_QUERY_TRACECAP_STATUS:
            /* Read the trace capture value */
            CSL_etbGetTraceCapture(hEtb, response);
            break;       

        case CSL_ETB_QUERY_TRIGGERCOUNT:
            /* Read the value of Trigger Counter  */
            CSL_etbGetTriggerCounter(hEtb, response);
            break;       
            
        case CSL_ETB_QUERY_ENAFORMATTING:
            /* Get the value of Enable Formatting bit */
            CSL_etbGetEnableFormatting (hEtb, response);
            break;
            
        case CSL_ETB_QUERY_ENACONTFORMATTING:
            /* Get the value of Enable Continuous Formatting bit */
            CSL_etbGetEnableContFormatting (hEtb, response);
            break;     
            
        default:
            status = CSL_ESYS_INVQUERY;
            break;
    }
    return(status);
}

