/* =============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 * =============================================================================
 */ 
/** ============================================================================
 *   @file  csl_etbHwControl.c
 *
 *   @path  $(CSLPATH)\etb\src
 *
 *   @desc  
 *    - The CSL_etbHwControl() function definition & it's associated functions
 *
 */
/* =============================================================================
 *  Revision History
 *  ===============
 *  11-Apr-2006 NG  Author File Created 
 *
 * =============================================================================
 */
#include <csl_etb.h>

/** ============================================================================
 *   @n@b CSL_etbHwControl
 *
 *   @b Description
 *   @n This function takes an input control command with an optional argument
 *      and accordingly controls the operation/configuration of ETB.
 *
 *   @b Arguments
 *   @verbatim
            hEtb       ETB handle returned by successful 'open'

            cmd         The command to this API indicates the action to be
                        taken on ETB.
                        Control command, refer @a CSL_EtbControlCmd for the 
                        list of commands supported

            arg         An optional argument.
                        Optional argument as per the control command, 
                        @a void * casted

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li         CSL_SOK               - Status info return successful.
 *   @li         CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li         CSL_ESYS_INVCMD       - Invalid command
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_etbInit() and CSL_etbOpen() must be called successfully 
 *       in order before calling CSL_etbHwControl(). 
 *
 *       Refer to @a CSL_EtbHwControlCmd for the argument type (@a void*) that 
 *       needs to be passed with the control command
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n The hardware registers of ETB.
 *
 *   @b Example
 *   @verbatim
      CSL_Status    status;
      Uint32        arg;
      CSL_EtbHandle hEtb;
      ...
      // Init successfully done
      ...
      // Open successfully done
      ...
       
      arg = CSL_ETB_TRACEACPEN_ENABLE;
      status = CSL_etbHwControl(hEtb,
                                CSL_ETB_CMD_ENA_TRACE_CAPTURE,
                                &arg);
      ...

     @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_etbHwControl, ".text:csl_section:etb");

CSL_Status  CSL_etbHwControl (
    CSL_EtbHandle        hEtb,
    CSL_EtbControlCmd    cmd,
    void*                arg
)
{
    CSL_Status          status  = CSL_SOK;
    Uint32              argv = *(Uint32*)arg;
    
    if (hEtb == NULL) {
        status = CSL_ESYS_BADHANDLE;
        return status;
    }

    switch (cmd) {

        case CSL_ETB_CMD_SET_RAM_RD_POINTER:
            /* RAM read pointer register */
            CSL_FINS(hEtb->regs->RRP, ETB_RRP_RRP, argv);
            break;

        case CSL_ETB_CMD_SET_RAM_WR_POINTER:
            /* RAM write pointer register */
            CSL_FINS(hEtb->regs->RWP, ETB_RWP_RWP, argv);
            break;

        case CSL_ETB_CMD_SET_TRIGG_COUNT:
            /* Trigger Counter Register */
            CSL_FINS(hEtb->regs->TRG, ETB_TRG_TRG, argv);
            break;

        case CSL_ETB_CMD_ENA_TRACE_CAPTURE:
            /* Trace Capture Enable */
            CSL_FINST(hEtb->regs->CTL, ETB_CTL_TRACEACPEN, ENABLE);
            break;

        case CSL_ETB_CMD_DIS_TRACE_CAPTURE:
            /* Trace Capture Disable */
            CSL_FINST(hEtb->regs->CTL, ETB_CTL_TRACEACPEN, DISABLE);
            break;

        case CSL_ETB_CMD_SET_ENAFORMATTING:
            /* Enable Formatting */
            CSL_FINS(hEtb->regs->FFCR, ETB_FFCR_ENFTC, argv);
            break;

        case CSL_ETB_CMD_SET_CONT_FORMATTING:
            /* Continuous Formatting */
            CSL_FINS(hEtb->regs->FFCR, ETB_FFCR_ENFCNT, argv);
            break;

        case CSL_ETB_CMD_ENA_FLUSHIN:
            /* Enable FLUSHIN */
            CSL_FINS(hEtb->regs->FFCR, ETB_FFCR_FONFIIN, 1);
            break;

        case CSL_ETB_CMD_SET_MANUAL_FLUSH:
            /* FONMAN - Set Manual Flush */
            CSL_FINS(hEtb->regs->FFCR, ETB_FFCR_FONMAN, 1);
            break;

        case CSL_ETB_CMD_ENA_STOP_FLUSH:
            /* STOPFI - Stop flush enable */
            CSL_FINS(hEtb->regs->FFCR, ETB_FFCR_STOPFI, 1);
            break;

        case CSL_ETB_CMD_SET_ACQCOMP:
            /* Set the value of ACQCOMP */
            CSL_FINS(hEtb->regs->ITMISCOP0, ETB_ITMISCOP0_ACQCOMP, argv);
            break;

        case CSL_ETB_CMD_SET_FULL:
            /* Set the value of FULL */
            CSL_FINS(hEtb->regs->ITMISCOP0, ETB_ITMISCOP0_FULL, argv);
            break;

        case CSL_ETB_CMD_SET_TRIGACK:
            /* Set the value of TRIGACK */
            CSL_FINS(hEtb->regs->ITTRFLINACK, ETB_ITTRFLINACK_TRIGINACK, argv);
            break;

        case CSL_ETB_CMD_SET_FLUSHACK:
            /* Set the value of FLUSHACK */
            CSL_FINS(hEtb->regs->ITTRFLINACK, ETB_ITTRFLINACK_FLUSHINACK, argv);
            break;

        case CSL_ETB_CMD_SET_ATREADY:
            /* Set the value of ATREADY */
            CSL_FINS(hEtb->regs->ITATBCTR2, ETB_ITATBCTR2_ATREADY, argv);
            break;

        case CSL_ETB_CMD_SET_ATVALID:
            /* Set the value of ATVALID */
            CSL_FINS(hEtb->regs->ITATBCTR2, ETB_ITATBCTR2_AFVALID, argv);
            break;

        default:
            status = CSL_ESYS_INVCMD;
            break;
    }
    return (status);
}
