/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *   ===========================================================================
 */
/** @file  csl_etbOpen.c
 *
 *  @path $(CSLPATH)\etb\src
 *
 *  @brief File for functional layer of CSL API @a CSL_etbOpen()
 *
 */
/* =============================================================================
 * Revision History
 * ===============
 *
 *  10-Apr-2006 NG  Author File created
 * =============================================================================
 */
#include <csl_etb.h>

/** ============================================================================
 *   @n@b CSL_etbOpen
 *
 *   @b Description
 *   @n This function returns the handle to the ETB controller
 *      instance. This handle is passed to all other CSL APIs.  After successful 
 *      open, unlocks ETB i.e., by setting the LAR register of ETB in order to 
 *      enable accesses to any ETB registers.
 *
 *   @b Arguments
 *   @verbatim
 *          pEtbObj     Pointer to etb object.
 *
 *          etbNum      Instance of  DSP ETB to be opened. 
 *                      There are three instances of the etb available.
 *                      The instances are CSL_ETB_0, CSL_ETB_1 and CSL_ETB_2.
 *
 *          pEtbParam   Module specific parameters.
 *
 *          pStatus     Status of the function call
 *    @endverbatim
 *
 *   <b> Return Value </b>  CSL_EtbHandle
 *   @n                         Valid etb handle will be returned if
 *                              status value is equal to CSL_SOK.
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_etbInit() must be called successfully in order before calling 
 *       CSL_etbOpen().
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned in the status variable. If status
 *              returned is
 *   @li            CSL_SOK             Valid etb handle is returned
 *   @li            CSL_ESYS_FAIL       The etb instance is invalid
 *   @li            CSL_ESYS_INVPARAMS  Invalid parameter
 *
 *        2.    ETB object structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. ETB object structure
 *
 *   @b Example
 *   @verbatim
 *          CSL_Status         status;
 *          CSL_EtbObj         etbObj;
 *          CSL_EtbHandle      hEtb;
 *
 *          ...
 *               
 *          hEtb = CSL_etbOpen(&etbObj, 
 *							   CSL_ETB_0, 
 *							   NULL, 
 *							   &status
 *							  );
 *           
 *          ...
 *  @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_etbOpen, ".text:csl_section:etb");

CSL_EtbHandle CSL_etbOpen (
    CSL_EtbObj 	    *pEtbObj,
    CSL_InstNum 	etbNum,
    CSL_EtbParam 	*pEtbParam,
    CSL_Status 		*pStatus
)
{
    CSL_Status              status;
    CSL_EtbHandle           hEtb;
    CSL_EtbBaseAddress      baseAddress;

    if (pStatus == NULL) {
        *pStatus = CSL_ESYS_INVPARAMS;
        hEtb = (CSL_EtbHandle) NULL;
        return hEtb;
    }

    if (pEtbObj == NULL) {
        *pStatus = CSL_ESYS_INVPARAMS;
        hEtb = (CSL_EtbHandle) NULL;
        return hEtb;
    }

    status = CSL_etbGetBaseAddress(etbNum, pEtbParam, &baseAddress);

    if (status == CSL_SOK) {
        pEtbObj->regs = baseAddress.regs;
        pEtbObj->etbNum = (CSL_InstNum)etbNum;
        hEtb = (CSL_EtbHandle)pEtbObj;
        /* Unlock ETB in order to enable accesses to any ETB registers */
        CSL_FINS(hEtb->regs->LAR, ETB_LAR_LAR, CSL_ETB_UNLOCK_VAL);
    }
    else {
        pEtbObj->regs = (CSL_EtbRegsOvly) NULL;
        pEtbObj->etbNum = (CSL_InstNum) -1;
        hEtb = (CSL_EtbHandle) NULL;
    }

    *pStatus = status;

    return hEtb;
}

