/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */
 
/** ===========================================================================
 *   @file  csl_fsyncGetHwStatus.c
 *
 *   @brief File for functional layer of CSL API CSL_fsyncGetHwStatus() 
 *
 *   @path  $(CSLPATH)\src\fsync
 *
 *   @desc  The CSL_fsyncGetHwStatus() function definition.
 */ 

/* =============================================================================
 * Revision History
 * ===============
 *  07-Mar-2006 NS updated the file for DOXYGEN compatibiliy
 *
 * =============================================================================
 */
   
#include <csl_fsyncAux.h>
/** ============================================================================
 *   @n@b CSL_fsyncGetHwStatus
 *
 *   @b Description
 *   @n This function is used to get the value of various parameters of the
 *      fsync instance. The value returned depends on the query passed.
 *
 *   @b Arguments
 *   @verbatim
            hFsync        Handle to the FSYNC instance
 
            query         Query to be performed 
 
            response      Pointer to buffer to return the data requested by
                          the query passed
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK            - Successful completion of the
 *                                               query
 *
 *   @li                    CSL_ESYS_INVQUERY  - Query command not supported
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *       Data requested by query is returned through the variable "response"
 *
 *   @b Modifies
 *   @n The input argument "response" is modified.      
 *
 *   @b Example
 *   @verbatim
         // handle for Fsync
         CSL_FsyncHandle handleFsync;
         //  Fsync object
         CSL_FsyncObj myFsyncObj;
         // Fsync setup 
         CSL_FsyncSetup myFsyncCfg;
         // CSL status
         CSL_Status status;
         CSL_FsyncTimerTermCountObj terminalCount;    
         // ctrl arg for Hw command
         CSL_BitMask16 ctrlArg;
         // query response
         CSL_FsyncTimerCountObj queryRp3TimerValue;

         // Do config Fsync with RP1 interface 
         myFsyncCfg.syncRP3Timer = CSL_FSYNC_FRAME_BURST;   

         // Initialize CSL library, this step is not required 
         CSL_fsyncInit(NULL);

         // Open handle to FSync 
         handleFsync = CSL_fsyncOpen(&myFsyncObj, CSL_FSYNC, NULL, &status);

         // FSync configuration 
         myFsyncCfg.syncRP3Timer = CSL_FSYNC_FRAME_BURST;
         myFsyncCfg.syncSystemTimer = CSL_FSYNC_FRAME_BURST;
         myFsyncCfg.clkRP3Timer = CSL_FSYNC_FRAME_SYNC_CLK;
         myFsyncCfg.clkSystemTimer = CSL_FSYNC_FRAME_SYNC_CLK;
         myFsyncCfg.pTerminalCountRP3Timer = &terminalCount;
         myFsyncCfg.pTerminalCountSystemTimer = &terminalCount;
         myFsyncCfg.systemTimerRp1Sync = CSL_FSYNC_RP1_TYPE_WCDMA_FDD_FRAME_NUM;
         myFsyncCfg.rp3EqualsSysTimer = FALSE;
         myFsyncCfg.syncMode = CSL_FSYNC_RP1_SYNC_MODE;
         myFsyncCfg.reSyncMode = CSL_FSYNC_AUTO_RESYNC_MODE;
         myFsyncCfg.crcUsage = CSL_FSYNC_USE_SYNC_BURST_ON_CRC_FAIL;
    
        // FSync h/w setup
        CSL_fsyncHwSetup(handleFsync, &myFsyncCfg);
        // Enable timer
        ctrlArg = CSL_FSYNC_RP3_TIMER_ENABLE | CSL_FSYNC_SYSTEM_TIMER_ENABLE |
                 CSL_FSYNC_TOD_TIMER_ENABLE;
   
        CSL_fsyncHwControl(handleFsync, CSL_FSYNC_CMD_ENABLE_TIMER, (void *)&ctrlArg);
        // Read value of RP3 timer 
        CSL_fsyncGetHwStatus(handleFsync, CSL_FSYNC_QUERY_RP3_TIMER_VALUE, (void *)&queryRp3TimerValue);
      @endverbatim
 * =============================================================================
 */

#pragma CODE_SECTION (CSL_fsyncGetHwStatus, ".text:csl_section:fsync");
CSL_Status  CSL_fsyncGetHwStatus(
   CSL_FsyncHandle                      hFsync,
   CSL_FsyncHwStatusQuery               query,
   void                                 *response
)
{
   CSL_Status st = CSL_SOK;

   if (hFsync == NULL) {
       return CSL_ESYS_BADHANDLE;
   }

   switch (query)
   {
      case CSL_FSYNC_QUERY_PID:
            
         CSL_fsyncGetPid(hFsync, (CSL_BitMask32*)response);
         break;

      case CSL_FSYNC_QUERY_RP3_TIMER_VALUE:
         
         CSL_fsyncGetRp3TimerValue(hFsync, (CSL_FsyncTimerCountObj *)response);
         break;
      
      case CSL_FSYNC_QUERY_SYSTEM_TIMER_VALUE:
         
         CSL_fsyncGetSysTimerValue(hFsync, (CSL_FsyncTimerCountObj *)response);
         break;
      
      case CSL_FSYNC_QUERY_TOD_TIMER_VALUE:
         
         CSL_fsyncGetTodTimerValue(hFsync, (Uint32*)response);
         break;

      case CSL_FSYNC_QUERY_ERR_EVENT_0:
         
         CSL_fsyncGetErrEvent0(hFsync, (CSL_BitMask32*)response);
         break;

      case CSL_FSYNC_QUERY_ERR_EVENT_1:
         
         CSL_fsyncGetErrEvent1(hFsync, (CSL_BitMask32*)response);
         break;

      case CSL_FSYNC_QUERY_RP1_TYPE_CAPTURE:
         
         CSL_fsyncGetRp1TypeCapture(hFsync, (CSL_BitMask32*)response);
         break;
      
      case CSL_FSYNC_QUERY_RP1_PAYLOAD_CAPTURE_TOD_TYPE:
         
         CSL_fsyncGetCapTodType(hFsync,(CSL_FsyncRp1PayloadObj *)response);
         break;
      
      case CSL_FSYNC_QUERY_RP1_PAYLOAD_CAPTURE_RP3_TYPE:
         
         CSL_fsyncGetCapRp3Type(hFsync,(CSL_FsyncRp1PayloadObj *)response);
         break;

      case CSL_FSYNC_QUERY_RP1_PAYLOAD_CAPTURE_WCDMA_FDD_TYPE:
         
         CSL_fsyncGetCapWcdmaFddType(hFsync,(CSL_FsyncRp1PayloadObj *)response);
         break;

      case CSL_FSYNC_QUERY_OK_STATUS_BIT:
         
         *(Uint8*)response  = CSL_fsyncGetOkStatusBit(hFsync);
         break;
         
       case CSL_FSYNC_QUERY_ERR_INT_MASK0_REG:  
         CSL_fsyncGetErrIntMask0Reg(hFsync, (CSL_BitMask32 *)response);
         break;
         
       case CSL_FSYNC_QUERY_ERR_INT_MASK1_REG:  
         CSL_fsyncGetErrIntMask1Reg(hFsync, (CSL_BitMask32 *)response);
         break;
         
       case CSL_FSYNC_QUERY_ERR_INT_SRC_RAW_REG:  
         CSL_fsyncGetErrIntSourceRawReg(hFsync, (CSL_BitMask32 *)response);
         break;
         
      default:
         st = CSL_ESYS_INVQUERY;
         break;

   }        
   return(st);
}


