/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */
 
/** ===========================================================================
 *   @file  csl_fsyncHwControl.c
 *
 *   @brief File for functional layer of CSL API CSL_fsyncHwControl() 
 *
 *   @path  $(CSLPATH)\src\fsync
 *
 *   @desc  The CSL_fsyncHwControl() function definition.
 */
 

/* =============================================================================
 * Revision History
 * ===============
 *  07-Mar-2006 NS updated the file for DOXYGEN compatibiliy
 *
 * =============================================================================
 */
 
#include <csl_fsync.h>
#include <csl_fsyncAux.h>

/** ============================================================================
 *   @n@b CSL_fsyncHwControl
 *
 *   @b Description
 *   @n Takes a command of FSYNC with an optional argument & implements it.
 *
 *   @b Arguments
 *   @verbatim
            hFsync          Handle to the FSYNC instance

            cmd             The command to this API indicates the action to be
                            taken on FSYNC.

            arg             An optional argument.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Status info return successful.
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVCMD     - Invalid command
 *                               
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n The hardware registers of FSYNC.
 *
 *   @b Example
 *   @verbatim
        // handle for Fsync
        CSL_FsyncHandle handleFsync;
        //  Fsync object
        CSL_FsyncObj myFsyncObj;
        // Fsync setup 
        CSL_FsyncSetup myFsyncCfg;
        // CSL status
        CSL_Status status;
   
        CSL_FsyncTimerTermCountObj terminalCount;
 
        // ctrl arg for Hw command
        CSL_BitMask16 ctrlArg;

        // Do config Fsync with RP1 interface 
        myFsyncCfg.syncRP3Timer = CSL_FSYNC_FRAME_BURST;   

        // Initialize CSL library, this step is not required 
        CSL_fsyncInit(NULL);

        // Open handle to FSync 
        handleFsync = CSL_fsyncOpen(&myFsyncObj, CSL_FSYNC, NULL, &status);

        // FSync configuration 
        myFsyncCfg.syncRP3Timer = CSL_FSYNC_FRAME_BURST;
        myFsyncCfg.syncSystemTimer = CSL_FSYNC_FRAME_BURST;
        myFsyncCfg.clkRP3Timer = CSL_FSYNC_FRAME_SYNC_CLK;
        myFsyncCfg.clkSystemTimer = CSL_FSYNC_FRAME_SYNC_CLK;
        myFsyncCfg.pTerminalCountRP3Timer = &terminalCount;
        myFsyncCfg.pTerminalCountSystemTimer = &terminalCount;
        myFsyncCfg.systemTimerRp1Sync = CSL_FSYNC_RP1_TYPE_WCDMA_FDD_FRAME_NUM;
        myFsyncCfg.rp3EqualsSysTimer = FALSE;
        myFsyncCfg.syncMode = CSL_FSYNC_RP1_SYNC_MODE;
        myFsyncCfg.reSyncMode = CSL_FSYNC_AUTO_RESYNC_MODE;
        myFsyncCfg.crcUsage = CSL_FSYNC_USE_SYNC_BURST_ON_CRC_FAIL;
    
        // FSync h/w setup
        CSL_fsyncHwSetup(handleFsync, &myFsyncCfg);
        // Enable timer
        ctrlArg = CSL_FSYNC_RP3_TIMER_ENABLE | CSL_FSYNC_SYSTEM_TIMER_ENABLE |
                 CSL_FSYNC_TOD_TIMER_ENABLE;
   
        CSL_fsyncHwControl(handleFsync, CSL_FSYNC_CMD_ENABLE_TIMER, (void *)&ctrlArg);

     @endverbatim
 * =============================================================================
 */

#pragma CODE_SECTION (CSL_fsyncHwControl, ".text:csl_section:fsync");
CSL_Status  CSL_fsyncHwControl(
   CSL_FsyncHandle                      hFsync,
   CSL_FsyncHwControlCmd                ctrlCmd,
   void                                 *arg
)
{
   CSL_Status status = CSL_SOK;
   if (hFsync == NULL) {
       return CSL_ESYS_BADHANDLE;
   }

   switch (ctrlCmd)
   {  
      case CSL_FSYNC_CMD_ENABLE_TIMER:
         CSL_fsyncEnableTimer(hFsync);
         break;
      
      case CSL_FSYNC_CMD_DISABLE_TIMER:
         CSL_fsyncDisableTimer(hFsync);
         break;

      case CSL_FSYNC_CMD_HALT_TIMER:
         CSL_fsyncHaltTimer(hFsync);
         break;

      case CSL_FSYNC_CMD_ENABLE_RP1_SYNC_MODE:
         CSL_fsyncEnableRp1SyncMode(hFsync);
         break;

      case CSL_FSYNC_CMD_DISABLE_RP1_SYNC_MODE:
         CSL_fsyncDisableRp1SyncMode(hFsync);
         break;
 
      case CSL_FSYNC_CMD_TRIGGER_SYSTEM_TEST_SYNC:
         CSL_fsyncTrigSysTestSync(hFsync);
         break;

      case CSL_FSYNC_CMD_CHNG_SYNC_BURST_FRAME_UPDATE_RATE_TOD_INFO:
         CSL_fsyncSetUpdateTodRate(hFsync, *(Uint32 *)arg );
         break;

      case CSL_FSYNC_CMD_CHNG_SYNC_BURST_FRAME_UPDATE_RATE_RP3_INFO:
         CSL_fsyncSetUpdateRp3Rate(hFsync, *(Uint32 *)arg );
         break;

      case CSL_FSYNC_CMD_CHNG_SYNC_BURST_FRAME_UPDATE_RATE_WCDMA_INFO:
         CSL_fsyncSetUpdateSysRate(hFsync, *(Uint32 *)arg );
         break;

      case CSL_FSYNC_CMD_CHNG_INIT_RP3_TIMER_VALUE:
         CSL_fsyncSetInitRp3Value(hFsync, *(CSL_FsyncTimerCountObj *)arg );
         break;
      
      case CSL_FSYNC_CMD_CHNG_INIT_SYSTEM_TIMER_VALUE:
         CSL_fsyncSetInitSysValue(hFsync, *(CSL_FsyncTimerCountObj *)arg );
         break;

      case CSL_FSYNC_CMD_SET_EMU_CONTROL:
         CSL_fsyncSetEmuControl(hFsync, *(CSL_BitMask16 *)arg );
         break;

      case CSL_FSYNC_CMD_SET_EMU_MASK:
         CSL_fsyncSetEmuMask(hFsync, *(CSL_BitMask32 *)arg );
         break;

      case CSL_FSYNC_CMD_ENABLE_ERR_EVENT:
         CSL_fsyncEnableErrEvt(hFsync, *(CSL_FsyncErrEventMaskObj *)arg );
         break;

      case CSL_FSYNC_CMD_DISABLE_ERR_EVENT:
         CSL_fsyncDisableErrEvt(hFsync, *(CSL_FsyncErrEventMaskObj *)arg );
         break;

      case CSL_FSYNC_CMD_SET_ERR_EVENT:
         CSL_fsyncSetErrEvt(hFsync, *(CSL_BitMask32 *)arg );
         break;

      case CSL_FSYNC_CMD_CLEAR_ERR_EVENT:
         CSL_fsyncClearErrEvt(hFsync, *(CSL_BitMask32 *)arg );
         break;

      case CSL_FSYNC_CMD_REEVALUATE_ERR_INTERRUT_LINE:
         CSL_fsyncRevalErrIntrLine(hFsync, *(CSL_FsyncErrAlarmIndex *)arg );
         break;


      case CSL_FSYNC_CMD_CONFIG_MASK_BASED_TRIGGER_GEN:
         CSL_fsyncSetupMaskEventGen(hFsync, *(CSL_FsyncMaskTriggerGenObj *)arg);
         break;

      case CSL_FSYNC_CMD_CONFIG_COUNTER_BASED_TRIGGER_GEN:
         CSL_fsyncSetupCountEventGen(hFsync, *(CSL_FsyncCounterTriggerGenObj *)arg);
         break;

      case CSL_FSYNC_CMD_ENABLE_TRIGGER_GEN:
         CSL_fsyncEnableEventGen(hFsync, *(CSL_FsyncTriggerGenNum *)arg);
         break;

      case CSL_FSYNC_CMD_DISABLE_TRIGGER_GEN:
         CSL_fsyncDisableEventGen(hFsync, *(CSL_FsyncTriggerGenNum *)arg);
         break;

      case CSL_FSYNC_CMD_SET_TRIGGER_EVENT:
         CSL_fsyncSetEvent(hFsync, *(CSL_FsyncTriggerGenNum *)arg);
         //hFsync->regs->EVTFORCE = *(CSL_FsyncTriggerGenNum *)arg;
         break;
      
      case CSL_FSYNC_CMD_SET_OK_STATUS_BIT:
         CSL_fsyncSetOkStatBit(hFsync);
         break;

      case CSL_FSYNC_CMD_CLEAR_OK_STATUS_BIT:
         CSL_fsyncClrOkStatBit(hFsync);
         
         break;
           
      case CSL_FSYNC_CMD_ENABLE_CRC_INVERT:
         CSL_fsyncCrcInvert(hFsync);
         break;
         
      case CSL_FSYNC_CMD_CLEAR_TYPE_SELECT:   
         CSL_fsyncClearTypeSelect(hFsync);
         break;
         
      case CSL_FSYNC_CMD_CHNG_SYS_TIMER_RP1_SYNC:
         CSL_fsyncChngTmrRp1Sync(hFsync, *(CSL_FsyncRp1TypeField *)arg);
         break;
         
      case CSL_FSYNC_CMD_CHNG_CRC_INIT_VALUE:
         CSL_FsyncChngCrcInit(hFsync, *(CSL_FsyncRp1CrcInitValue *)arg);
         break;
         
      case CSL_FSYNC_CMD_CHNG_CRC_POSITION:
         CSL_FsyncChngCrcPosition(hFsync, *(CSL_FsyncRp1CrcPosition *)arg);
         break;      
         
      default:
         status = CSL_ESYS_INVCMD;
         break;
   }

   return (status);

}




