/*  ==========================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */
 
/** ===========================================================================
 *   @file  csl_fsyncHwSetup.c
 *
 *   @brief File for functional layer of CSL API CSL_fsyncHwSetup() 
 *
 *   @path  $(CSLPATH)\src\fsync
 *
 *   @desc  The CSL_fsyncHwSetup() function definition.
 */ 

/* =============================================================================
 * Revision History
 * ===============
 *  07-Mar-2006 NS updated the file for DOXYGEN compatibiliy
 *
 * =============================================================================
 */

#include <csl_fsync.h>

/** ============================================================================
 *   @n@b CSL_fsyncHwSetup
 *
 *   @b Description       
 *   @n It configures the FSYNC instance registers as per the values passed
 *      in the hardware setup structure.
 *
 *   @b Arguments
 *   @verbatim
            hFsync            Handle to the FSYNC instance

            pFsyncSetup       Pointer to hardware setup structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Hardware setup successful.
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Hardware structure is not
 *                                                properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The specified instance will be setup according to value passed.
 *
 *   @b Modifies
 *   @n The hardware registers of FSYNC.
 *
 *   @b Example                  
 *   @verbatim
            // handle for Fsync
            CSL_FsyncHandle handleFsync;
            //  Fsync object
            CSL_FsyncObj myFsyncObj;
            // Fsync setup 
            CSL_FsyncSetup myFsyncCfg;

            CSL_FsyncTimerTermCountObj terminalCount;

            // CSL status
            CSL_Status status;
 
            // FSync configuration 
            myFsyncCfg.syncRP3Timer = CSL_FSYNC_FRAME_BURST;
            myFsyncCfg.syncSystemTimer = CSL_FSYNC_FRAME_BURST;
            myFsyncCfg.clkRP3Timer = CSL_FSYNC_FRAME_SYNC_CLK;
            myFsyncCfg.clkSystemTimer = CSL_FSYNC_FRAME_SYNC_CLK;
            myFsyncCfg.pTerminalCountRP3Timer = &terminalCount;
            myFsyncCfg.pTerminalCountSystemTimer = &terminalCount;
            myFsyncCfg.systemTimerRp1Sync = CSL_FSYNC_RP1_TYPE_WCDMA_FDD_FRAME_NUM;
            myFsyncCfg.rp3EqualsSysTimer = FALSE;
            myFsyncCfg.syncMode = CSL_FSYNC_RP1_SYNC_MODE;
            myFsyncCfg.reSyncMode = CSL_FSYNC_AUTO_RESYNC_MODE;
            myFsyncCfg.crcUsage = CSL_FSYNC_USE_SYNC_BURST_ON_CRC_FAIL;
    
   
            // Initialize CSL library, this step is not required 
            CSL_fsyncInit(NULL);

            // Open handle to FSync 
            handleFsync = CSL_fsyncOpen(&myFsyncObj, CSL_FSYNC, NULL, &status);
   
            // Do setup for FSync
            CSL_fsyncHwSetup(handleFsync, &myFsyncCfg); 
   @endverbatim
 * =============================================================================
 */ 
 
#pragma CODE_SECTION (CSL_fsyncHwSetup, ".text:csl_section:fsync");
                                                                        
 CSL_Status  CSL_fsyncHwSetup(
    /** Pointer to the object that holds reference to the
     *  instance of FSYNC link requested after the call
    */
    CSL_FsyncHandle             hFsync,
    /** Pointer to setup structure which contains the
     *  information to program FSYNC to a useful state
    */
    CSL_FsyncSetup              *pFsyncSetup
)
{
   Uint32 tempRegVal;
   Uint8 count;
   CSL_FsyncTimerTermCountObj    *pSysTimerTc, *pRp3TimerTc;
   CSL_FsyncTimerCountObj        *pSysTimerInit, *pRp3TimerInit;
      
   if (hFsync == NULL) {
        return CSL_ESYS_BADHANDLE;
    }
      
   if (pFsyncSetup == NULL)
   {
      return(CSL_ESYS_INVPARAMS);
   }
   // CTL1 reg       
   tempRegVal =   CSL_FMK(FSYNC_CTL1_RP3_SYNC_SEL, pFsyncSetup->syncRP3Timer) |
                  CSL_FMK(FSYNC_CTL1_SYS_SYNC_SEL, pFsyncSetup->syncSystemTimer) |
                  CSL_FMK(FSYNC_CTL1_RP3_CLK_SEL,  pFsyncSetup->clkRP3Timer) |
                  CSL_FMK(FSYNC_CTL1_SYS_CLK_SEL,  pFsyncSetup->clkSystemTimer) |
                  CSL_FMK(FSYNC_CTL1_RP3_FRAME_DLY,  pFsyncSetup->rp3SyncDelay) |
                  CSL_FMK(FSYNC_CTL1_SYS_FRAME_DLY,  pFsyncSetup->systemSyncDelay) |
                  CSL_FMK(FSYNC_CTL1_TOD_FRAME_DLY,  pFsyncSetup->todSyncDelay) |
                  CSL_FMK(FSYNC_CTL1_RP3_EQUAL_SYS,  pFsyncSetup->rp3EqualsSysTimer) |
                  CSL_FMK(FSYNC_CTL1_RP1_MODE,  pFsyncSetup->syncMode) |
                  CSL_FMK(FSYNC_CTL1_AUTO_RESYNC,  pFsyncSetup->reSyncMode) |
                  CSL_FMK(FSYNC_CTL1_CRC_USE,  pFsyncSetup->crcUsage) |
                  CSL_FMK(FSYNC_CTL1_ADD_LEAPSEC,  pFsyncSetup->todLeapUsage) |
                  CSL_FMK(FSYNC_CTL1_CRC_FLIP,  pFsyncSetup->crcPosition) |
                  CSL_FMK(FSYNC_CTL1_CRC_INIT_ONE,  pFsyncSetup->crcInitValue) ;
   
   hFsync->regs->CTL1 = tempRegVal;
   
   CSL_FINS(hFsync->regs->RP1TS, FSYNC_RP1TS_RP1_TYPE, pFsyncSetup->systemTimerRp1Sync);
   
   tempRegVal =   CSL_FMK(FSYNC_UPDATE_TOD_RATE, pFsyncSetup->setupWatchDog.todFrameUpdateRate) |
                  CSL_FMK(FSYNC_UPDATE_RP3_RATE, pFsyncSetup->setupWatchDog.rp3FrameUpdateRate) |
                  CSL_FMK(FSYNC_UPDATE_SYS_RATE,  pFsyncSetup->setupWatchDog.wcdmaFrameUpdateRate);
   
   hFsync->regs->UPDATE = tempRegVal;
   

   // Sytem timer terminal count
   pSysTimerTc = pFsyncSetup->pTerminalCountSystemTimer; 
   
   if (pSysTimerTc != NULL)
   {
   
      tempRegVal = CSL_FMK(FSYNC_SYSTC_SUBCHIP, pSysTimerTc->lastSampleNum) |
                   CSL_FMK(FSYNC_SYSTC_SLOT, pSysTimerTc->lastSlotNum) |
                   CSL_FMK(FSYNC_SYSTC_WRAPADDR, pSysTimerTc->numChipTerminalCount-1);
   
      hFsync->regs->SYSTC = tempRegVal;
   
      // system timer terminal chip count ram
      for (count=0; count < pSysTimerTc->numChipTerminalCount; count++)
      {
         tempRegVal = CSL_FMK(FSYNC_SYSTC_TCOUNT, pSysTimerTc->pLastChipNum[count]);
         hFsync->regs->SYSTC_RAM[count].SYSTC = tempRegVal;
      }
   }

   // RP3 timer terminal count
   pRp3TimerTc = pFsyncSetup->pTerminalCountRP3Timer; 
   
   if (pRp3TimerTc != NULL)
   {
      tempRegVal = CSL_FMK(FSYNC_RP3TC_SUBCHIP, pRp3TimerTc->lastSampleNum) |
                   CSL_FMK(FSYNC_RP3TC_SLOT, pRp3TimerTc->lastSlotNum) |
                   CSL_FMK(FSYNC_RP3TC_WRAPADDR, pRp3TimerTc->numChipTerminalCount-1);
   
      hFsync->regs->RP3TC = tempRegVal;
   
      // RP3 timer terminal chip count ram
      for (count=0; count < pRp3TimerTc->numChipTerminalCount; count++)
      {
         tempRegVal = CSL_FMK(FSYNC_RP3TC_TCOUNT, pRp3TimerTc->pLastChipNum[count]);
         hFsync->regs->RP3TC_RAM[count].RP3TC = tempRegVal;
      }

   }
   
   // init Sys timer
   pSysTimerInit = pFsyncSetup->timerInit.pSystemTimerInit;

   if (pSysTimerInit != NULL)
   {
      tempRegVal = CSL_FMK(FSYNC_SYSINIT_CHIP, pSysTimerInit->chipNum) |
                   CSL_FMK(FSYNC_SYSINIT_SLOT, pSysTimerInit->slotNum) |
                   CSL_FMK(FSYNC_SYSINIT_FRAME, pSysTimerInit->frameNum);

      hFsync->regs->SYSINIT = tempRegVal;
   }

   // init Sys timer
   pRp3TimerInit = pFsyncSetup->timerInit.pRp3TimerInit;

   if (pRp3TimerInit != NULL)
   {
      tempRegVal = CSL_FMK(FSYNC_RP3INIT_CHIP, pRp3TimerInit->chipNum) |
                   CSL_FMK(FSYNC_RP3INIT_SLOT, pRp3TimerInit->slotNum) |
                   CSL_FMK(FSYNC_RP3INIT_FRAME, pRp3TimerInit->frameNum);

      hFsync->regs->RP3INIT = tempRegVal;
   }

   
   return(CSL_SOK);
 
}



