/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */
 
/** ===========================================================================
 *   @file  csl_fsyncOpen.c
 *
 *   @brief File for functional layer of CSL API CSL_fsyncOpen () 
 *
 *   @path  $(CSLPATH)\src\fsync
 *
 *   @desc  The CSL_fsyncOpen () function definition.
 */
 

/* =============================================================================
 * Revision History
 * ===============
 *  07-Mar-2006 NS updated the file for DOXYGEN compatibiliy
 *
 * =============================================================================
 */

#include <csl_fsync.h>

 
/** ============================================================================
 *   @n@b CSL_fsyncOpen 
 *
 *   @b Description
 *   @n  This function on successful execution populates the elements of the
 *       link object which is referenced by the first argument of the function 
 *       (*pFsyncObj) and returns a handle to the instance.The device can be 
 *       re-opened anytime after it has been normally closed if so required.
 *       The handle returned by this call is input as an essential argument 
 *       for rest of the APIs described for this module.
 *
 *   @b Arguments
 *   @verbatim            
           pFsyncObj        Pointer to the object that holds reference to the
                            instance of FSYNC requested after the call
                           
           fsyncNum         Instance of FSYNC to which a handle is requested
           
           pFsyncParam      Module specific parameters
 
           pStatus          pointer for returning status of the function call

     @endverbatim
 *
 *   <b> Return Value </b>  
 *      CSL_FsyncHandle
 *        Valid FSYNC instance handle will be returned if status value is
 *        equal to CSL_SOK.
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned in the status variable. If status
 *              returned is
 *   @li            CSL_SOK             Valid FSYNC handle is returned
 *   @li            CSL_ESYS_FAIL       The FSYNC instance is invalid
 *   @li            CSL_ESYS_INVPARAMS  Invalid parameters
 *
 *        2.    FSYNC object structure is populated
 * 
 *   @b  Modifies
 *   @n    1. The status variable
 *
 *         2. FSYNC object structure
 *
 *   @b Example:
 *   @verbatim
          // handle for link 0
          CSL_FsyncHandle handleFsync;
          //  link object for link 0
          CSL_FsyncObj fsyncObj;
          //Fsync module specific parameters
          CSL_FsyncParam  FsyncParam;
          // CSL status
          CSL_Status status;
  
          CSL_fsyncInit(NULL); // Init CSL for FSync module, this step is not required
 
          // Open handle for FSync module 
          handleFsync = CSL_fsyncOpen(&fsyncObj, CSL_FSYNC, NULL, &status);

          if ((handleFsync == NULL) || (status != CSL_SOK)) 
          {
            printf ("\nError opening CSL_FSYNC");
            exit(1);
          }
     @endverbatim
 *
 * ===========================================================================
 */

#pragma CODE_SECTION (CSL_fsyncOpen, ".text:csl_section:fsync");
CSL_FsyncHandle CSL_fsyncOpen (
   /** Pointer to the object that holds reference to the
    *  instance of FSYNC requested after the call
    */
   CSL_FsyncObj              *pFsyncObj,
   /** Instance of FSYNC to which a handle is requested
    */
   CSL_InstNum               fsyncNum,
   /** Module specific parameters;
    */
   CSL_FsyncParam            *pFsyncParam,
   /** This returns the status (success/errors) of the call.
    * Could be 'NULL' if the user does not want status information.
    */
   CSL_Status                *pStatus
   )
{
   CSL_Status          st;
   CSL_FsyncHandle       hFsync;
   CSL_FsyncBaseAddress  baseAddress;

   if (pStatus == NULL) {
       return NULL;
   }
   if (pFsyncObj == NULL) {
       *pStatus = CSL_ESYS_INVPARAMS;
       return NULL;
   }
   
   if (CSL_fsyncGetBaseAddress(fsyncNum, pFsyncParam, &baseAddress) == CSL_SOK) 
   {
      pFsyncObj->regs = baseAddress.regs;
      pFsyncObj->perNum = (CSL_InstNum)fsyncNum;
      hFsync = (CSL_FsyncHandle)pFsyncObj;
   
      st = CSL_SOK;
   } 
   else 
   {
      pFsyncObj->regs = (CSL_FsyncRegsOvly)NULL;
      pFsyncObj->perNum = (CSL_InstNum)-1;
      hFsync = (CSL_FsyncHandle)NULL;
      
      st = CSL_ESYS_FAIL;
   }
   
      *pStatus = st;
   
   return hFsync;
}

