/* ==========================================================================
 * Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 * Use of this software is controlled by the terms and conditions found
 * in the license agreement under which this software has been supplied
 * provided
 * ==========================================================================
*/

/** ========================================================================== 
 * @file csl_gpioGetHwStatus.c
 *
 * @brief File for functional layer of CSL API CSL_gpioGetHwStatus() 
 * 
 * @Path $(CSLPATH)\src\gpio
 * 
 * @desc The CSL_gpioGetHwStatus() function definition & it's associated functions
 * ============================================================================ 
 */
 
/* ===========================================================================
 * Revision History
 * ===============
 * 11-Jun-2004 PGR file created
 * 04-Sep-2004 Nsr Updated the source csl_gpioGetHwStatus.c for the new CSL 
 *                  format.
 * ============================================================================
 */
 
#include <csl_gpio.h>
#include <csl_gpioAux.h>

/** ============================================================================
 *   @n@b CSL_gpioGetHwStatus
 *
 *   @b Description
 *   @n Gets the status of the different operations of GPIO.
 *
 *   @b Arguments
 *   @verbatim
            hGpio           Handle to the GPIO instance

            query           The query to this API of GPIO which indicates the
                            status to be returned.

            response        Placeholder to return the status.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Status info return successful.
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVQUERY   - Invalid Query
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_GpioHandle        hGpio;
        CSL_GpioHwStatusQuery query;
        void                  reponse;

        status = CSL_gpioGetHwStatus(hGpio, query, &response);

     @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_gpioGetHwStatus, ".text:csl_section:gpio");
CSL_Status CSL_gpioGetHwStatus (
    CSL_GpioHandle        hGpio,
    CSL_GpioHwStatusQuery query, 
    void                  *response
)
{
    CSL_Status status = CSL_SOK;
    
    if (hGpio == NULL) {
        return CSL_ESYS_BADHANDLE; 
    }

    if (response == NULL) {
        return CSL_ESYS_INVPARAMS;
    }
    
    switch (query) {            
        case CSL_GPIO_QUERY_BINTEN_STAT:
            CSL_gpioGetBintenStat(hGpio, response);
            break;    
            
        default:
            status = CSL_ESYS_INVQUERY;
            break;
    }
    
    return status;
}
