/* ===========================================================================
 * Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 * Use of this software is controlled by the terms and conditions found in 
 * the license agreement under which this software has been supplied.
 * ==========================================================================
 */

/** ===========================================================================
 * @file csl_gpioHwSetupRaw.c
 *
 * @brief File for functional layer of CSL API CSL_gpioHwSetupRaw() 
 * 
 * @path $(CSLPATH)\src\gpio
 * 
 * @Desc The CSL_gpioHwSetupRaw() function definition & it's associated
 *         functions
 * ============================================================================
 */

/* ============================================================================
 *  Revision History
 *  ===============
 *  07-Sep-2004 Nsr File Created.
 *  28-Jul-2005 PSK updated changes to support only one BANK
 * ============================================================================
 */

#include <csl_gpio.h>

/** ===========================================================================
 *   @n@b CSL_gpioHwSetupRaw
 *
 *   @b Description
 *   @n This function initializes the device registers with the register-values
 *      provided through the Config Data structure.
 *
 *   @b Arguments
 *   @verbatim
            hGpio        Handle to the Gpio instance

            config       Pointer to config structure
            
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Configuration successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Configuration is not
 *                                                properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The registers of the specified GPIO instance will be setup
 *       according to value passed.
 *
 *   @b Modifies
 *   @n Hardware registers of the specified GPIO instance.
 *
 *   @b Example
 *   @verbatim
        CSL_GpioHandle       hGpio;
        CSL_GpioConfig       config = CSL_GPIO_CONFIG_DEFAULTS;
        CSL_Status          status;

        status = CSL_gpioHwSetupRaw (hGpio, &config);

     @endverbatim
 * ===========================================================================
 */ 
#pragma CODE_SECTION (CSL_gpioHwSetupRaw, ".text:csl_section:gpio");
CSL_Status CSL_gpioHwSetupRaw (
    CSL_GpioHandle hGpio,    
    CSL_GpioConfig *config
)
{
    if (hGpio == NULL) {
        return CSL_ESYS_BADHANDLE;
    }

    if (config == NULL) {
        return CSL_ESYS_INVPARAMS;
    }

    hGpio->regs->BINTEN = config->BINTEN;   
    hGpio->regs->DIR = config->DIR;   
    hGpio->regs->OUT_DATA = config->OUT_DATA;   
    hGpio->regs->SET_DATA = config->SET_DATA;   
    hGpio->regs->CLR_DATA = config->CLR_DATA;   
    hGpio->regs->SET_RIS_TRIG = config->SET_RIS_TRIG;   
    hGpio->regs->CLR_RIS_TRIG = config->CLR_RIS_TRIG;   
    hGpio->regs->SET_FAL_TRIG = config->SET_FAL_TRIG;   
    hGpio->regs->CLR_FAL_TRIG = config->CLR_FAL_TRIG; 
        
    return CSL_SOK;
}
