/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** @file csl_pllcGetHwSetup.c
 *
 *  @brief    File for functional layer of CSL API @a CSL_pllcGetHwSetup()
 *
 *  Path: \\(CSLPATH)\src\pllc
 *
 *  Date 26 June, 2004
 *
 *  Author Pratheesh Gangadhar
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  10-Feb-2004 kpn CSL3X Upgradation.
 *  21-Apr-2006 RR   updated for faraday
 * =============================================================================
 */

#include <csl_pllc.h>

/** ============================================================================
 *   @n@b CSL_pllcGetHwSetup
 *
 *   @b Description
 *   @n It retrives the hardware setup parameters of the pllc
 *      specified by the given handle.
 *
 *   @b Arguments
 *   @verbatim
            hPllc        Handle to the pllc

            hwSetup      Pointer to the hardware setup structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Retrieving the hardware setup
 *                                                parameters is successful
 *   @li                    CSL_ESYS_BADHANDLE  - The handle is passed is
 *                                                invalid
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The hardware setup structure is populated with the hardware setup
 *       parameters
 *
 *   @b Modifies
 *   @n hwSetup variable
 *
 *   @b Example
 *   @verbatim
            CSL_PllcHandle   hPllc;
            CSL_PllcHwSetup  hwSetup;
            ...

            status = CSL_pllcGetHwSetup(hPllc, &hwSetup);
     @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_pllcGetHwSetup, ".text:csl_section:pllc");
CSL_Status CSL_pllcGetHwSetup (
    CSL_PllcHandle               hPllc,
    CSL_PllcHwSetup             *hwSetup
)
{
    CSL_Status       status   = CSL_SOK;
    CSL_PllcRegsOvly pllcRegs = hPllc->regs;

    if (hPllc == NULL ) {
        return CSL_ESYS_BADHANDLE;
    }

    if (hwSetup == NULL) {
        return CSL_ESYS_INVPARAMS;
    }

    hwSetup->divEnable = 0;

    hwSetup->pllM = CSL_FEXT (pllcRegs->PLLM, PLLC_PLLM_PLLM);

    if (CSL_FEXT (pllcRegs->OSCDIV1, PLLC_OSCDIV1_OD1EN)) {
        hwSetup->divEnable |= CSL_PLLC_DIVEN_OSCDIV1;
        hwSetup->oscDiv1 = CSL_FEXT (pllcRegs->OSCDIV1, PLLC_OSCDIV1_RATIO);
    }

    if (CSL_FEXT (pllcRegs->PLLDIV1, PLLC_PLLDIV1_D1EN)) {
        hwSetup->divEnable |= CSL_PLLC_DIVEN_PLLDIV0;
        hwSetup->pllDiv0 = CSL_FEXT (pllcRegs->PLLDIV1, PLLC_PLLDIV1_RATIO);
    }
    if (CSL_FEXT (pllcRegs->PLLDIV2, PLLC_PLLDIV2_D2EN)) {
        hwSetup->divEnable |= CSL_PLLC_DIVEN_PLLDIV1;
        hwSetup->pllDiv1 = CSL_FEXT (pllcRegs->PLLDIV2, PLLC_PLLDIV2_RATIO);
    }

    if (CSL_FEXT (pllcRegs->PLLDIV3, PLLC_PLLDIV3_D3EN)) {
        hwSetup->divEnable |= CSL_PLLC_DIVEN_PLLDIV2;
        hwSetup->pllDiv2 = CSL_FEXT (pllcRegs->PLLDIV3, PLLC_PLLDIV3_RATIO);
    }

    if (CSL_FEXT (pllcRegs->PLLDIV4, PLLC_PLLDIV4_D4EN)) {
        hwSetup->divEnable |= CSL_PLLC_DIVEN_PLLDIV3;
        hwSetup->pllDiv3 = CSL_FEXT (pllcRegs->PLLDIV4, PLLC_PLLDIV4_RATIO);
    }

    if (CSL_FEXT (pllcRegs->PLLDIV5, PLLC_PLLDIV5_D5EN)) {
        hwSetup->divEnable |= CSL_PLLC_DIVEN_PLLDIV4;
        hwSetup->pllDiv4 = CSL_FEXT (pllcRegs->PLLDIV5, PLLC_PLLDIV5_RATIO);
    }

    if (CSL_FEXT (pllcRegs->PLLDIV6, PLLC_PLLDIV6_D6EN)) {
        hwSetup->divEnable |= CSL_PLLC_DIVEN_PLLDIV5;
        hwSetup->pllDiv5 = CSL_FEXT (pllcRegs->PLLDIV6, PLLC_PLLDIV6_RATIO);
    }

    if (CSL_FEXT (pllcRegs->PLLDIV11, PLLC_PLLDIV11_D11EN)) {
        hwSetup->divEnable |= CSL_PLLC_DIVEN_PLLDIV10;
        hwSetup->pllDiv10 = CSL_FEXT (pllcRegs->PLLDIV11, PLLC_PLLDIV11_RATIO);
    }

    if (CSL_FEXT (pllcRegs->PLLDIV13, PLLC_PLLDIV13_D13EN)) {
        hwSetup->divEnable |= CSL_PLLC_DIVEN_PLLDIV12;
        hwSetup->pllDiv12 = CSL_FEXT (pllcRegs->PLLDIV13, PLLC_PLLDIV13_RATIO);
    }

	if (CSL_FEXT (pllcRegs->PLLDIV14, PLLC_PLLDIV14_D14EN)) {
        hwSetup->divEnable |= CSL_PLLC_DIVEN_PLLDIV13;
        hwSetup->pllDiv13 = CSL_FEXT (pllcRegs->PLLDIV14, PLLC_PLLDIV14_RATIO);
    }

    
    return status;
}

