/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** @file csl_pllcGetHwStatus.c
 *
 *  @brief   File for functional layer of CSL API @a CSL_pllcGetHwStatus()
 *
 *  Path: \\(CSLPATH)\src\pllc
 *
 *  Date 25 June, 2004
 *
 *  Author Pratheesh Gangadhar
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  10-Feb-2004 kpn  CSL3X Upgradation.
 * =============================================================================
 */

#include <csl_pllc.h>
#include <csl_pllcAux.h>

/** ============================================================================
 *   @n@b CSL_pllcGetHwStatus
 *
 *   @b Description
 *   @n Gets the status of the different operations of PLLC.
 *
 *   @b Arguments
 *   @verbatim
            hPllc         Handle to the PLLC instance

            query         The query to this API of PLLC which indicates the
                          status to be returned.

            response      Placeholder to return the status.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Status info return successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVQUERY   - Invalid query command
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcHwStatusQuery   query;
        void                    reponse;
        ...

        status = CSL_pllcGetHwStatus (hPllc, query, &response);
     @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_pllcGetHwStatus, ".text:csl_section:pllc");
CSL_Status CSL_pllcGetHwStatus (
    CSL_PllcHandle                hPllc,
    CSL_PllcHwStatusQuery         query,
    void                         *response
)
{
    CSL_Status status = CSL_SOK;

    if (hPllc == NULL) {
        return CSL_ESYS_BADHANDLE;
    }

    if (response == NULL) {
        return CSL_ESYS_INVPARAMS;
    }

    switch (query) {
        case CSL_PLLC_QUERY_PID:
            *(Uint32*) response = CSL_pllcGetPID (hPllc);
            break;
        case CSL_PLLC_QUERY_STATUS:
            *(CSL_BitMask32*) response = CSL_pllcGetStatus (hPllc);
            break;
        case CSL_PLLC_QUERY_DIVRATIO_CHANGE:
            *(CSL_BitMask32*) response = CSL_pllcGetDivRatioChange (hPllc);
            break;
        case CSL_PLLC_QUERY_CLKSTAT:
            *(CSL_BitMask32*) response = CSL_pllcGetClkStatus (hPllc);
            break;
        case CSL_PLLC_QUERY_CLKENSTAT:
            *(CSL_BitMask32*) response = CSL_pllcGetClkEnableStatus (hPllc);
            break;
        case CSL_PLLC_QUERY_SYSCLKSTAT:
            *(CSL_BitMask32*) response = CSL_pllcGetSysClkStatus (hPllc);
            break;
        case CSL_PLLC_QUERY_RESETSTAT:
            *(CSL_BitMask32*) response = CSL_pllcGetResetStatus (hPllc);
            break;
        case CSL_PLLC_QUERY_EFUSEERR:
            *(Uint32*) response = CSL_pllcGetFusefarmErrorStatus (hPllc);
            break;
        default:
           status = CSL_ESYS_INVQUERY ;
           break;
    }

    return status;
}

