/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** @file csl_pllcHwControl.c
 *
 *  @brief    File for functional layer of CSL API @a CSL_pllcHwControl()
 *
 *  Path: \\(CSLPATH)\src\pllc
 *
 *  Date 25 June, 2004
 *
 *  Author Pratheesh Gangadhar
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  10-Feb-2004 kpn CSL3X Upgradation.
 * =============================================================================
 */

#include <csl_pllc.h>
#include <csl_pllcAux.h>

/** ============================================================================
 *   @n@b CSL_pllcHwControl
 *
 *   @b Description
 *   @n Takes a command of PLLC with an optional argument & implements it.
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance

            cmd             The command to this API indicates the action to be
                            taken on PLLC.

            arg             An optional argument.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li            CSL_SOK               - Status info return successful.
 *   @li            CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li            CSL_ESYS_INVCMD       - Invalid command
 *   @li            CSL_ESYS_INVPARAMS    - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n The hardware registers of PLLC.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle         hPllc;
        CSL_PllcHwControlCmd   cmd;
        void                   arg;
        ...

        status = CSL_pllcHwControl (hPllc, cmd, &arg);
     @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_pllcHwControl, ".text:csl_section:pllc");
CSL_Status CSL_pllcHwControl (
    CSL_PllcHandle              hPllc,
    CSL_PllcHwControlCmd        cmd,
    void                       *arg
)
{
    CSL_Status status = CSL_SOK;

    if (hPllc == NULL) {
        return CSL_ESYS_BADHANDLE;
    }

    if (arg == NULL) {
        return CSL_ESYS_INVPARAMS;
    }

    switch (cmd) {
        case CSL_PLLC_CMD_PLLCONTROL:
            CSL_pllcCommandCtrl (hPllc, *(CSL_BitMask32*)arg);
            break;
        case CSL_PLLC_CMD_CLOCK_ENABLE:
            CSL_pllcClockEnable (hPllc, *(CSL_BitMask32*)arg);
            break;
        case CSL_PLLC_CMD_CLOCK_DISABLE:
            CSL_pllcClockDisable (hPllc, *(CSL_BitMask32*)arg);
            break;
        case CSL_PLLC_CMD_DEFINE_RESET:
            CSL_pllcDefineReset (hPllc, *(CSL_PllcResetDef*)arg);
            break;
        case CSL_PLLC_CMD_SET_PLLM:
            CSL_pllcMultiplierCtrl (hPllc, *(Uint32*)arg);
            break;
        case CSL_PLLC_CMD_SET_OSCRATIO:
            CSL_pllcSetOscDivRatio (hPllc, *(Uint32*)arg);
            break;
        case CSL_PLLC_CMD_SET_PLLRATIO:
            CSL_pllcSetPLLDivRatio (hPllc, *(CSL_PllcDivRatio*) arg, &status);
            break;
        case CSL_PLLC_CMD_OSCDIV_CONTROL:
            CSL_pllcOscDivCtrl (hPllc, *(CSL_PllcOscDivCtrl*)arg);
            break;
        case CSL_PLLC_CMD_PLLDIV_CONTROL:
            CSL_pllcPLLDivCtrl (hPllc, *(CSL_PllcDivideControl*) arg, &status);
            break;
        case CSL_PLLC_CMD_WAKEUP:
            CSL_pllcCmdWakeup (hPllc, *(CSL_BitMask16*)arg);
            break;
        default:
            status = CSL_ESYS_INVCMD;
            break;
    }

    return status;
}

