/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** @file csl_pllcHwSetup.c
 *
 *  @brief    File for functional layer of CSL API @a CSL_pllcHwSetup()
 *
 *  Path: \\(CSLPATH)\src\pllc
 *
 *  Date 25 June, 2004
 *
 *  Author Pratheesh Gangadhar
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  10-Feb-2004 kpn CSL3X Upgradation.
 *  21-Apr-2006 RR   updated for faraday
 * =============================================================================
 */

#include <csl_pllc.h>

/** ============================================================================
 *   @n@b CSL_pllcHwSetup
 *
 *   @b Description
 *   @n It configures the pllc registers as per the values passed
 *      in the hardware setup structure.
 *
 *   @b Arguments
 *   @verbatim
            hPllc            Handle to the pllc

            hwSetup          Pointer to hardware setup structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Hardware setup successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Hardware structure is not
                                                  properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  PLLC registers are configured according
 *       to the hardware setup parameters
 *
 *   @b Modifies
 *   @n PLLC registers
 *
 *   @b Example
 *   @verbatim
            CSL_PllcHandle   hPllc;
            CSL_PllcObj      pllcObj;
            CSL_PllcHwSetup  hwSetup;
            CSL_status       status;
            ...

            hPllc = CSL_pllcOpen(&pllcObj, CSL_PLLC_0, NULL, &status);
            ...

            status = CSL_pllcHwSetup(hPllc, &hwSetup);
     @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_pllcHwSetup, ".text:csl_section:pllc");
CSL_Status CSL_pllcHwSetup (
    CSL_PllcHandle            hPllc,
    CSL_PllcHwSetup          *hwSetup
)
{
    CSL_PllcRegsOvly pllcRegs = hPllc->regs;
    CSL_Status       status = CSL_SOK;
    volatile unsigned int loopCount;

    if (hPllc == NULL) {
        return CSL_ESYS_BADHANDLE;
    }

    if (hwSetup == NULL) {
        return CSL_ESYS_INVPARAMS;
    }

    CSL_FINS (pllcRegs->PLLCTL, PLLC_PLLCTL_PLLEN,
              CSL_PLLC_PLLCTL_PLLEN_BYPASS);

    /* Wait for 4 cycles of the slowest of the  CLKIN/OSCIN
     * Pratheesh G: 06/29/2004
     * This part requires fine tuing, right now I am not aware of CLKIN/OSCIN
     * values. So introducing some arbitrary delay, if CPU speed is 27Mhz, loop
     * will take around 15 ms
     */

    loopCount = 65535;
    while (loopCount--) {
        asm("   NOP");
    }


    CSL_FINS (pllcRegs->PLLCTL, PLLC_PLLCTL_PLLRST,
              CSL_PLLC_PLLCTL_PLLRST_YES);

    CSL_FINS (pllcRegs->PLLM, PLLC_PLLM_PLLM, hwSetup->pllM);

    if (hwSetup->divEnable & CSL_PLLC_DIVEN_OSCDIV1) {
        CSL_FINS (pllcRegs->OSCDIV1, PLLC_OSCDIV1_OD1EN,
                  CSL_PLLC_OSCDIV1_OD1EN_ENABLE);
        CSL_FINS (pllcRegs->OSCDIV1, PLLC_OSCDIV1_RATIO, hwSetup->oscDiv1);
    }
    else {
        CSL_FINS (pllcRegs->OSCDIV1, PLLC_OSCDIV1_OD1EN,
                  CSL_PLLC_OSCDIV1_OD1EN_DISABLE);
    }

    if (hwSetup->divEnable & CSL_PLLC_DIVEN_PLLDIV0) {
        CSL_FINS (pllcRegs->PLLDIV1, PLLC_PLLDIV1_D1EN,
                  CSL_PLLC_PLLDIV1_D1EN_ENABLE);
    }
    else {
        CSL_FINS (pllcRegs->PLLDIV1, PLLC_PLLDIV1_D1EN,
                  CSL_PLLC_PLLDIV1_D1EN_DISABLE);
    }

    if (hwSetup->divEnable & CSL_PLLC_DIVEN_PLLDIV1) {
        CSL_FINS (pllcRegs->PLLDIV2, PLLC_PLLDIV2_D2EN,
                  CSL_PLLC_PLLDIV2_D2EN_ENABLE);
    }
    else {
        CSL_FINS (pllcRegs->PLLDIV2, PLLC_PLLDIV2_D2EN,
                  CSL_PLLC_PLLDIV2_D2EN_DISABLE);
    }

    if (hwSetup->divEnable & CSL_PLLC_DIVEN_PLLDIV2) {
        CSL_FINS (pllcRegs->PLLDIV3, PLLC_PLLDIV3_D3EN,
                  CSL_PLLC_PLLDIV3_D3EN_ENABLE);
    }
    else {
        CSL_FINS (pllcRegs->PLLDIV3, PLLC_PLLDIV3_D3EN,
                  CSL_PLLC_PLLDIV3_D3EN_DISABLE);
    }

    if (hwSetup->divEnable & CSL_PLLC_DIVEN_PLLDIV3) {
        CSL_FINS (pllcRegs->PLLDIV4, PLLC_PLLDIV4_D4EN,
                  CSL_PLLC_PLLDIV4_D4EN_ENABLE);
    }
    else {
        CSL_FINS (pllcRegs->PLLDIV4, PLLC_PLLDIV4_D4EN,
                  CSL_PLLC_PLLDIV4_D4EN_DISABLE);
    }

    if (hwSetup->divEnable & CSL_PLLC_DIVEN_PLLDIV4) {
        CSL_FINS (pllcRegs->PLLDIV5, PLLC_PLLDIV5_D5EN,
                  CSL_PLLC_PLLDIV5_D5EN_ENABLE);
    }
    else {
        CSL_FINS (pllcRegs->PLLDIV5, PLLC_PLLDIV5_D5EN,
                  CSL_PLLC_PLLDIV5_D5EN_DISABLE);
    }

    if (hwSetup->divEnable & CSL_PLLC_DIVEN_PLLDIV5) {
        CSL_FINS (pllcRegs->PLLDIV6, PLLC_PLLDIV6_D6EN,
                  CSL_PLLC_PLLDIV6_D6EN_ENABLE);
    }
    else {
        CSL_FINS (pllcRegs->PLLDIV6, PLLC_PLLDIV6_D6EN,
                  CSL_PLLC_PLLDIV6_D6EN_DISABLE);
    }

    if (hwSetup->divEnable & CSL_PLLC_DIVEN_PLLDIV10) {
        CSL_FINS (pllcRegs->PLLDIV11, PLLC_PLLDIV11_D11EN,
                  CSL_PLLC_PLLDIV11_D11EN_ENABLE);
        CSL_FINS (pllcRegs->PLLDIV11, PLLC_PLLDIV11_RATIO, hwSetup->pllDiv10);
    }
    else {
        CSL_FINS (pllcRegs->PLLDIV11, PLLC_PLLDIV11_D11EN,
                  CSL_PLLC_PLLDIV11_D11EN_DISABLE);
    }

    if (hwSetup->divEnable & CSL_PLLC_DIVEN_PLLDIV12) {
        CSL_FINS (pllcRegs->PLLDIV13, PLLC_PLLDIV13_D13EN,
                  CSL_PLLC_PLLDIV13_D13EN_ENABLE);
        CSL_FINS (pllcRegs->PLLDIV13, PLLC_PLLDIV13_RATIO, hwSetup->pllDiv12);
    }
    else {
        CSL_FINS (pllcRegs->PLLDIV13, PLLC_PLLDIV13_D13EN,
                  CSL_PLLC_PLLDIV13_D13EN_DISABLE);
    }

	if (hwSetup->divEnable & CSL_PLLC_DIVEN_PLLDIV13) {
        CSL_FINS (pllcRegs->PLLDIV14, PLLC_PLLDIV14_D14EN,
                  CSL_PLLC_PLLDIV14_D14EN_ENABLE);
        CSL_FINS (pllcRegs->PLLDIV14, PLLC_PLLDIV14_RATIO, hwSetup->pllDiv13);
    }
    else {
        CSL_FINS (pllcRegs->PLLDIV14, PLLC_PLLDIV14_D14EN,
                  CSL_PLLC_PLLDIV14_D14EN_DISABLE);
    }


    CSL_FINS (pllcRegs->PLLCMD, PLLC_PLLCMD_GOSET, CSL_PLLC_PLLCMD_GOSET_SET);

    while (CSL_FEXT (pllcRegs->PLLSTAT, PLLC_PLLSTAT_GOSTAT)) {
        /* Poll for divide-ratio change and clock alignment to complete */
    }

    CSL_FINS (pllcRegs->PLLCTL, PLLC_PLLCTL_PLLRST, CSL_PLLC_PLLCTL_PLLRST_NO);

    CSL_FINS (pllcRegs->PLLCTL, PLLC_PLLCTL_PLLEN, CSL_PLLC_PLLCTL_PLLEN_PLL);

    return status;
}

