/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */

/** @file  csl_pllcHwSetupRaw.c
 *
 *  @brief File for functional layer of CSL API @a CSL_pllcHwSetupRaw()
 *
 *  Path: \\(CSLPATH)\src\pllc
 *
 */

/* =============================================================================
 *  Revision History
 *  ===============
 *  10-Feb-2004 kpn  File Created.
 *  21-Apr-2006 RR   updated for faraday 
 * =============================================================================
 */

#include <csl_pllc.h>

/** ============================================================================
 *   @n@b CSL_pllcHwSetupRaw
 *
 *   @b Description
 *   @n This function initializes the device registers with the register-values
 *      provided through the Config Data structure.
 *
 *   @b Arguments
 *   @verbatim
            hpllc        Handle to the PLLC instance

            config       Pointer to config structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li            CSL_SOK               - Configuration successful
 *   @li            CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li            CSL_ESYS_INVPARAMS    - Configuration is not
                                            properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The registers of the specified PLLC instance will be setup
 *       according to input configuration structure values.
 *
 *   @b Modifies
 *   @n Hardware registers of the specified PLLC instance.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle       hPllc;
        CSL_PllcConfig       config = CSL_PLLC_CONFIG_DEFAULTS;
        CSL_Status           status;
        ...

        status = CSL_pllcHwSetupRaw (hPllc, &config);
     @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_pllcHwSetupRaw, ".text:csl_section:pllc");
CSL_Status CSL_pllcHwSetupRaw (
    CSL_PllcHandle               hPllc,
    CSL_PllcConfig              *config
)
{
    volatile Uint32 loopCount;

    if (hPllc == NULL) {
        return CSL_ESYS_BADHANDLE;
    }

    if (config == NULL) {
        return CSL_ESYS_INVPARAMS;
    }

    loopCount = 65535;

    /* Reset Definition register */
    hPllc->regs->RSTDEF   = config->RSTDEF;

    /* PLL Control register */
    hPllc->regs->PLLCTL   = config->PLLCTL;

    /* OBSCLK Select register */
    hPllc->regs->OCSEL   = config->OCSEL;

    /* PLL Secondary Control register */
    hPllc->regs->SECCTL   = config->SECCTL;

    /* PLL Multiplier Control register */
    hPllc->regs->PLLM   = config->PLLM;

    /* PLL Controller Divider 1 register */
    hPllc->regs->PLLDIV1   = config->PLLDIV1;

    /* PLL Controller Divider 2 register */
    hPllc->regs->PLLDIV2   = config->PLLDIV2;

    /* PLL Controller Divider 3 register */
    hPllc->regs->PLLDIV3   = config->PLLDIV3;

    /* PLL Controller Divider 4 register */
    hPllc->regs->OSCDIV1   = config->OSCDIV1;

    /* Bypass Divider register */
    hPllc->regs->BPDIV   = config->BPDIV;

    /* Wakeup register */
    hPllc->regs->WAKEUP   = config->WAKEUP;

    /* PLL Controller Command register */
    hPllc->regs->PLLCMD   = config->PLLCMD;
    
    /* Clock Enable Control register */
    hPllc->regs->CKEN   = config->CKEN;

    /* PLL Controller Divider 4 register */
    hPllc->regs->PLLDIV4   = config->PLLDIV4;

    /* PLL Controller Divider 5 register */
    hPllc->regs->PLLDIV5   = config->PLLDIV5;

    /* PLL Controller Divider 6 register */
    hPllc->regs->PLLDIV6   = config->PLLDIV6;

    /* PLL Controller Divider 11 register */
    hPllc->regs->PLLDIV11   = config->PLLDIV11;

    /* PLL Controller Divider 13 register */
    hPllc->regs->PLLDIV13   = config->PLLDIV13;

	/* PLL Controller Divider 14 register */
    hPllc->regs->PLLDIV14   = config->PLLDIV14;

    /* Fixed time delay - waiting for the PLL to re-lock */
    while (loopCount--) {
        asm("   NOP");
    }

    return CSL_SOK;
}

