/* ============================================================================
 * Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 * Use of this software is controlled by the terms and conditions found in the
 * license agreement under which this software has been supplied.
 * ============================================================================
 */

/** ===========================================================================
 *  @file csl_pwrdwnGetHwStatus.c
 *
 *  @brief File for functional layer of CSL API @a CSL_pwrdwnGetHwStatus()
 *
 *  @path $(CSLPATH)\pwrdwn\src
 *
 *  Description
 *    - The @a CSL_pwrdwnGetHwStatus() function definition & it's associated
 *      functions
 *
 *  Modification 1
 *    - Modified on: 7/16/2004
 *    - Reason: created the sources
 *
 *  @author Ruchika Kharwar.
 * ============================================================================
 */

#include <csl_pwrdwn.h>
#include <csl_pwrdwnAux.h>

/** ===========================================================================
 *   @n@b CSL_pwrdwnGetHwStatus
 *
 *   @b Description
 *   @n This function is used to get the value of various parameters of the
 *      PWRDWN instance. The value returned depends on the query passed.
 *
 *   @b Arguments
 *   @verbatim
            hPwr            Handle to the PWRDWN instance

            query           Query to be performed

            response        Pointer to buffer to return the data requested by
                            the query passed
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK            - Successful completion of the
 *                                               query
 *
 *   @li                    CSL_ESYS_BADHANDLE - Invalid handle
 *
 *   @li                    CSL_ESYS_INVQUERY  - Query command not supported
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_pwrdwnInit(), CSL_pwrdwnOpen() must be opened prior to this call
 *
 *   <b> Post Condition </b>
 *       Data requested by the query is returned through the variable "response"
 *
 *   @b Modifies
 *   @n The input arguement "response" is modified
 *
 *   @b Example:
     @verbatim
            CSL_PwrdwnObj pwrObj;
            CSL_PwrdwnHwSetup pwrSetup;
            CSL_PwrdwnHandle hPwr;
            CSL_PwrdwnPortData pageSleep;

            pageSleep.portNum = 0x0;

            // Init Module
            ...
            if (CSL_pwrdwnInit(NULL) != CSL_SOK)
                exit;
            // Opening a handle for the Module	
	        hPwr = CSL_pwrdwnOpen (&pwrObj, CSL_PWRDWN, NULL, NULL);
	
	        // Setup the arguments for the Setup structure
	        ...
	
	        // Setup
	        CSL_pwrdwnHwSetup(hPwr,&pwrSetup);
    
            // Hw Status Query
            CSL_pwrdwnGetHwStatus(hPwr,CSL_PWRDWN_QUERY_PAGE0_STATUS,&pageSleep);
	
	        // Close handle
            CSL_pwrdwnClose(hPwr);
   @endverbatim
 * ============================================================================
 */
#pragma CODE_SECTION (CSL_pwrdwnGetHwStatus, ".text:csl_section:pwrdwn");
CSL_Status CSL_pwrdwnGetHwStatus (
    /* Pointer to the object that holds reference to the
     * instance of PWRDWN requested after the call
	 */
    CSL_PwrdwnHandle        hPwrdwn,
    
    /* The query to this API which indicates the status
     * to be returned
	 */
    CSL_PwrdwnHwStatusQuery query,
    
    /* Placeholder to return the status. @a void* casted */
    void                    *response
)
{

    CSL_Status status = CSL_SOK;
	
	if (hPwrdwn == NULL) {
		return CSL_ESYS_BADHANDLE;
    }

    if (response == NULL) {
        return CSL_ESYS_INVPARAMS;
    }

	switch (query) {
	    case CSL_PWRDWN_QUERY_PAGE0_STATUS:
		    CSL_pwrdwnGetPage0SleepStatus(hPwrdwn, (CSL_PwrdwnPortData *)response);
		    break;

  	    case CSL_PWRDWN_QUERY_PAGE1_STATUS:
		    CSL_pwrdwnGetPage1SleepStatus(hPwrdwn, (CSL_PwrdwnPortData *)response);
		    break;

        default:
        	status = CSL_ESYS_INVQUERY ;
            break;
	}
	
    return status;
}
