 /* =============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006.
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *  ============================================================================
 */

/** ============================================================================
 *   @file  csl_rac_regsBEAux.c
 *
 *   @path  $(CSLPATH)\src\rac
 *
 *   @desc  This file contains all the API operation prototypes for RAC COMMON CSL
 *			for the BackEnd interface
 *
 */      

/* =============================================================================
 *  Revision History
 *  ================
 *  31-May-2005   RDefosseux     File Created.
 *
 * =============================================================================
 */
 
#include <csl_rac_regsBEAux.h>

/* =============================================================================
 * Functions linked to BETI management
 * =============================================================================
 */

/**
 *  @func    CSL_RAC_BETI_enable
 *
 *  @brief   This function enables the RAC Back End Transfer Interface (BETI)
 *			 by writing 0x1 into the RAC_BETI_CTRL register. 
 *
 *  @arg     None
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_BETI_CTRL register.
 *
 *  @enter   None          
 *
 *  @leave   The Back End Transfer Interface is then enabled.
 *
 *  @see
 *
 */
void CSL_RAC_BETI_enable ()
{
	CSL_FWRIT (hRacRegs->BETI_ENA, RAC_CFG_BETI_ENA_ENABLE, ENABLE);
}

/**
 *  @func    CSL_RAC_BETI_disable
 *
 *  @brief   This function disables the RAC Back End Transfer Interface (BETI)
 *			 by writing 0x0 into the RAC_BETI_CTRL register. 
 *
 *  @arg     None
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_BETI_CTRL register.
 *
 *  @enter   None          
 *
 *  @leave   The Back End Transfer Interface is then disabled.
 *
 *  @see
 *
 */
void CSL_RAC_BETI_disable ()
{
	CSL_FWRIT (hRacRegs->BETI_ENA, RAC_CFG_BETI_ENA_ENABLE, DISABLE);
}

/**
 *  @func    CSL_RAC_BETI_getStatus
 *
 *  @brief   This function returns the BETI status indicating whether the
 *			 BETI is enabled or not.
 *			 register.
 *
 *  @arg     None
 *
 *  @return  CSL_RAC_BETI_statusBit_Disable
 *               When the BETI is disabled.
 * 
 *  @return  CSL_RAC_BETI_statusBit_Enable
 *               When the BETI is enabled.
 *
 *  @desc    This function reads the RAC_BETI_STAT register. Reading this
 *			 register is relevant especially when disabling the BETI. When
 *			 disabling, a transfer can be in process and the BETI has to
 *			 complete it before being disabled.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
CSL_RAC_BETI_statusBit CSL_RAC_BETI_getStatus ()
{
	return (CSL_RAC_BETI_statusBit) \
		CSL_FEXT (hRacRegs->BETI_STAT, RAC_CFG_BETI_STAT_STATUS);
}

/**
 *  @func    CSL_RAC_BETI_setWatchDog
 *
 *  @brief   This function configures the minimal activity watch dog
 *           in the BETI. 
 *
 *  @arg     nbCyclesToReload
 *           Half-word with the value to set in the register.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_BETI_MIN_WCFG register with
 *           the nbCyclesToReload value.
 *
 *  @enter   None          
 *
 *  @leave   The Back End watch-dogs are now configured.
 *
 *  @see
 *
 */
void CSL_RAC_BETI_setWatchDog (
	Uint16 nbCyclesToReload
)
{
	CSL_FWRI (hRacRegs->BETI_MIN_WCFG, RAC_CFG_BETI_MIN_WCFG_BE_WD_CFG, nbCyclesToReload);
}

/**
 *  @func    CSL_RAC_BETI_getWatchDogStatus
 *
 *  @brief   This function reads the content of the minimal activity watch dog
 *           decounter for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP minimal activity watch-dog decounter to read.
 *
 *  @return  Uint16
 *				Current value of the decounter.
 *
 *  @desc    This operation reads the corresponding RAC_BETI_MIN_WST[gccpId]
 *           register and extracts the decounter field.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_BETI_getWatchDogStatus (
	Uint8 gccpId
)
{
	return CSL_FEXT (hRacRegs->BETI_MIN_WST[gccpId], RAC_CFG_BETI_MIN_WST_BE_WD_STAT);
}

/**
 *  @func    CSL_RAC_BETI_getWatchDogInterruptStatus
 *
 *  @brief   This function reads the content of the minimal activity watch dog
 *           interrupt status for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP minimal activity watch-dog status to read.
 *
 *  @return  CSL_RAC_BETI_wdInterruptStatus_NoInt
 *               No interrupt has been generated.
 * 
 *  @return  CSL_RAC_BETI_wdInterruptStatus_Int
 *               An interrupt has been generated and forwarded to the BEII.
 * 
 *
 *  @desc    This operation reads the corresponding RAC_BETI_MIN_WST[gccpId] register and extracts the
 *			 interrupt status field.
 *
 *  @enter   None          
 *
 *  @leave   When the user reads the RAC_BETI_MIN_WST[gccpId] register, the
 *           interrupt status field is cleared by the H/W.
 *
 *  @see
 *
 */
CSL_RAC_BETI_wdInterruptStatus CSL_RAC_BETI_getWatchDogInterruptStatus (
	Uint8 gccpId
)
{
	return (CSL_RAC_BETI_wdInterruptStatus) \
		CSL_FEXT (hRacRegs->BETI_MIN_WST[gccpId], RAC_CFG_BETI_MIN_WST_INT_STAT);
}

/**
 *  @func    CSL_RAC_BETI_getOdbtStatus
 *
 *  @brief   This function returns if the corresponding Output Descriptor Buffer
 *           has received at least one new output block descriptor.
 *
 *  @arg     odbtEntryId
 *           Which buffer status to get.
 *
 *  @return  CSL_RAC_BETI_odbtStatusBit_noNewObd
 *               When no new Output Block Descriptor has been generated for this
 *				 output buffer.
 * 
 *  @return  CSL_RAC_BETI_odbtStatusBit_newObd
 *               A new Output Block Descriptor has been written to the buffer.
 * 
 *  @desc    This operation reads the corresponding RAC_BETI_NEW_OBD register and extracts
 *           status bit for the given buffer.
 *
 *  @enter   None          
 *
 *  @leave   ???
 *
 *  @see
 *
 */
CSL_RAC_BETI_odbtStatusBit CSL_RAC_BETI_getOdbtStatus (
	Uint8 odbtEntryId
)
{
	CSL_RAC_BETI_odbtStatusBit tmp;
	tmp = (CSL_RAC_BETI_odbtStatusBit) \
		(( hRacRegs->BETI_NEW_OD >> odbtEntryId) & 0x1);
	if (tmp == CSL_RAC_BETI_odbtStatusBit_newObd)
		hRacRegs->BETI_NEW_OD = (1 << odbtEntryId);
	return (CSL_RAC_BETI_odbtStatusBit) tmp;
}

/* =============================================================================
 * Functions linked to BEII management
 * =============================================================================
 */

/**
 *  @func    CSL_RAC_BEII_enable
 *
 *  @brief   This function enables the RAC Back End Interrupt Interface (BEII)
 *			 by writing 0x1 into the RAC_BEII_CTRL register. 
 *
 *  @arg     None
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_BEII_CTRL register.
 *
 *  @enter   None          
 *
 *  @leave   The Back End Interrupt Interface is then enabled.
 *
 *  @see
 *
 */
void CSL_RAC_BEII_enable ()
{
	CSL_FWRIT (hRacRegs->BEII_ENA, RAC_CFG_BEII_ENA_ENABLE, ENABLE);
}

/**
 *  @func    CSL_RAC_BEII_disable
 *
 *  @brief   This function disables the RAC Back End Interrupt Interface (BEII)
 *			 by writing 0x0 into the RAC_BEII_CTRL register. 
 *
 *  @arg     None
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_BEII_CTRL register.
 *
 *  @enter   None          
 *
 *  @leave   The Back End Interrupt Interface is then disabled.
 *
 *  @see
 *
 */
void CSL_RAC_BEII_disable ()
{
	CSL_FWRIT (hRacRegs->BEII_ENA, RAC_CFG_BEII_ENA_ENABLE, DISABLE);
}


/**
 *  @func    CSL_RAC_BEII_setMasterMask
 *
 *  @brief   This function configures the BEII Master Mask for the specified
 *           cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     cpuIntContrlEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_BEII_MINT_ENA register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setMasterMask (
	Uint8                  cpuId,
	CSL_RAC_BEII_interrupt cpuIntContrlEnable
)
{
	switch(cpuId)
	{
		case 0:
			CSL_FINS(hRacRegs->BEII_MINT_ENA, RAC_CFG_BEII_MINT_ENA_DSP0_INT_ENA, cpuIntContrlEnable);
			break;
		case 1:
			CSL_FINS(hRacRegs->BEII_MINT_ENA, RAC_CFG_BEII_MINT_ENA_DSP1_INT_ENA, cpuIntContrlEnable);
			break;
		default:
			CSL_FINS(hRacRegs->BEII_MINT_ENA, RAC_CFG_BEII_MINT_ENA_DSP2_INT_ENA, cpuIntContrlEnable);
			break;
	}
}

/**
 *  @func    CSL_RAC_BEII_getInterruptStatus
 *
 *  @brief   This function reads the BEII Interrupt Status for the specified
 *           cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @return  CSL_RAC_BEII_interruptStatus
 *
 *  @desc    This operation modifies the RAC_BEII_INT_STATx register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_getInterruptStatus (
	Uint8                          cpuId,
	CSL_RAC_BEII_interruptStatus * cpuIntStatus
)
{
	Uint32 word;
/*	switch(cpuId)
	{
		case 0:
            word = hBeRegs.BEII_INT_STAT0;
			break;
		case 1:
            word = hBeRegs.BEII_INT_STAT1;
			break;
		default:
            word = hBeRegs.BEII_INT_STAT2;
			break;
	}
*/
    word = hRacRegs->BEII_INT_STAT[cpuId];

	cpuIntStatus->gccp1WdStat        = CSL_FEXT(word,RAC_CFG_BEII_INT_STAT_GCCP1_WD_STAT);
	cpuIntStatus->gccp0WdStat        = CSL_FEXT(word,RAC_CFG_BEII_INT_STAT_GCCP0_WD_STAT);
	cpuIntStatus->feWdStat           = CSL_FEXT(word,RAC_CFG_BEII_INT_STAT_FE_WD_STAT);
	cpuIntStatus->odbtRdPtrStat      = CSL_FEXT(word,RAC_CFG_BEII_INT_STAT_ODBT_RD_PTR_STAT);
	cpuIntStatus->obbtRdPtrStat      = CSL_FEXT(word,RAC_CFG_BEII_INT_STAT_OBBT_RD_PTR_STAT);
	cpuIntStatus->eotStat            = CSL_FEXT(word,RAC_CFG_BEII_INT_STAT_EOT_STAT);
	cpuIntStatus->gccp1SeqStat       = CSL_FEXT(word,RAC_CFG_BEII_INT_STAT_GCCP1_SEQ_STAT);
	cpuIntStatus->gccp1FifoOverStat  = CSL_FEXT(word,RAC_CFG_BEII_INT_STAT_GCCP1_FIFO_OVER_STAT);
	cpuIntStatus->gccp1CycOverStat   = CSL_FEXT(word,RAC_CFG_BEII_INT_STAT_GCCP1_CYC_OVER_STAT);
	cpuIntStatus->gccp0SeqStat       = CSL_FEXT(word,RAC_CFG_BEII_INT_STAT_GCCP0_SEQ_STAT);
	cpuIntStatus->gccp0FifoOverStat  = CSL_FEXT(word,RAC_CFG_BEII_INT_STAT_GCCP0_FIFO_OVER_STAT);
	cpuIntStatus->gccp0CycOverStat   = CSL_FEXT(word,RAC_CFG_BEII_INT_STAT_GCCP0_CYC_OVER_STAT);
}


/**
 *  @func    CSL_RAC_BEII_setCycleOverflowMask
 *
 *  @brief   This function configures the Cycle Overflow Mask for the specified
 *           cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     gccpId
 *           Which Gccp source to enable.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setCycleOverflowMask (
	Uint8                  cpuId,
	Uint8                  gccpId,
	CSL_RAC_BEII_interrupt interruptEnable
)
{
	if (gccpId == 0)
		CSL_FINS(hRacRegs->BEII_INT_MSK[cpuId],
				 RAC_CFG_BEII_INT_MSK_GCCP0_CYC_OVER_ENA,
				 interruptEnable);
	else
		CSL_FINS(hRacRegs->BEII_INT_MSK[cpuId],
				 RAC_CFG_BEII_INT_MSK_GCCP1_CYC_OVER_ENA,
				 interruptEnable);
}

/**
 *  @func    CSL_RAC_BEII_setFifoOverflowMask
 *
 *  @brief   This function configures the FIFO Overflow Mask for the specified
 *           cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     gccpId
 *           Which Gccp source to enable.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setFifoOverflowMask (
	Uint8                  cpuId,
	Uint8                  gccpId,
	CSL_RAC_BEII_interrupt interruptEnable
)
{
	if (gccpId == 0)
		CSL_FINS(hRacRegs->BEII_INT_MSK[cpuId],
				 RAC_CFG_BEII_INT_MSK_GCCP0_FIFO_OVER_ENA,
				 interruptEnable);
	else
		CSL_FINS(hRacRegs->BEII_INT_MSK[cpuId],
				 RAC_CFG_BEII_INT_MSK_GCCP1_FIFO_OVER_ENA,
				 interruptEnable);
}

/**
 *  @func    CSL_RAC_BEII_setSequencerIdleMask
 *
 *  @brief   This function configures the Sequencer Idle Mask for the specified
 *           cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     gccpId
 *           Which Gccp source to enable.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setSequencerIdleMask (
	Uint8                  cpuId,
	Uint8                  gccpId,
	CSL_RAC_BEII_interrupt interruptEnable
)
{
	if (gccpId == 0)
		CSL_FINS(hRacRegs->BEII_INT_MSK[cpuId],
				 RAC_CFG_BEII_INT_MSK_GCCP0_SEQ_ENA,
				 interruptEnable);
	else
		CSL_FINS(hRacRegs->BEII_INT_MSK[cpuId],
				 RAC_CFG_BEII_INT_MSK_GCCP1_SEQ_ENA,
				 interruptEnable);
}

/**
 *  @func    CSL_RAC_BEII_setBeWatchDogMask
 *
 *  @brief   This function configures the Back End Min Activity Watch Dog Mask
 *           for the specified cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     gccpId
 *           Which Gccp source to enable.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setBeWatchDogMask (
	Uint8                  cpuId,
	Uint8                  gccpId,
	CSL_RAC_BEII_interrupt interruptEnable
)
{
	if (gccpId == 0)
		CSL_FINS(hRacRegs->BEII_INT_MSK[cpuId],
				 RAC_CFG_BEII_INT_MSK_GCCP0_WD_ENA,
				 interruptEnable);
	else
		CSL_FINS(hRacRegs->BEII_INT_MSK[cpuId],
				 RAC_CFG_BEII_INT_MSK_GCCP1_WD_ENA,
				 interruptEnable);
}

/**
 *  @func    CSL_RAC_BEII_setBetiEotMask
 *
 *  @brief   This function configures the Back End End-Of-Transfer Mask
 *           for the specified cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setBetiEotMask (
	Uint8                  cpuId,
	CSL_RAC_BEII_interrupt interruptEnable
)
{
	CSL_FINS(hRacRegs->BEII_INT_MSK[cpuId],
			 RAC_CFG_BEII_INT_MSK_EOT_ENA,
			 interruptEnable);
}

/**
 *  @func    CSL_RAC_BEII_setBetiObbtRdCrossingMask
 *
 *  @brief   This function configures the Back End OBBT Read Pointer Crossing Mask
 *           for the specified cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setBetiObbtRdCrossingMask (
	Uint8                  cpuId,
	CSL_RAC_BEII_interrupt interruptEnable
)
{
	CSL_FINS(hRacRegs->BEII_INT_MSK[cpuId],
			 RAC_CFG_BEII_INT_MSK_OBBT_RD_PTR_ENA,
			 interruptEnable);
}

/**
 *  @func    CSL_RAC_BEII_setBetiOdbtRdCrossingMask
 *
 *  @brief   This function configures the Back End ODBT Read Pointer Crossing Mask
 *           for the specified cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setBetiOdbtRdCrossingMask (
	Uint8                  cpuId,
	CSL_RAC_BEII_interrupt interruptEnable
)
{
	CSL_FINS(hRacRegs->BEII_INT_MSK[cpuId],
			 RAC_CFG_BEII_INT_MSK_ODBT_RD_PTR_ENA,
			 interruptEnable);
}


/**
 *  @func    CSL_RAC_BEII_setFeWatchDogMask
 *
 *  @brief   This function configures the Front End Watch Dog Mask
 *           for the specified cpu interrupt controller. 
 *
 *  @arg     cpuId
 *           Which CPU interrupt controller to configure.
 *
 *  @arg     interruptEnable
 *			 CSL_RAC_BEII_interrupt_Disable: no interrupt is sent to the specified interrupt controller.
 *			 CSL_RAC_BEII_interrupt_Enable: interrupts are forwarded to the specified interrupt controller.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the corresponding RAC_BEII_INT_MSK[cpuId] register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_BEII_setFeWatchDogMask (
	Uint8                  cpuId,
	CSL_RAC_BEII_interrupt interruptEnable
)
{
	CSL_FINS(hRacRegs->BEII_INT_MSK[cpuId],
			 RAC_CFG_BEII_INT_MSK_FE_WD_ENA,
			 interruptEnable);
}

/**
 *  @func    CSL_RAC_BETI_getObbtRdCrossingStatus
 *
 *  @brief   This function reads the content of the OBBT read pointer crossing register. 
 *
 *  @arg     None
 *
 *  @return  
 *				All the OBBT Status.
 *
 *  @desc    This operation reads the RAC_BETI_OBBT_STAT.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_BETI_getObbtRdCrossingStatus ()
{
	return hRacRegs->BETI_OBBT_STAT;
}

/**
 *  @func    CSL_RAC_BETI_getOdbtRdCrossingStatus
 *
 *  @brief   This function reads the content of the ODBT read pointer crossing register. 
 *
 *  @arg     None
 *
 *  @return  
 *				All the ODBT Status.
 *
 *  @desc    This operation reads the RAC_BETI_ODBT_STAT.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_BETI_getOdbtRdCrossingStatus ()
{
	return hRacRegs->BETI_ODBT_STAT;
}

/**
 *  @func    CSL_RAC_BETI_getEotInterruptStatus
 *
 *  @brief   This function reads the content of the End-Of-Transfer Interrupt
 *			 Status register to know which Output Descriptor Buffer has a new
 *			 descriptor.
 *
 *  @arg     cpuId
 *           Which CPU interrupt register to read.
 *
 *  @return  
 *				All the EOT Interrupt registers.
 *
 *  @desc    This operation reads the corresponding RAC_BETI_EOT_STAT# register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_BETI_getEotInterruptStatus (
	Uint8 cpuId
)
{
	return hRacRegs->BETI_EOTI_STAT[cpuId];
}

/* =============================================================================
 * Functions linked to VBUS Stats registers
 * =============================================================================
 */

/**
 *  @func    CSL_RAC_Stats_getCfgTotalAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of data phases used by the Configuration interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_CFG_TOT_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_Stats_getCfgTotalAccess ()
{
	return hRacRegs->CFG_TOT;
}

/**
 *  @func    CSL_RAC_Stats_getCfgWriteAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of write data phases used by the Configuration interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_CFG_WRITE_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */

Uint32 CSL_RAC_Stats_getCfgWriteAccess ()
{
	return hRacRegs->CFG_WRIT;
}

/**
 *  @func    CSL_RAC_Stats_getCfgReadAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of read data phases used by the Configuration interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_CFG_READ_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_Stats_getCfgReadAccess ()
{
	return hRacRegs->CFG_READ;
}

/**
 *  @func    CSL_RAC_Stats_getSlaveTotalAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of data phases used by the DMA slave interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_SLV_TOT_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_Stats_getSlaveTotalAccess ()
{
	return hRacRegs->SLV_TOT;
}

/**
 *  @func    CSL_RAC_Stats_getSlaveWriteAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of write data phases used by the DMA Slave interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_SLV_WRITE_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */

Uint32 CSL_RAC_Stats_getSlaveWriteAccess ()
{
	return hRacRegs->SLV_WRIT;
}

/**
 *  @func    CSL_RAC_Stats_getSlaveReadAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of read data phases used by the DMA Slave interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_SLV_READ_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_Stats_getSlaveReadAccess ()
{
	return hRacRegs->SLV_READ;
}

/**
 *  @func    CSL_RAC_Stats_getMasterLowPrioAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of data phases used by the low priority master
 *           interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_MST_LP_TOT_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_Stats_getMasterLowPrioAccess ()
{
	return hRacRegs->MST_TOT_LP;
}

/**
 *  @func    CSL_RAC_Stats_getMasterHighPrioAccess
 *
 *  @brief   This function reads the content of the statistics register for 
 *           the total number of data phases used by the high priority master
 *           interface.
 *
 *  @arg     None
 *
 *  @return  
 *				The value of the register.
 *
 *  @desc    This operation reads the corresponding RAC_MST_HP_TOT_REG register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_Stats_getMasterHighPrioAccess ()
{
	return hRacRegs->MST_TOT_HP;
}


