/* =============================================================================
 *   @copy    Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *  ============================================================================
 */

/** ============================================================================
 *   @file  csl_rac_regsFEAux.c
 *
 *   @path  $(CSLPATH)\src\rac
 *
 *   @desc  This file contains all the API operation prototypes for RAC COMMON CSL
 *			for the FrontEnd interface
 *
 */

/* =============================================================================
 *  Revision History
 *  ================
 *  31-May-2005   RDefosseux     File Created.
 *
 * =============================================================================
 */
 
#include <csl_rac_regsFEAux.h>

/**
 *  @func    CSL_RAC_FE_enable
 *
 *  @brief   This function enables the RAC Front End Interface (FEI)
 *			 by writing 0x1 into the RAC_FE_ENA register. 
 *
 *  @arg     None
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_FE_ENA register.
 *
 *  @enter   None          
 *
 *  @leave   The Front End Interface is then enabled.
 *
 *  @see
 *
 */
void CSL_RAC_FE_enable ()
{
	CSL_FWRIT (hRacRegs->FE_ENA, RAC_CFG_FE_ENA_ENABLE, ENABLE);
}

/**
 *  @func    CSL_RAC_FE_disable
 *
 *  @brief   This function disables the RAC Front End Interface (FEI)
 *			 by writing 0x0 into the RAC_FE_ENA register. 
 *
 *  @arg     None
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_FE_ENA register.
 *
 *  @enter   None          
 *
 *  @leave   The Front End Interface is then disabled.
 *
 *  @see
 *
 */
void CSL_RAC_FE_disable ()
{
	CSL_FWRIT (hRacRegs->FE_ENA, RAC_CFG_FE_ENA_ENABLE, DISABLE);
}

/**
 *  @func    CSL_RAC_FE_getStatus
 *
 *  @brief   This function returns the Front End Transfer State by reading
 *			 the Front End status register register.
 *
 *  @arg     None
 *
 *  @return  CSL_RAC_FE_transferState_WaitingTimestamp
 *               When the Front End is waiting for the timestamp write.
 * 
 *  @return  CSL_RAC_FE_transferState_ReceivingSamples
 *               When the Front End receives samples.
 * 
 *  @return  CSL_RAC_FE_transferState_ReadyToStart
 *               When the Front End has received all samples and is ready to start the GCCPs.
 * 
 *  @return  CSL_RAC_FE_transferState_StartGccpIteration
 *               When the Front End enables the GCCPs.
 *
 *  @desc    This function reads the RAC_FE_STAT register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
CSL_RAC_FE_transferState CSL_RAC_FE_getStatus ()
{
	return (CSL_RAC_FE_transferState) \
		CSL_FEXT (hRacRegs->FE_STAT, RAC_CFG_FE_STAT_TRANSFER_FSM);
}

/**
 *  @func    CSL_RAC_FE_getGccpStatus
 *
 *  @brief   This function returns the corresponding GCCP Status by reading
 *			 the Front End status register register.
 *
 *  @arg     gccpId
 *           Byte to select the GCCP status.
 *
 *  @return  CSL_RAC_FE_gccpStatus_Idle
 *               When the GCCP is idle. The Data memories are accessible for debug.
 * 
 *  @return  CSL_RAC_FE_gccpStatus_Busy
 *               When the GCCP is busy. The Data memories are not accessible for debug.
 * 
 *  @desc    This function reads the RAC_FE_STAT register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
CSL_RAC_FE_gccpStatus CSL_RAC_FE_getGccpStatus (
	Uint8 gccpId
)
{
	if (gccpId == 0)
		return (CSL_RAC_FE_gccpStatus) \
			CSL_FEXT (hRacRegs->FE_STAT, RAC_CFG_FE_STAT_GCCP0_STAT);
	else
		return (CSL_RAC_FE_gccpStatus) \
			CSL_FEXT (hRacRegs->FE_STAT, RAC_CFG_FE_STAT_GCCP1_STAT);
}

/**
 *  @func    CSL_RAC_FE_setInputBufferDepth
 *
 *  @brief   This function configures the Input Buffer Depth register. 
 *
 *  @arg     ibDepth
 *           Byte with the value to set in the register.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_FE_IB_DEPTH register with
 *           the ibDepth value.
 *
 *  @enter   None          
 *
 *  @leave   The Front End is now able to compute the read time for the GCCPs.
 *
 *  @see
 *
 */
void CSL_RAC_FE_setInputBufferDepth (
	Uint8 ibDepth
)
{
	CSL_FWRI (hRacRegs->FE_ID_DEPTH, RAC_CFG_FE_ID_DEPTH_IB_DEPTH, ibDepth);
}

/**
 *  @func    CSL_RAC_FE_setMaxCyclesPerIteration
 *
 *  @brief   This function configures the Front End Max Cycles watch-dog. 
 *
 *  @arg     maxCyclesNb
 *           Half-word with the value to set in the register.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_FE_MAX register with
 *           the maxCyclesNb value.
 *
 *  @enter   None          
 *
 *  @leave   The Front End watch-dog is configured.
 *
 *  @see
 *
 */
void CSL_RAC_FE_setMaxCyclesPerIteration (
	Uint16 maxCyclesNb
)
{
	CSL_FWRI (hRacRegs->FE_WCFG, RAC_CFG_FE_WCFG_FE_WD_CFG, maxCyclesNb);
}

/**
 *  @func    CSL_RAC_FE_getWatchDogStatus
 *
 *  @brief   This function reads the content of the watch dog decounter.
 *
 *  @arg     None
 *
 *  @return  Uint16
 *				Current value of the decounter.
 *
 *  @desc    This operation reads the RAC_FE_WST register and extracts the
 *			 decounter field.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint16 CSL_RAC_FE_getWatchDogStatus ()
{
	return CSL_FEXT (hRacRegs->FE_WST, RAC_CFG_FE_WST_FE_WD_STATUS);
}

/**
 *  @func    CSL_RAC_FE_getWatchDogInterruptStatus
 *
 *  @brief   This function reads the content of the watch dog
 *           interrupt status. 
 *
 *  @arg     None
 *
 *  @return  CSL_RAC_FE_wdInterruptStatus_NoInt
 *               No interrupt has been generated.
 * 
 *  @return   CSL_RAC_FE_wdInterruptStatus_Int
 *               An interrupt has been generated and forwarded to the BEII.
 * 
 *
 *  @desc    This operation reads the RAC_FE_WINT register and extracts the
 *			 interrupt status field.
 *
 *  @enter   None          
 *
 *  @leave   When the user reads the RAC_FE_WINT register, the H/W clears the
 *           interrupt status bit. If the user reads once again the register,
 *           'no-interrupt' status will be return.
 *
 *  @see
 *
 */
CSL_RAC_FE_wdInterruptStatus CSL_RAC_FE_getWatchDogInterruptStatus ()
{
	return (CSL_RAC_FE_wdInterruptStatus) \
		CSL_FEXT (hRacRegs->FE_INT, RAC_CFG_FE_INT_INT_STATUS);
}


/**
 *  @func    CSL_RAC_FE_setTimestamp
 *
 *  @brief   This function configures the Front End timestamp register. 
 *
 *  @arg     timestamp
 *           structure with timestamp parameters.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_FE_TIME register with
 *           the timestamp parameters. This function can be used for a
 *           non-periodic usage of the RAC.
 *
 *  @enter   None          
 *
 *  @leave   The Front End is ready to receive antenna samples.
 *
 *  @see
 *
 */
void CSL_RAC_FE_setTimestamp (
	CSL_RAC_FE_Timestamp_req * timestamp
)
{
	hRacData->FE_TIME = 
		CSL_FMK(RAC_DATA_FE_TIME_CHIP,timestamp->chipId) |
		CSL_FMK(RAC_DATA_FE_TIME_SLOT,timestamp->slotId) |
		CSL_FMK(RAC_DATA_FE_TIME_FRAME,timestamp->frameId);
}

/**
 *  @func    CSL_RAC_FE_getTimestamp
 *
 *  @brief   This function reads the Front End timestamp register. 
 *
 *  @arg     timestamp
 *           structure with timestamp parameters.
 *
 *  @return  None
 *
 *  @desc    This operation reads the content of the Front-End timestamp
 *           register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_FE_getTimestamp (
	CSL_RAC_FE_Timestamp_req * timestamp
)
{
	Uint32 lWord = hRacData->FE_TIME;
	timestamp->chipId  = CSL_FEXT (lWord, RAC_DATA_FE_TIME_CHIP);
	timestamp->slotId  = CSL_FEXT (lWord, RAC_DATA_FE_TIME_SLOT);
	timestamp->frameId = CSL_FEXT (lWord, RAC_DATA_FE_TIME_FRAME);
}

