/* =============================================================================
 *   @copy    Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *  ============================================================================
 */                

/** ============================================================================
 *   @file  csl_rac_regsGCCPAux.c
 *
 *   @path  $(CSLPATH)\src\rac
 *
 *   @desc  This file contains all the API operation prototypes for RAC COMMON CSL
 *			for the GCCP
 *
 */                     
 
/* =============================================================================
 *  Revision History
 *  ================
 *  31-May-2005   RDefosseux     File Created.
 *
 * =============================================================================
 */
 
#include <csl_rac_regsGCCPAux.h>

/* =============================================================================
 * Functions linked to GCCP management
 * =============================================================================
 */

/**
 *  @func    CSL_RAC_GCCP_enable
 *
 *  @brief   This function enables the corresponding Generic Correlation Co-Processor
 *			 (GCCP) by writing 0x1 into the RAC_GCCP_SEQ_ENA register. 
 *
 *  @arg     gccpId
 *           Which GCCP to activate.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_GCCP_SEQ_ENA register.
 *
 *  @enter   None          
 *
 *  @leave   The GCCP Sequencer is then enabled.
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_enable (
	Uint8 gccpId
)
{
	CSL_FWRIT (hGccpRegs[gccpId].GCCP_SEQ_ENA,
			   GCCP_CFG_GCCP_SEQ_ENA_ENABLE,
			   ENABLE);
}

/**
 *  @func    CSL_RAC_GCCP_disable
 *
 *  @brief   This function disables the corresponding Generic Correlation Co-Processor
 *			 (GCCP) by writing 0x0 into the RAC_GCCP_SEQ_ENA register. 
 *
 *  @arg     gccpId
 *           Which GCCP to disactivate.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_GCCP_SEQ_ENA register.
 *
 *  @enter   None          
 *
 *  @leave   The GCCP Sequencer is then disabled.
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_disable (
	Uint8 gccpId
)
{
	CSL_FWRIT (hGccpRegs[gccpId].GCCP_SEQ_ENA,
			   GCCP_CFG_GCCP_SEQ_ENA_ENABLE,
			   DISABLE);
}

/**
 *  @func    CSL_RAC_GCCP_getActiveCycles
 *
 *  @brief   This function returns the number of active cycles in the previous
 *           iteration.
 *
 *  @arg     gccpId
 *           Which GCCP to get status from.
 *
 *  @return  The number of active cycles
 *
 *  @desc    This function reads the RAC_GCCP_SEQ_ACT register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint16 CSL_RAC_GCCP_getActiveCycles (
	Uint8 gccpId
)
{
	return CSL_FEXT (hGccpRegs[gccpId].GCCP_SEQ_ACT,
					 GCCP_CFG_GCCP_SEQ_ACT_ACT_COUNT);
}

/**
 *  @func    CSL_RAC_GCCP_getSequencerCycles
 *
 *  @brief   This function returns the number of sequencer-used cycles in the previous
 *           iteration.
 *
 *  @arg     gccpId
 *           Which GCCP to get status from.
 *
 *  @return  The number of sequencer cycles
 *
 *  @desc    This function reads the RAC_GCCP_SEQ_CYC register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint16 CSL_RAC_GCCP_getSequencerCycles (
	Uint8 gccpId
)
{
	return CSL_FEXT (hGccpRegs[gccpId].GCCP_SEQ_CYC,
					 GCCP_CFG_GCCP_SEQ_CYC_SEQ_COUNT);
}

/**
 *  @func    CSL_RAC_GCCP_resetHighPriorityQueue
 *
 *  @brief   This function resets the High Priority Output Queue of the 
 *			 corresponding GCCP by writing into the RAC_GCCP_FIFO_RESET register. 
 *
 *  @arg     gccpId
 *           Which GCCP FIFO to reset.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_GCCP_FIFO_RESET register.
 *
 *  @enter   None          
 *
 *  @leave   The GCCP High Priority Output Queue is reset.
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_resetHighPriorityQueue (
	Uint8 gccpId
)
{
	CSL_FWRIT (hGccpRegs[gccpId].GCCP_FIFO_RESET,
			   GCCP_CFG_GCCP_FIFO_RESET_HIGH_RESET,
			   RESET);
}

/**
 *  @func    CSL_RAC_GCCP_resetLowPriorityQueue
 *
 *  @brief   This function resets the Low Priority Output Queue of the 
 *			 corresponding GCCP by writing into the RAC_GCCP_FIFO_RESET register. 
 *
 *  @arg     gccpId
 *           Which GCCP FIFO to reset.
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_GCCP_FIFO_RESET register.
 *
 *  @enter   None          
 *
 *  @leave   The GCCP Low Priority Output Queue is reset.
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_resetLowPriorityQueue (
	Uint8 gccpId
)
{
	CSL_FWRIT (hGccpRegs[gccpId].GCCP_FIFO_RESET,
			   GCCP_CFG_GCCP_FIFO_RESET_LOW_RESET,
			   RESET);
}

/**
 *  @func    CSL_RAC_GCCP_getReadTime
 *
 *  @brief   This function reads the GCCP[0] read timestamp register. 
 *
 *  @arg     timestamp
 *           structure with timestamp parameters.
 *
 *  @return  None
 *
 *  @desc    This operation reads the content of the GCCP[0] read timestamp
 *           register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_getReadTime (
	CSL_RAC_FE_Timestamp_req * timestamp
)
{
	Uint32 lWord = hGccpRegs[0].GCCP_RD_SYS_TIME;
	timestamp->chipId  = CSL_FEXT (lWord, RAC_DATA_FE_TIME_CHIP);
	timestamp->slotId  = CSL_FEXT (lWord, RAC_DATA_FE_TIME_SLOT);
	timestamp->frameId = CSL_FEXT (lWord, RAC_DATA_FE_TIME_FRAME);
}

/**
 *  @func    CSL_RAC_GCCP_getLowPrioDataLevel
 *
 *  @brief   This function returns the Low Priority Data component level 
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  The current value of the level
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_LDQ_CURR_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getLowPrioDataLevel (
	Uint8 gccpId
)
{
	return CSL_FEXT (hGccpRegs[gccpId].LDQ_CURR_LVL,
					 GCCP_CFG_LDQ_CURR_LVL_CURR_LVL);
}

/**
 *  @func    CSL_RAC_GCCP_getLowPrioDataWatermark
 *
 *  @brief   This function reads the Low Priority Data component water-mark
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  The current value of the water-mark
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_LDQ_WTMK_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   The register is reset with the current level value.
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getLowPrioDataWatermark (
	Uint8 gccpId
)
{
	return CSL_FEXT (hGccpRegs[gccpId].LDQ_WTMK_LVL,
					 GCCP_CFG_LDQ_WTMK_LVL_WATERMARK);
}

/**
 *  @func    CSL_RAC_GCCP_getHighPrioDataLevel
 *
 *  @brief   This function returns the High Priority Data component level 
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  The current value of the level
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_HDQ_CURR_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getHighPrioDataLevel (
	Uint8 gccpId
)
{
	return CSL_FEXT (hGccpRegs[gccpId].HDQ_CURR_LVL,
					 GCCP_CFG_HDQ_CURR_LVL_CURR_LVL);
}

/**
 *  @func    CSL_RAC_GCCP_getHighPrioDataWatermark
 *
 *  @brief   This function reads the High Priority Data component water-mark
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  Uint16
 *				Current value of the water-mark
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_HDQ_WTMK_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   The register is reset with the current level value.
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getHighPrioDataWatermark (
	Uint8 gccpId
)
{
	return CSL_FEXT (hGccpRegs[gccpId].HDQ_WTMK_LVL,
					 GCCP_CFG_HDQ_WTMK_LVL_WATERMARK);
}

/**
 *  @func    CSL_RAC_GCCP_getLowPrioControlLevel
 *
 *  @brief   This function returns the Low Priority Control component level 
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  The current value of the level
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_LCQ_CURR_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getLowPrioControlLevel (
	Uint8 gccpId
)
{
	return CSL_FEXT (hGccpRegs[gccpId].LCQ_CURR_LVL,
					 GCCP_CFG_LCQ_CURR_LVL_CURR_LVL);
}

/**
 *  @func    CSL_RAC_GCCP_getLowPrioControlWatermark
 *
 *  @brief   This function reads the Low Priority Control component water-mark
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  The current value of the water-mark
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_LCQ_WTMK_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   The register is reset with the current level value.
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getLowPrioControlWatermark (
	Uint8 gccpId
)
{
	return CSL_FEXT (hGccpRegs[gccpId].LCQ_WTMK_LVL,
					 GCCP_CFG_LCQ_WTMK_LVL_WATERMARK);
}

/**
 *  @func    CSL_RAC_GCCP_getHighPrioControlLevel
 *
 *  @brief   This function returns the High Priority Control component level 
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  The current value of the level
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_HCQ_CURR_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getHighPrioControlLevel (
	Uint8 gccpId
)
{
	return CSL_FEXT (hGccpRegs[gccpId].HCQ_CURR_LVL,
					 GCCP_CFG_HCQ_CURR_LVL_CURR_LVL);
}

/**
 *  @func    CSL_RAC_GCCP_getHighPrioControlWatermark
 *
 *  @brief   This function reads the High Priority Control component water-mark
 *           of the output queue for a given GCCP ID. 
 *
 *  @arg     gccpId
 *           Which GCCP status to read.
 *
 *  @return  The current value of the water-mark
 *
 *  @desc    This operation reads the corresponding RAC_GCCP_HCQ_WTMK_LVL register.
 *
 *  @enter   None          
 *
 *  @leave   The register is reset with the current level value.
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getHighPrioControlWatermark (
	Uint8 gccpId
)
{
	return CSL_FEXT (hGccpRegs[gccpId].HCQ_WTMK_LVL,
					 GCCP_CFG_HCQ_WTMK_LVL_WATERMARK);
}

/**
 *  @func    CSL_RAC_GCCP_setIctEntry
 *
 *  @brief   This function configures the Interpolation Coefficients Table
 *           by writing into one of the RAC_ICT registers. 
 *
 *  @arg     gccpId
 *           Which GCCP ICT to configure.
 *
 *  @arg     entryId
 *           Which GCCP ICT to configure. It ranges from 0 to 4.
 *
 *  @arg     coeff0
 *           Value to set into the coefficient #0 field. It should range from -64 to 64
 *
 *  @arg     coeff1
 *           Value to set into the coefficient #1 field. It should range from -64 to 64
 *
 *  @arg     coeff2
 *           Value to set into the coefficient #2 field. It should range from -64 to 64
 *
 *  @arg     coeff3
 *           Value to set into the coefficient #3 field. It should range from -64 to 64
 *
 *  @return  None
 *
 *  @desc    This operation modifies the RAC_GCCP_ICT register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_setIctEntry (
	Uint8 gccpId,
	Uint8 entryId,
	Int8  coeff0,
	Int8  coeff1,
	Int8  coeff2,
	Int8  coeff3
)
{
	hGccpRegs[gccpId].INT_COEF[entryId] =
		CSL_FMK(GCCP_CFG_INT_COEF_COEFF0,coeff0) |
		CSL_FMK(GCCP_CFG_INT_COEF_COEFF1,coeff1) |
		CSL_FMK(GCCP_CFG_INT_COEF_COEFF2,coeff2) |
		CSL_FMK(GCCP_CFG_INT_COEF_COEFF3,coeff3) ;
}

/**
 *  @func    CSL_RAC_GCCP_setCgtEntry
 *
 *  @brief   This function configures the Code Generation Table by writing
 *           into one of the RAC_GCCP_CGT registers. 
 *
 *  @arg     gccpId
 *           Which GCCP CGT to configure.
 *
 *  @arg     entryId
 *           Which entry to configure.
 *
 *  @arg     partY
 *           Value to write into the register.
 *
 *  @return  None
 *
 *  @desc    This operation modifies one of the RAC_GCCP_CGT register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_setCgtEntry (
	Uint8  gccpId,
	Uint8  entryId,
	Uint32 partY
)
{
	CSL_FWRI (hGccpRegs[gccpId].CGT[entryId],
			  GCCP_CFG_CGT_Y_PART,
			  partY);
}

/**
 *  @func    CSL_RAC_GCCP_getCgtEntry
 *
 *  @brief   This function reads one of the RAC_GCCP_CGT registers. 
 *
 *  @arg     gccpId
 *           Which GCCP CGT to configure.
 *
 *  @arg     entryId
 *           Which entry to configure.
 *
 *  @return  None
 *
 *  @desc    This operation returns the Y part for the given GCCP and entry ID.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getCgtEntry (
	Uint8  gccpId,
	Uint8  entryId
)
{
	return CSL_FEXT (hGccpRegs[gccpId].CGT[entryId],
					 GCCP_CFG_CGT_Y_PART);
}

/**
 *  @func    CSL_RAC_GCCP_setPrtEntry
 *
 *  @brief   This function configures the Pilot Rotation Table by writing
 *           into one of the RAC_GCCP_PRT registers. 
 *
 *  @arg     gccpId
 *           Which GCCP PRT to configure.
 *
 *  @arg     entryId
 *           Which entry to configure.
 *
 *  @arg     wordId
 *           Which slot to configure.
 *
 *  @arg     format
 *           Value to insert into the register.
 *
 *  @return  None
 *
 *  @desc    This operation modifies one of the RAC_GCCP_PRT register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_setPrtEntry (
	Uint8  gccpId,
	Uint8  entryId,
	Uint8  wordId,
	Uint32 format
)
{
	switch(wordId)
	{
	case 0:
		hGccpRegs[gccpId].PRT[entryId].PRT_W0 = format;
		break;
	case 1:
		hGccpRegs[gccpId].PRT[entryId].PRT_W1 = format;
		break;
	case 2:
		hGccpRegs[gccpId].PRT[entryId].PRT_W2 = format;
		break;
	case 3:
		hGccpRegs[gccpId].PRT[entryId].PRT_W3 = format;
		break;
	default:
		hGccpRegs[gccpId].PRT[entryId].PRT_W4 = format;
		break;
	}
}

/**
 *  @func    CSL_RAC_GCCP_getPrtEntry
 *
 *  @brief   This function reads one of the RAC_GCCP_PRT registers. 
 *
 *  @arg     gccpId
 *           Which GCCP PRT to configure.
 *
 *  @arg     entryId
 *           Which entry to configure.
 *
 *  @arg     wordId
 *           Which slot to configure.
 *
 *  @return  None
 *
 *  @desc    This operation returns the slot format for the given slot ID.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
Uint32 CSL_RAC_GCCP_getPrtEntry (
	Uint8  gccpId,
	Uint8  entryId,
	Uint8  wordId
)
{
	switch(wordId)
	{
	case 0:
		return hGccpRegs[gccpId].PRT[entryId].PRT_W0;
	case 1:
		return hGccpRegs[gccpId].PRT[entryId].PRT_W1;
	case 2:
		return hGccpRegs[gccpId].PRT[entryId].PRT_W2;
	case 3:
		return hGccpRegs[gccpId].PRT[entryId].PRT_W3;
	default:
		return hGccpRegs[gccpId].PRT[entryId].PRT_W4;
	}
}



/**
 *  @func    CSL_RAC_GCCP_getFifoOverflowStatus
 *
 *  @brief   This function returns the last active TRM address
 *
 *  @arg     gccpId
 *           Which GCCP to get status from.
 *
 *  @return  CSL_RAC_GCCP_fifoOverflowStatus
 *           The last active TRM address before a FIFO overflow
 *
 *  @desc    This function reads the RAC_GCCP_FIFO_OVER register.
 *
 *  @enter   None          
 *
 *  @leave   None
 *
 *  @see
 *
 */
void CSL_RAC_GCCP_getFifoOverflowStatus (
	Uint8 gccpId,
	CSL_RAC_GCCP_fifoOverflowStatus * fifoOverflowStatus
)
{

	Uint32                    word;

	word = hGccpRegs[gccpId].GCCP_FIFO_OVER;
	fifoOverflowStatus->iteNb       = CSL_FEXT(word, GCCP_CFG_GCCP_FIFO_OVER_ITE_NB);
	fifoOverflowStatus->fifoId      = CSL_FEXT(word, GCCP_CFG_GCCP_FIFO_OVER_FIFO_ID);
	fifoOverflowStatus->cycOverFlag = CSL_FEXT(word, GCCP_CFG_GCCP_FIFO_OVER_CYC_OVER_FLG);
	fifoOverflowStatus->pageIdx     = CSL_FEXT(word, GCCP_CFG_GCCP_FIFO_OVER_PAGE_IDX);
	fifoOverflowStatus->taskIdx     = CSL_FEXT(word, GCCP_CFG_GCCP_FIFO_OVER_TASK_IDX);

}

