 /*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */
 
/** ===========================================================================
 *  @file csl_semGetHwStatus.c
 *
 *  @brief File for functional layer of CSL API @a CSL_semGetHwStatus()
 *
 *  @path $(CSLPATH)\sem\src
 * 
 *  Description
 *    - The @a CSL_semGetHwStatus() function definition & it's associated
 *      functions
 */ 

/* =============================================================================
 * Revision History
 * ===============
 *  10-Apr-2006 NS updated the file for DOXYGEN compatibiliy
 *
 * =============================================================================
 */
 
/* HW SEM CSL API  */
#include <csl_sem.h>

/** ============================================================================
 *   @n@b CSL_semGetHwStatus
 *
 *   @b Description
 *   @n The Status Query Command call queries the semaphore module
 *
 *   @b Arguments
 *   @verbatim
            hSem            Handle to the SEM instance
 
            query           Query to be performed 
 
            response        Pointer to buffer to return the data requested by
                            the query passed
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK            - Successful completion of the
 *                                               query
 *                          CSL_ESYS_BADHANDLE - Invalid Handle
 *                          CSL_ESYS_INVQUERY  - Invalid query
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *       Data requested by query is returned through the variable "response"
 *
 *   @b Modifies
 *   @n The input argument "response" is modified.      
 *
 *   @b Example
 *   @verbatim
      // handle for SEM
      CSL_SemHandle handleSem;
      //  SEM object 
      CSL_SemObj            *pSemObj;
      // CSL status
      CSL_Status status;
      void semVal;
      CSL_SemParam       pSemParam;
      //Number of SEM resources    
      pSemParam.flags = 2

  
      CSL_semInit(NULL); // Init CSL for SEM module, this step is not required
      ...
      // Open handle for SEM module 
      handleSem = CSL_semOpen (pSemObj, CSL_SEM, &pSemParam, &status);
      ...
      status = CSL_semGetHwStatus(hSem, CSL_SEM_QUERY_READ_PEND, &semVal);
 
     @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_semGetHwStatus, ".text:csl_section:sem");  
CSL_Status  CSL_semGetHwStatus(
   CSL_SemHandle           hSem,
   CSL_SemHwStatusQuery    query,
   void                    *response
)
{
	CSL_Status status = CSL_SOK;
	volatile unsigned int *ptmpUint0,*ptmpUint,tmpUint,sem;
	sem = hSem->semNum;

	if (hSem == NULL) {
        return CSL_ESYS_BADHANDLE;
    }

	switch(query){
		case CSL_SEM_QUERY_SCRATCH_READ:
			ptmpUint = (unsigned int *)response;
			ptmpUint0 = &(hSem->regs->SCRATCH); //HW_SEM_SCRATCH_REG_ADDR;
			*ptmpUint = *ptmpUint0;
			break;

		case CSL_SEM_QUERY_RESET_RUN_READ:
			ptmpUint = (unsigned int *)response;
			ptmpUint0 = &(hSem->regs->RST_RUN); //HW_SEM_RST_RUN_REG_ADDR;
			*ptmpUint = *ptmpUint0;
			break;

    	case CSL_SEM_QUERY_REVISION:   //(Arg: Uint32*)           
			ptmpUint = (unsigned int *)response;
            ptmpUint0 = &(hSem->regs->PID);
			*ptmpUint = *ptmpUint0; //CSL_SEM_VERSION;
			break;
  
    	case CSL_SEM_QUERY_ERROR:      //(Arg: CSL_SemFaultStatus*)   
			{
			CSL_SemFaultStatus *pStatus;
    	   	pStatus = (CSL_SemFaultStatus *)response;
			ptmpUint0 = &(hSem->regs->ERR); //HW_SEM_ERR_REG_ADDR;
			tmpUint = *ptmpUint0;
			pStatus->errorMask = CSL_FEXT(tmpUint, SEM_ERR_ERR);//perrReg->errCode;
			pStatus->semNum = CSL_FEXT(tmpUint, SEM_ERR_SEMNUM);//perrReg->semNum;
			pStatus->faultID = CSL_FEXT(tmpUint, SEM_ERR_FAULTID); //perrReg->MstId;  
			}
			break;

        case CSL_SEM_QUERY_FLAGS:      //(Arg: CSL_SemFlags*)
            {
            CSL_SemFlagClear_Arg *pFlags;
            pFlags = (CSL_SemFlagClear_Arg *)response;
			sem	= pFlags->masterId;
            ptmpUint0 = hSem->regs->FLAG_FLAG_CLEAR; //HW_SEM_FLAG_READ_ADDR;
			pFlags->BitMask32 = ptmpUint0[sem];
            }
			break;

    	case CSL_SEM_QUERY_STATUS:     //(Arg: CSL_SemVal*)
			{
			CSL_SemVal *pSemVal;
            pSemVal = (CSL_SemVal *)response;
			ptmpUint0 = hSem->regs->QSEM; //HW_SEM_QUERY_ADDR;
			tmpUint = ptmpUint0[sem];
			pSemVal->semFree = (CSL_SemFlag)CSL_FEXT(tmpUint, SEM_QSEM_FREE);
			pSemVal->semOwner =(CSL_SemOwnerId)CSL_FEXT(tmpUint, SEM_QSEM_OWNER);
			pSemVal->semNum = sem;
			}
			break;
	    case CSL_SEM_QUERY_READ_PEND:  //(Arg: CSL_SemVal*)  
   			{
   			CSL_SemVal *pSemVal;
			pSemVal = (CSL_SemVal *)response; 
			ptmpUint0 = hSem->regs->SEM; //HW_SEM_DIRECT_ACCESS;
			pSemVal->semFree =(CSL_SemFlag)ptmpUint0[sem];
			}
			break;
    	case CSL_SEM_QUERY_READ_POST:   //(Arg: CSL_SemVal*)
   			{
   			CSL_SemVal *pSemVal;
			pSemVal = (CSL_SemVal *)response; 
			ptmpUint0 = hSem->regs->ISEM;//HW_SEM_COMBINED_ACCESS;
			pSemVal->semFree = (CSL_SemFlag)ptmpUint0[sem];
			}    	 
			break;
		default:
		    status = CSL_ESYS_INVQUERY;
			break;
	}

	return status;
}


