 /*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */
 
/** ===========================================================================
 *  @file csl_semHwControl.c
 *
 *  @brief File for functional layer of CSL API @a CSL_semHwControl()
 *
 *  @path $(CSLPATH)\sem\src
 * 
 *  Description
 *    - The @a CSL_semHwControl() function definition & it's associated
 *      functions
 */ 

/* =============================================================================
 * Revision History
 * ===============
 *  10-Apr-2006 NS updated the file for DOXYGEN compatibiliy
 *
 * =============================================================================
 */
 
#include <csl_sem.h>

/** ============================================================================
 *   @n@b  CSL_semHwControl
 *
 *   @b Description
 *   @n  The Control Command call writes a command value to the semaphore
 *
 *   @b Arguments
 *   @verbatim
            hSem            Handle to the SEM instance

            ctrlCmd         The command to this API indicates the action to be
                            taken on SEM.

            arg             An optional argument.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Status info return successful.
 *                               
 *   <b> Pre Condition </b>
 *   @n Both CSL_semInit() (optional) and CSL_semOpen()must be called
 *   successfully in that order before CSL_semHwControl() can be called.
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n The hardware registers of SEM.
 *
 *   @b Example
 *   @verbatim
      // handle for SEM
      CSL_SemHandle handleSem;
      //  SEM object 
      CSL_SemObj            *pSemObj;
      // CSL status
      CSL_Status status;
      CSL_SemParam       pSemParam;
      //Number of SEM resources    
      pSemParam.flags = 2

  
      CSL_semInit(NULL); // Init CSL for SEM module, this step is not required
      ...
      // Open handle for SEM module 
      handleSem = CSL_semOpen (pSemObj, CSL_SEM, &pSemParam, &status);
      ...
      status = CSL_semHwControlCmd(handleSem, CSL_SEM_CMD_WRITE_POST_DIRECT, NULL);
 
     @endverbatim
 * =============================================================================
 */
#pragma CODE_SECTION (CSL_semHwControl, ".text:csl_section:sem");          
CSL_Status  CSL_semHwControl(
   CSL_SemHandle            hSem,
   CSL_SemHwControlCmd      ctrlCmd,
   void                     *arg
)
{
	CSL_Status status = CSL_SOK;
	volatile unsigned int *pSemReg, *pUInt;
	int sem;
	unsigned int tmpUInt;
	volatile CSL_SemFlagClear_Arg	*pArg;
	volatile CSL_SemErrSet_Arg	*pArgErrSet;
	volatile CSL_SemFlagSet_Arg	*pArgFlagSet;
	volatile CSL_SemEOISet_Arg	*pArgEOISet;

    if (hSem == NULL) {
        return CSL_ESYS_BADHANDLE;
    }

	sem = hSem->semNum;
	pSemReg = hSem->regs->SEM;           //HW_SEM_DIRECT_ACCESS;
	pArg = (CSL_SemFlagClear_Arg *)arg;
	pUInt = (unsigned int*)arg;

	switch(ctrlCmd){
        case CSL_SEM_CMD_SCRATCH_WRITE:  
    		pSemReg = &hSem->regs->SCRATCH; //HW_SEM_SCRATCH_REG_ADDR;  
			*pSemReg = *pUInt;
			break;

    	case CSL_SEM_CMD_RESET_RUN_WRITE:  
    		pSemReg = &(hSem->regs->RST_RUN); //HW_SEM_RST_RUN_REG_ADDR;  
			*pSemReg = *pUInt;
			break;

    	case CSL_SEM_CMD_EOI_WRITE:  
            pSemReg = &(hSem->regs->EOI); //HW_SEM_EOI_REG_ADDR;  
			pArgEOISet = (CSL_SemEOISet_Arg *)arg;
			tmpUInt	= pArgEOISet->masterId;
            if(pArgEOISet->enableErrorInt){
				tmpUInt = HW_SEM_EOI_REG_ERR_BIT_POS;
			}
			*pSemReg = tmpUInt;
			break;

    	case CSL_SEM_CMD_FREE_DIRECT:                  /* Release semaphore            */
			pSemReg[sem] = HW_SEM_RELEASE;
			break;

    	case CSL_SEM_CMD_WRITE_POST_DIRECT:            /* Perform Indirect acquisition */
			pSemReg[sem] = HW_SEM_REQUEST;
			break;
        
        case CSL_SEM_CMD_FREE_INDIRECT:                  /* Release semaphore            */
            pSemReg = hSem->regs->ISEM;
            pSemReg[sem] = HW_SEM_RELEASE;
            break;

        case CSL_SEM_CMD_WRITE_POST_INDIRECT:            /* Perform Indirect acquisition */
            pSemReg = hSem->regs->ISEM;
            pSemReg[sem] = HW_SEM_REQUEST;
            break;
    
        case CSL_SEM_CMD_FREE_QSEM:                  /* Release semaphore            */
            pSemReg = hSem->regs->QSEM;
            pSemReg[sem] = HW_SEM_RELEASE;
            break;

        case CSL_SEM_CMD_WRITE_POST_QSEM:            /* Perform Indirect acquisition */
            pSemReg = hSem->regs->QSEM;
            pSemReg[sem] = HW_SEM_REQUEST;
            break;


		case CSL_SEM_CLEAR_FLAGS:
			pSemReg = hSem->regs->FLAG_FLAG_CLEAR; //HW_SEM_FLAG_CLEAR_ADDR;
			sem	= pArg->masterId;
			pSemReg[sem] = pArg->BitMask32;  
			break;

		case CSL_SEM_CMD_FLAG_SET:
			pSemReg = hSem->regs->FLAG_SET; //HW_SEM_FLAG_SET_ADDR;
			pArgFlagSet = (CSL_SemFlagSet_Arg *)arg;
			sem	= pArgFlagSet->masterId;
			pSemReg[sem] = (1 << pArgFlagSet->semId);  
			break;

	  	case CSL_SEM_CLEAR_ERR:                  /* Clear error condition        */			
  			pSemReg = &(hSem->regs->ERR_CLEAR); //HW_SEM_ERR_CLEAR_ADDR;
			*pSemReg = 1;
			break;

		case CSL_SEM_CMD_ERR_SET:
			pSemReg = &(hSem->regs->ERR_SET); //HW_SEM_ERR_REG_SET_ADDR;
			pArgErrSet = (CSL_SemErrSet_Arg *)arg;
			*pSemReg = CSL_FMK(SEM_ERR_SET_ERR, pArgErrSet->errCode) |
                       CSL_FMK(SEM_ERR_SET_SEMNUM, pArgErrSet->semNum) |
                       CSL_FMK(SEM_ERR_SET_FAULTID, pArgErrSet->mstId);  
			break;

        default:
            status = CSL_ESYS_INVCMD;
			break;
	}

	return status;
}
