/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */
 
/** ===========================================================================
 *  @file csl_semOpen.c
 *
 *  @brief File for functional layer of CSL API @a CSL_semOpen()
 *
 *  @path $(CSLPATH)\sem\src
 * 
 *  Description
 *    - The @a CSL_semOpen() function definition & it's associated functions
 *
 */ 

/* =============================================================================
 * Revision History
 * ===============
 *  10-Apr-2006 NS updated the file for DOXYGEN compatibiliy
 *
 * =============================================================================
 */
 
#include <csl_sem.h>

/** ============================================================================
 *   @n@b csl_semOpen
 *
 *   @b Description
 *   @n The open call sets up the data structures for the particular instance of
 *      SEM device. The device can be re-opened anytime after it has been normally
 *      closed if so required. The handle returned by this call is input as an
 *      essential argument for rest of the APIs described for this module. 
 *   @b Arguments
 *   @verbatim            
           pSemObj          Pointer to the object that holds reference to the
                            instance of SEM requested after the call
                           
           instNum          Instance of SEM to which a handle is requested
           
           pSemParam        Pointer to module specific parameters,(Number of SEM resource)
            
           pStatus          pointer for returning status of the function call

     @endverbatim
 *
 *   <b> Return Value </b>  
 *      CSL_SemHandle
 *        Valid SEM instance handle will be returned if status value is
 *        equal to CSL_SOK.
 *
 *   <b> Pre Condition </b>
 *   @n  SEM must be successfully initialized via CSL_semInit() before calling
 *       this function. Memory for the CSL_SemObj must be allocated outside
 *       this call. This object must be retained while using this peripheral. 
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned in the status variable. If status
 *              returned is
 *   @li            CSL_SOK             Valid SEM handle is returned
 *   @li            CSL_ESYS_FAIL       The SEM instance is invalid
 *                  CSL_ESYS_INVPARAMS  Invalid Parameters
 *
 *        2.    SEM object structure is populated
 * 
 *   @b  Modifies
 *   @n    1. The status variable
 *
 *         2. SEM object structure
 *
 *   @b Example:
 *   @verbatim
          // handle for SEM
          CSL_SemHandle handleSem;
          //  SEM object 
          CSL_SemObj            *pSemObj;
          // CSL status
          CSL_Status status;
          CSL_SemParam       pSemParam;
          //Number of SEM resource
          pSemParam.flags = 2
  
          CSL_semInit(NULL); // Init CSL for SEM module, this step is not required
 
          // Open handle for SEM module 
          handleSem = CSL_semOpen (pSemObj, CSL_SEM, &pSemParam, &status);

          if ((handleSem == NULL) || (status != CSL_SOK)) 
          {
            printf ("\nError opening CSL_SEM");
            exit(1);
          }
     @endverbatim
 *
 * ===========================================================================
 */
#pragma CODE_SECTION (CSL_semOpen, ".text:csl_section:sem");
CSL_SemHandle CSL_semOpen (
   /** Pointer to the object that holds reference to the
    *  instance of SEM requested after the call
    */
   CSL_SemObj         *pSemObj,
   /** Instance of AIF to which a handle is requested
    */
   CSL_InstNum        instNum,
   /** Module specific parameters;
    */
   CSL_SemParam       *pSemParam,
   /** This returns the status (success/errors) of the call.
    * Could be 'NULL' if the user does not want status information.
    */
   CSL_Status         *pStatus
)
{
    CSL_Status          st;
    CSL_SemHandle       hSem;
    CSL_SemBaseAddress  baseAddress;

    if (pStatus == NULL) {
        return NULL;
    }
    if (pSemObj == NULL) {
        *pStatus = CSL_ESYS_INVPARAMS;
        return NULL;
    }
    if (pSemParam->flags <= 1 || pSemParam->flags > 31) {
        *pStatus = CSL_ESYS_INVPARAMS;
        return NULL;
    }

    if (CSL_semGetBaseAddress(instNum, pSemParam, &baseAddress) == CSL_SOK) {
        pSemObj->regs = baseAddress.regs;
        pSemObj->instNum = (CSL_InstNum)instNum;
        pSemObj->semNum = pSemParam->flags;
        hSem = (CSL_SemHandle)pSemObj;
        st = CSL_SOK;
    } else {
        pSemObj->regs = (CSL_SemRegsOvly)NULL;
        pSemObj->instNum = (CSL_InstNum)-1;
        pSemObj->semNum = -1;
        hSem = (CSL_SemHandle)NULL;

        st = CSL_ESYS_FAIL;
    }

    *pStatus = st;
    
    return hSem;
}                   
