/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2011
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */
/** ============================================================================
 example for HyperLink configuration on Keystone DSP
 * =============================================================================
 *  Revision History
 *  ===============
 *  Nov 4, 2011 Brighton  file created
 * =============================================================================*/
#include <c6x.h>
#include "Keystone_HyperLink_init.h"

#define DSP_TYPE_TCI6614

CSL_VusrRegs * hyperLinkRegs = (CSL_VusrRegs *)CSL_MCM_CONFIG_REGS;
SerdesRegs * hyperLinkSerdesRegs;

/*soft shutdown and reset HyperLink*/
void Keystone_HyperLink_soft_reset()
{
	int i;
	
	/*disable all portal or remote register operation
	This bit should be set before iloop or reset bits are changed.*/
	hyperLinkRegs->CTL |= CSL_VUSR_CTL_SERIAL_STOP_MASK;

	/*Wait until no Remote Pending Request*/
	while(hyperLinkRegs->STS&CSL_VUSR_STS_RPEND_MASK);

	/*Reset*/
	hyperLinkRegs->CTL |= CSL_VUSR_CTL_RESET_MASK;

	/*wait for a while*/
	for(i=0;i<100;i++)asm(" NOP 5 ");

#ifdef DSP_TYPE_TCI6614
	//disable HyperLink through PSC
	Keystone_disable_PSC_module(CSL_PSC_PD_ALWAYSON, CSL_PSC_LPSC_VUSR);
#else
	Keystone_disable_PSC_module(CSL_PSC_PD_HYPERBRIDGE, CSL_PSC_LPSC_HYPERBRIDGE);
	Keystone_disable_PSC_Power_Domain(CSL_PSC_PD_HYPERBRIDGE);
#endif
}

void Keystone_HyperLink_Addr_Map(
	HyperLink_Address_Map * addr_map)
{
	int i;
	
	hyperLinkRegs->TX_SEL_CTL = 
		(addr_map->tx_addr_mask<<CSL_VUSR_TX_SEL_CTL_TXIGNMSK_SHIFT)
		|(addr_map->tx_priv_id_ovl<<CSL_VUSR_TX_SEL_CTL_TXPRIVIDOVL_SHIFT);
		
	hyperLinkRegs->RX_SEL_CTL = 
		(addr_map->rx_seg_sel<<CSL_VUSR_RX_SEL_CTL_RXSEGSEL_SHIFT)
		|(addr_map->rx_priv_id_sel<<CSL_VUSR_RX_SEL_CTL_RXPRIVIDSEL_SHIFT);
		
	for(i= 0; i< 16; i++)
	{
		hyperLinkRegs->RX_PRIV_IDX= i;
		hyperLinkRegs->RX_PRIV_VAL= addr_map->rx_priv_id_map[i];
	}

	for(i= 0; i< 64; i++)
	{
		hyperLinkRegs->RX_SEG_IDX= i;
		hyperLinkRegs->RX_SEG_VAL= 
			addr_map->rx_addr_segs[i].Seg_Base_Addr
			|addr_map->rx_addr_segs[i].Seg_Length;
	}

}

void Keystone_HyperLink_Interrupt_init(
	HyperLink_Interrupt_Cfg * int_cfg)
{
	int i;
	
	hyperLinkRegs->CTL = hyperLinkRegs->CTL
		|(int_cfg->int_local<<CSL_VUSR_CTL_INTLOCAL_SHIFT)
		|(int_cfg->sts_int_enable<<CSL_VUSR_CTL_INTENABLE_SHIFT)
		|(int_cfg->sts_int_vec<<CSL_VUSR_CTL_INTVEC_SHIFT)
		|(int_cfg->int2cfg<<CSL_VUSR_CTL_INT2CFG_SHIFT);

	for(i=0; i<64; i++)
	{
		hyperLinkRegs->INT_CTL_IDX = i;
		hyperLinkRegs->INT_CTL_VAL= 
			(int_cfg->int_event_cntl[i].Int_en<<CSL_VUSR_INT_CTL_VAL_INTEN_SHIFT)
			|(int_cfg->int_event_cntl[i].Int_type<<CSL_VUSR_INT_CTL_VAL_INTTYPE_SHIFT)
			|(int_cfg->int_event_cntl[i].Int_pol<<CSL_VUSR_INT_CTL_VAL_INTPOL_SHIFT)
			|(int_cfg->int_event_cntl[i].si_en<<CSL_VUSR_INT_CTL_VAL_SIEN_SHIFT)
			|(int_cfg->int_event_cntl[i].mps<<CSL_VUSR_INT_CTL_VAL_MPS_SHIFT)
			|(int_cfg->int_event_cntl[i].vector<<CSL_VUSR_INT_CTL_VAL_VECTOR_SHIFT);
	}
		
	for(i=0; i<16; i++)
	{
		hyperLinkRegs->INT_PTR_IDX = i;
		hyperLinkRegs->INT_PTR_VAL= int_cfg->int_set_register_pointer[i];
	}
}

void Keystone_HyperLink_Init(HyperLink_Config * hyperLink_cfg)
{
	if(hyperLinkRegs->STS) 	//soft reset HyperLink if it is already enabled
		Keystone_HyperLink_soft_reset();	

#ifdef DSP_TYPE_TCI6614
	//enable HyperLink power and clock domain
	Keystone_enable_PSC_module(CSL_PSC_PD_ALWAYSON, CSL_PSC_LPSC_VUSR);
#else
	Keystone_enable_PSC_module(CSL_PSC_PD_HYPERBRIDGE, CSL_PSC_LPSC_HYPERBRIDGE);
#endif

	if(HyperLink_LOOPBACK==hyperLink_cfg->loopback_mode)
	{
		/*disable all portal or remote register operation
		This bit should be set before iloop or reset bits are changed.*/
		//hyperLinkRegs->CTL |= CSL_VUSR_CTL_SERIAL_STOP_MASK;

		hyperLinkRegs->CTL |= CSL_VUSR_CTL_LOOPBACK_MASK;

		/*enable operation*/
		//hyperLinkRegs->CTL &= ~(CSL_VUSR_CTL_SERIAL_STOP_MASK);
	}

	hyperLinkSerdesRegs = (SerdesRegs *)&boot_cfg_regs->VUSR_CFGPLL;
	Keystone_HyperLink_Serdes_init(&hyperLink_cfg->serdes_cfg, hyperLinkSerdesRegs);
	Wait_Hyperlink_PLL_Lock();

    Keystone_HyperLink_Addr_Map(&hyperLink_cfg->address_map);

    Keystone_HyperLink_Interrupt_init(&hyperLink_cfg->interrupt_cfg);

	/*---------wait for link status OK-------------*/
	while(0==(hyperLinkRegs->STS&CSL_VUSR_STS_LINK_MASK));
	while(hyperLinkRegs->STS&CSL_VUSR_STS_SERIAL_HALT_MASK);
	while(hyperLinkRegs->STS&CSL_VUSR_STS_PLL_UNLOCK_MASK);
}

