/*-----------------------------------------------------------------------------
 * Source Location: OMAPL1x/support/src/
 * File: OMAPL1x_common.c
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *---------------------------------------------------------------------------*/
#ifdef OMAPL137
#include <OMAPL137_common.h>
#elif OMAPL138
#include <OMAPL138_common.h>
#endif


/*-----------------------------------------------------------------------------
 *
 *					Global Variables Used in rCSL Examples
 * 
 *---------------------------------------------------------------------------*/
// Pointers to OMAPL1x Module Memory Mapped Registers
CSL_SyscfgRegsOvly sysRegs = (CSL_SyscfgRegsOvly)(CSL_SYSCFG_0_REGS);
CSL_PscRegsOvly psc0Regs = (CSL_PscRegsOvly)(CSL_PSC_0_REGS);
CSL_PscRegsOvly psc1Regs = (CSL_PscRegsOvly)(CSL_PSC_1_REGS);
CSL_PllcRegsOvly pll0Regs = (CSL_PllcRegsOvly)(CSL_PLLC_0_REGS);
CSL_GpioRegsOvly gpioRegs = (CSL_GpioRegsOvly)(CSL_GPIO_0_REGS);
CSL_I2cRegsOvly i2c0Regs = (CSL_I2cRegsOvly)(CSL_I2C_0_REGS);
CSL_TmrRegsOvly tmr0Regs = (CSL_TmrRegsOvly)(CSL_TMR_0_REGS);
CSL_TmrRegsOvly tmr1Regs = (CSL_TmrRegsOvly)(CSL_TMR_1_REGS);
CSL_UartRegsOvly uart0Regs = (CSL_UartRegsOvly)(CSL_UART_0_REGS);
CSL_UartRegsOvly uart2Regs = (CSL_UartRegsOvly)(CSL_UART_2_REGS);
CSL_AintcRegsOvly aintcRegs = (CSL_AintcRegsOvly)(CSL_AINTC_0_REGS);
CSL_DspintcRegsOvly dspintcRegs = (CSL_DspintcRegsOvly)(CSL_INTC_0_REGS);

// Common Global Example Variables (Not Specific to OMAPL1x)
volatile int runExample, runTimerDelay, counter, errorCount;

// ARM System Interrupt Table (Map System Interrupts to Corresponding ISRs)
ISRpointer sysISRtbl[NUM_AINTC_EVENTID] = {NULL};

// Global PSC Variables
volatile Bool modulesEnabled;
volatile int pscTimeoutCount;

// Global UART/I2C Variables (Use with Interrupts)
volatile Bool dataUnavailable;
volatile Bool transmitterFull;


/*-----------------------------------------------------------------------------
 *
 *				Global Variables Used in OMAPL1x_common.c Only
 * 
 *---------------------------------------------------------------------------*/
// Used in I2C0 Functions
static volatile int _counter;


/*-----------------------------------------------------------------------------
 *
 * 							OMAPL1x PSC1 Functions
 * 
 *---------------------------------------------------------------------------*/
// Function: check_psc_transition
// Return: 	Integer - Value of variable pscTimeoutCount
int check_psc_transition (void)
{
	// Reset the PSC Timeout Counter
	pscTimeoutCount = 0;
	
	// Increment the PSC Timeout Counter While the PSC is in Transition
	while((CSL_FEXT(psc1Regs->PTSTAT, PSC_PTSTAT_GOSTAT0)
			== CSL_PSC_PTSTAT_GOSTAT0_IN_TRANSITION)
			&& (pscTimeoutCount != pscTimeout))
		pscTimeoutCount++;
	
	// Return the PSC Timeout Counter
	return pscTimeoutCount;
}/* check_psc_transition */

// Function: check_psc_MDSTAT
// Return: 	Integer - Value of variable pscTimeoutCount
// Arguments:	Integer - PSC Module number
int check_psc_MDSTAT (int module)
{
	// Reset the PSC Timeout Counter
	pscTimeoutCount = 0;
	
	// Increment the PSC Timeout Counter While the 
	// Specified Module State is Disabled
	while((CSL_FEXT(psc1Regs->MDSTAT[module], PSC_MDSTAT_STATE) 
			!= CSL_PSC_MDSTAT_STATE_ENABLE) && (pscTimeoutCount != pscTimeout))
		pscTimeoutCount++;
	
	// Return the PSC Timeout Counter
	return pscTimeoutCount;
} /* check_psc_MDSTAT */


/*-----------------------------------------------------------------------------
 *
 * 							OMAPL1x I2C0 Functions
 * 
 *---------------------------------------------------------------------------*/
int I2C_write (Uint16 i2c_addr, Uint8* data, Uint16 len)
{
	// Specify Number of Data Words to Transfer (Only for Master Mode)
	CSL_FINS(i2c0Regs->ICCNT, I2C_ICCNT_ICDC, len);
	
	// Specify I2C Slave Address
	CSL_FINS(i2c0Regs->ICSAR, I2C_ICSAR_SADDR, i2c_addr);
	
	// Configure I2C Peripheral as Master Transmitter
	CSL_FINST(i2c0Regs->ICMDR, I2C_ICMDR_MST, MASTER_MODE);
	CSL_FINST(i2c0Regs->ICMDR, I2C_ICMDR_TRX, TX_MODE);
	
	// Send Start Signal on I2C Bus
	CSL_FINST(i2c0Regs->ICMDR, I2C_ICMDR_STT, SET);
	
	// Send Data to Slave on I2C Bus
	for(_counter = 0; _counter < len; _counter++)
	{
		// Send Data to the Transmit Buffer
		CSL_FINS(i2c0Regs->ICDXR, I2C_ICDXR_D, data[_counter]);
		
		// Wait for Transmit Buffer to Clear or NACK Signal
		while((CSL_FEXT(i2c0Regs->ICSTR, I2C_ICSTR_ICXRDY) == 0)
				&& (CSL_FEXT(i2c0Regs->ICSTR, I2C_ICSTR_ARDY) == 0));
		
		// Check for NACK Signal
		if(CSL_FEXT(i2c0Regs->ICSTR, I2C_ICSTR_NACK) == 1)
		{
			// Terminate I2C Transfer
			terminate_I2C_transfer();
			
			// Retry I2C Write
			return 1;
		}
	}
	
	// Wait for access to the I2C Registers
	while(CSL_FEXT(i2c0Regs->ICSTR, I2C_ICSTR_ARDY) == 0);
	
	// Terminate I2C Transfer
	terminate_I2C_transfer();
	
	// I2C Write Successful
	return 0;
}/* I2C_write */

/*---------------------------------------------------------------------------*/

int I2C_read (Uint16 i2c_addr, Uint8* data, Uint16 len)
{
	// Specify Number of Data Words to Receive (Only for Master Mode)
	CSL_FINS(i2c0Regs->ICCNT, I2C_ICCNT_ICDC, len);
	
	// Specify I2C Slave Address
	CSL_FINS(i2c0Regs->ICSAR, I2C_ICSAR_SADDR, i2c_addr);
	
	// Configure I2C Peripheral as Master Receiver
	CSL_FINST(i2c0Regs->ICMDR, I2C_ICMDR_MST, MASTER_MODE);
	CSL_FINST(i2c0Regs->ICMDR, I2C_ICMDR_TRX, RX_MODE);
	
	// Send Start Signal on I2C Bus
	CSL_FINST(i2c0Regs->ICMDR, I2C_ICMDR_STT, SET);
	
	// Receive Data from Slave on I2C Bus
	for(_counter = 0; _counter < len; _counter++)
	{
		// Wait for Receiver Buffer to Clear of NACK Signal
		while((CSL_FEXT(i2c0Regs->ICSTR, I2C_ICSTR_ICRRDY) == 0)
				&& (CSL_FEXT(i2c0Regs->ICSTR, I2C_ICSTR_ARDY) == 0));
		
		// Check for NACK Signal
		if(CSL_FEXT(i2c0Regs->ICSTR, I2C_ICSTR_NACK) == 1)
		{
			// Terminate I2C Transfer
			terminate_I2C_transfer();
			
			// Retry I2C Read
			return 1;
		}
		
		// Store Data from the Receiver Buffer
		data[_counter] = CSL_FEXT(i2c0Regs->ICDRR, I2C_ICDRR_D);
	}
	
	// Wait for access to the I2C Registers
	while(CSL_FEXT(i2c0Regs->ICSTR, I2C_ICSTR_ARDY) == 0);
	
	// Terminate I2C Transfer
	terminate_I2C_transfer();
	
	// I2C Read Successful
	return 0;
}/* I2C_read */

/*---------------------------------------------------------------------------*/

void terminate_I2C_transfer (void)
{
	// Send Stop Signal on I2C Bus
	CSL_FINST(i2c0Regs->ICMDR, I2C_ICMDR_STP, SET);
	
	// Clear NACK Bit (Does Nothing if NACK Bit Not Set)
	CSL_FINST(i2c0Regs->ICSTR, I2C_ICSTR_NACK, CLEAR);
	
	// Wait for Stop Signal to be Received (I2C Peripheral Bumped to Slave)
	while(CSL_FEXT(i2c0Regs->ICMDR, I2C_ICMDR_MST) == 1);
}/* terminate_I2C_transfer */


/*-----------------------------------------------------------------------------
 *
 * 							OMAPL1x UART2 Functions
 * 
 *---------------------------------------------------------------------------*/
void UART_sendData (Uint8 data)
{
	// Wait for Empty Transmitter Buffer
	while(transmitterFull);
	
	// Reset Bool Variable
	transmitterFull = TRUE;
	
	// Send Data to Transmitter Buffer
	CSL_FINS(uart2Regs->THR, UART_THR_DATA, data);
}/* UART_sendData */

/*---------------------------------------------------------------------------*/

Uint8 UART_receiveData (void)
{
	// Wait for Full Receiver Buffer
	while(dataUnavailable);
	
	// Reset Bool Variable
	dataUnavailable = TRUE;
	
	// Process Data in Receiver Buffer
	return CSL_FEXT(uart2Regs->RBR, UART_RBR_DATA);
}/* UART_receiveData */


//								END OF FILE
