#ifndef _DSIGTAPI_H
#define _DSIGTAPI_H

/************************************************************************************
 * FILE PURPOSE: Structures shard between DSP and host for DSIG
 ************************************************************************************
 * FILE NAME: disgtapi.h
 *
 * DESCRIPTION: Defines the shared structures for the dsigt module of the DSP.
 *              These structures are used by the DSP and HOST.
 *
 * Copyright (C) 2003, Texas Instruments Inc. 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#include <ti/mas/iface/ifeco/mgdefs.h>

/* Dummy Define for a Micro Macro */
#ifndef TNI_ATTR_PACK
#define TNI_ATTR_PACK
#endif

/* /------------------------------------\
   | Definition: Signal specification.  |
   \------------------------------------/ */
#ifndef GG_REF_PAGE
typedef struct dsigtstSigDef_s {

  /* ID fields */
  tuint signalSetId  : 8;       /* CAS variant, 0 = undefined */
  tuint signalId     : 8;       /* Unique signal ID, 0 - 11   */
  
  
  /* states */
  tuint makeRxValid  : 4;       /* 1 = valid bit, 0 = don't care */
  tuint makeRxState  : 4;       
  
  tuint breakRxValid : 4;
  tuint breakRxState : 4;
  
  tuint curTxValid   : 4;
  tuint curTxState   : 4;
  
  tuint matchLeading : 1;
  tuint rsvd         : 3;
  tuint intGroup     : 2;       /* Integration report group member        */
  tuint mutualGroup  : 2;       /* Mutual report group member             */
  
  
  
  /* times */
  tuint minMake      : 8;       /* All times are in 10ms units. A value of 255 */
  tuint maxMake      : 8;       /* matches all times of 2550 ms or longer.     */
                                /* A value of 0 indicates no minimum time      */
  tuint minBreak     : 8;
  tuint maxBreak     : 8;
  
} TNI_ATTR_PACK dsigtstSigDef_t;
#endif /* GG_REF_PAGE */

/*  /----------------------------------------------------\
    | Micro compatible definiton of signal specification |
    \----------------------------------------------------/ */
typedef struct dsigtstMSigDef_s {

  tuint signal_set_id_sig_id;  /* bits 15-8 - signal set id (CAS variant)
                                 * bits 7-0  - signal id 0-11 */
#define MGB_SIG_DEF_SIGNAL_SET_ID_MASK          0xFF00
#define MGB_SIG_DEF_SIGNAL_SET_ID_SHIFT         8                                 
#define MGB_SIG_DEF_SIGNAL_ID_MASK              0x00FF
#define MGB_SIG_DEF_SIGNAL_ID_SHIFT             0                                 
                                   
  tuint rxstates;              /* bits 15-12 - make rx valid
                                 * bits 11-8  - make rx state
                                 * bits 7-4   - break rx valid
                                 * bits 3-0   - break rx state */
#define MGB_SIG_DEF_MAKE_RX_STATE_MASK          0xFF00
#define MGB_SIG_DEF_MAKE_RX_STATE_SHIFT         8                                 
#define MGB_SIG_DEF_BREAK_RX_STATE_MASK         0x00FF
#define MGB_SIG_DEF_BREAK_RX_STATE_SHIFT        0                                 
                                
  tuint txstates_groups;       /* bits 15-12 - break tx valid 
                                 * bits 11-8  - break tx state  
                                 * bit  7     - match leading flag
                                 * bits 6-4   - reserved
                                 * bits 3-2   - integration group number
                                 * bits 1-0   - mutal report group number  */
#define MGB_SIG_DEF_CUR_TX_STATE_MASK           0xFF00
#define MGB_SIG_DEF_CUR_TX_STATE_SHIFT          8 
#define MGB_SIG_DEF_GROUP_OPTION_MASK           0x00FF
#define MGB_SIG_DEF_GROUP_OPTION_SHIFT          0 
                                 
                                  
  tuint makeTimes;             /* bits 15-8  - min make time  
                                 * bits 7-0   - max make time   */
#define MGB_SIG_DEF_MIN_MAKE_TIME_MASK          0xFF00
#define MGB_SIG_DEF_MIN_MAKE_TIME_SHIFT         8                                 
#define MGB_SIG_DEF_MAX_MAKE_TIME_MASK          0x00FF
#define MGB_SIG_DEF_MAX_MAKE_TIME_SHIFT         0                                 
                                  
  tuint breakTimes;            /* bits 15-8  - min break time
                                 * bits 7-0   - max break time
                                 * All times in units of 10ms */
#define MGB_SIG_DEF_MIN_BREAK_TIME_MASK         0xFF00
#define MGB_SIG_DEF_MIN_BREAK_TIME_SHIFT        8                                 
#define MGB_SIG_DEF_MAX_BREAK_TIME_MASK         0x00FF
#define MGB_SIG_DEF_MAX_BREAK_TIME_SHIFT        0                                 
} TNI_ATTR_PACK dsigtstMSigDef_t;                               

union MGB_DSIGTST_SIG_DEF {
#ifndef GG_REF_PAGE
  dsigtstSigDef_t  sigdef;
#endif
  dsigtstMSigDef_t msigdef;
} TNI_ATTR_PACK;

#define MGB_DSIGTST_NO_VARIANT      0   /* No variant for the signal (disable signal) */

/*******************************************
 * Global integration defintions 
 *******************************************/
#ifndef GG_REF_PAGE
typedef struct dsigtstGlobIntg_s {

  tuint     signalSetId       :  8;
  tuint     integrationGroup  :  8;
  tuint     reportEveryPeriod;
  
} TNI_ATTR_PACK dsigtstGlobIntg_t;
#endif

typedef struct dsigtstMGlobIntg_s {
  
  tuint    signalSetId_integrationGroup;   /* Bits 15 - 8 = signal set ID
                                             * Bits  7 - 0 = integration group */
#define MGB_SIG_INTG_GRP_SIGNAL_SET_ID_MASK         0xFF00
#define MGB_SIG_INTG_GRP_SIGNAL_SET_ID_SHIFT        8                                 
#define MGB_SIG_INTG_GRP_INTG_GROUP_ID_MASK         0x00FF
#define MGB_SIG_INTG_GRP_INTG_GROUP_ID_SHIFT        0                                 
  tuint    report_every_period;            /* 10ms units. Report at least this often.
                                             * 0 = no periodic reports */
} TNI_ATTR_PACK dsigtstMGlobIntg_t;

union MGB_DSIGTST_INTG_DEF {
#ifndef GG_REF_PAGE
  dsigtstGlobIntg_t  dsigtstGlobIntg;
#endif
  dsigtstMGlobIntg_t dsigtstMGlobIntg;
} TNI_ATTR_PACK;
  

/*******************************************
 * groups and integration group definitions 
 *******************************************/
enum {
  MGB_DSIGTST_REPORT_GROUP_NONE =       0, /* Signal is not a member of a report group */
  MGB_DSIGTST_REPORT_GROUP_A    =       1, /* Signal is a member of report group A     */
  MGB_DSIGTST_REPORT_GROUP_B    =       2  /* Signal is a member of report group B     */
};

enum {
  MGB_DSIGTST_INTEGRATION_GROUP_NONE  = 0, /* Signal is not a member of an integration group */
  MGB_DSIGTST_INTEGRATION_GROUP_A     = 1  /* Signal is a member of integration group A      */
}; 
 

/******************************************************************************
 * Channel independent host to DSP message
 ******************************************************************************/
struct MGB_DSIGTST_ID_CONTROL {
  tuint  signal_set;       /* Variant   */
  tuint  signal_id;        /* Signal ID */
} TNI_ATTR_PACK;
typedef struct MGB_DSIGTST_ID_CONTROL dsigtstId_t;
 
struct MGB_MDS_DSIGTST_GLOBAL_CONTROL {
  tuint code;  /* Operation */
  union  {
    struct MGB_DSIGTST_ID_CONTROL idc;
    union  MGB_DSIGTST_INTG_DEF   intg;
    union  MGB_DSIGTST_SIG_DEF    sig;
  } u;
} TNI_ATTR_PACK;

/* Code values */
#define MGB_DSIGTST_GLOBAL_SIG_DEF          1   /* Define a signal. Uses SIG_DEF        */
#define MGB_DSIGTST_GLOBAL_INTG             2   /* Integration group definition         */
#define MGB_DSIGTST_GLOBAL_SIG_QUERY        3   /* Return signal def. Uses ID_CONTROL   */
#define MGB_DSIGTST_GLOBAL_SET_DEL          4   /* Delete a variant uses ID_CONTROL     */
#define MGB_DSIGTST_GLOBAL_SIG_DEL          5   /* Deletes a signal uses ID_CONTROL     */


/*****************************************************************************
 * Channel specific host to DSP message
 *****************************************************************************/
 
#define MGB_DSIGTST_MAX_INTG 1


#ifndef GG_REF_PAGE
typedef struct dsigtstIntgChanDef_s { 
  tuint periodTolerance  : 8;    /* 10ms units. Report if period changes by more then
                                  * this amount. 0 for no reports on changes */
  tuint reportEveryCount : 8;    /* Reports on this many counts. 0 for no reports
                                  * based on counts */
} TNI_ATTR_PACK dsigtstIntgChanDef_t;                                
#endif

#ifndef GG_REF_PAGE
typedef struct dsigtstChControl_s  {
  tuint signalSetId  : 8;        /* Signal set, 0 for none (disable channel)      */
  tuint numIntgDefs  : 8;        /* Number of integration groups defined (0 or 1) */
  tuint debounceTime : 16;       /* Debounce period in ms                         */
  dsigtstIntgChanDef_t intg[MGB_DSIGTST_MAX_INTG];
} TNI_ATTR_PACK dsigtstChControl_t;
#endif
 
typedef struct dsigtstMIntgChanDef_s  {
  tuint  periodTolerance_reportEveryCount; /* Bits 15-8: period tolerance    */
                                            /* Bits  7-0: report every count */
#define MGB_DSIGTST_CHAN_CTRL_PERIOD_TOLERANCE_MASK     0xff00
#define MGB_DSIGTST_CHAN_CTRL_PERIOD_TOLERANCE_SHIFT    8 
#define MGB_DSIGTST_CHAN_CTRL_REPORT_EVERY_COUNT_MASK   0x00ff
#define MGB_DSIGTST_CHAN_CTRL_REPORT_EVERY_COUNT_SHIFT  0                                            
} TNI_ATTR_PACK dsigtstMIntgChanDef_t;
 
 
typedef struct dsigtstMChControl_s  {
  tuint signalSetId_numIntgDefs;  /* Bits 15-8: signal set  ID                     */
                                   /* Bits  7-0: number of integration groups (0-1) */
#define MGB_DSIGTST_CHAN_CTRL_SIGNAL_SET_ID_MASK            0xff00
#define MGB_DSIGTST_CHAN_CTRL_SIGNAL_SET_ID_SHIFT           8
#define MGB_DSIGTST_CHAN_CTRL_NUM_INTG_DEFS_MASK            0x00ff
#define MGB_DSIGTST_CHAN_CTRL_NUM_INTG_DEFS_SHIFT           0                                 
  tuint debounce_time;            /* Debounce period in ms                         */
  dsigtstMIntgChanDef_t intg[MGB_DSIGTST_MAX_INTG];
} TNI_ATTR_PACK dsigtstMChControl_t;


union MGB_DSIGTST_CH_CTL  {
#ifndef GG_REF_PAGE
  dsigtstChControl_t  dsigtstChControl;
#endif
  dsigtstMChControl_t dsigtstMChControl;
} TNI_ATTR_PACK;
  
struct MGB_MDS_DSIGTST_CH_CONTROL {
  union MGB_DSIGTST_CH_CTL  u;
} TNI_ATTR_PACK;


typedef struct MGB_MDS_DSIGTST_CH_CONTROL dsigtstChCtl_t;

  
  
/**********************************************************************************
 * DSP to host message
 **********************************************************************************/
#ifndef GG_REF_PAGE
typedef struct dsigtstIntEvent_s {
  tuint integrationGroup    :  8;       /* Integration group                       */
  tuint currentSig          :  8;       /* Integration Signal number (1-12)        */
  tuint events;                         /* Event types reported                    */
  tuint count;                          /* Total count since group created         */
  tuint period;                         /* 10ms units. Period from pulse to pulse  */
                                        /* measured from make match to make match  */
                                        /* 0 means only 1 pulse seen so far        */
                                        /* 0xffff means no pulses seen so far      */
                                        /* 0xfffe means pulse period >= 655.34 sec */
} TNI_ATTR_PACK dsigtstIntEvent_t;                                        
#endif                                      

#define MGB_DSIGTST_PERIOD_NO_PULSE     0xFFFF
#define MGB_DSIGTST_PERIOD_MAX          0xFFFE

typedef struct dsigtstRxSmEvent_s {
  tuint timestamp;                      /* Timestamp with arbitrary 0               */
  tuint currentRxState;                 /* Current rx bits                          */
  tuint currentTxState;                 /* Current tx bits                          */
  tuint reportedSignals;                /* All signals being reported as matched    */
  tuint matchedSignals;                 /* All signals currently in matched state   */
  tuint possibleMatchedSignals;         /* All signals in a partial matched state   */
  tuint latchedSignals;                 /* All signals that have matched but are    */
                                        /* waiting for other members of the mutual  */
                                        /* report group to resolve                  */
  tuint numIntegrationGroups;           /* Number of integration groups reported    */
  
  dsigtstIntEvent_t intGroup[MGB_DSIGTST_MAX_INTG]; /* Integration group report     */
} TNI_ATTR_PACK dsigtstRxSmEvent_t;


typedef struct dsigtstMIntEvent_s {
  tuint integrationGroup_currentSig;    /* Bits 15-8: integration group            */
                                         /* Bits  7-0: integration signal number    */
#define MGB_DSIGTST_INT_EVENT_INTG_GROUP_MASK       0xff00
#define MGB_DSIGTST_INT_EVENT_INTG_GROUP_SHIFT      8
#define MGB_DSIGTST_INT_EVENT_CURR_SIG_MASK         0x00ff
#define MGB_DSIGTST_INT_EVENT_CURR_SIG_SHIFT        0
                                         
  tuint events;                         /* Event types reported                    */
  tuint count;                          /* Total count since group created         */ 
  tuint period;                         /* 10ms units. Period from pulse to pulse  */
                                         /* measured from make match to make match  */
                                         /* 0 means only 1 pulse seen so far        */
                                         /* 0xffff means no pulses so far           */
                                         /* 0xfffe means pulse period >= 655.34 sec */
} TNI_ATTR_PACK dsigtstMIntEvent_t;                                        

typedef struct dsigtstMRxSmEvent_s  {
  tuint timestamp;                      /* Timestamp with arbitrary 0              */
  tuint currentRxState;                 /* Current rx bits                         */
  tuint currentTxState;                 /* Current tx bits                         */
  tuint reportedSignals;                /* All signals being reported as matched   */
  tuint matchedSignals;                 /* All signals currently in matched state  */
  tuint possibleMatchedSignals;         /* All signals in a partial matched state  */
  tuint latchedSignals;                 /* All signals that have matched but are   */
                                         /* waiting for toerh members of the mutual */
                                         /* report group to resolve                 */
  tuint numIntegrationGroups;           /* Number of integration groups reported   */
  
  dsigtstMIntEvent_t intGroup[MGB_DSIGTST_MAX_INTG];    /* Integration group report */
} TNI_ATTR_PACK dsigtstMRxSmEvent_t;    

union MGB_DSIGTST_SIG_REP  {
#ifndef GG_REF_PAGE
  dsigtstRxSmEvent_t  dsigtstRxSmEvent;
#endif
  dsigtstMRxSmEvent_t dsigtstMRxSmEvent;
} TNI_ATTR_PACK;


/* Bitfield mapping of event field */
#define MGB_DSIGTST_INTG_EV_COUNT      (1 << 0)      /* Report due to count reached    */
#define MGB_DSIGTST_INTG_EV_PERIOD     (1 << 1)      /* Report due to period reached   */
#define MGB_DSIGTST_INTG_EV_PRD_CHANGE (1 << 2)      /* Report due to change in period */
#define MGB_DSIGTST_INTG_EV_OWNER      (1 << 3)      /* Report due to new signal       */
   
   
struct MGB_DMS_DSIGTST_REPORT {
  union MGB_DSIGTST_SIG_REP report;
} TNI_ATTR_PACK;

typedef struct MGB_DMS_DSIGTST_REPORT dsigtstReport_t;
  
  
struct MGB_DMS_DSIGTST_RET_SIGDEF {
  union MGB_DSIGTST_SIG_DEF sig;
} TNI_ATTR_PACK;

/*******************************************************************************
 * Return values from dsigtst. Only the error values used for error indication
 * are defined here. Non-error values are defined in dsigt.h
 ******************************************************************************/
# define MGB_ERR_DSIGTST_INVLD_SET                231  /* Invalid signal set provided to dsigtst     */
# define MGB_ERR_DSIGTST_INVLD_SIGID              232  /* Invalid signal ID provided to dsigtst      */
# define MGB_ERR_DSIGTST_INVLD_SIG                233  /* Invalid signal definition to dsigtst       */
# define MGB_ERR_DSIGTST_INVLD_NSETS              234  /* The maximum number of signal sets exceeded */
# define MGB_ERR_DSIGTST_GMC_ALLOC_FAIL           235  /* GMC allocation failed                      */
# define MGB_ERR_DSIGTST_INVLD_CTL_RESP           236  /* Catch all dsigtst error                    */
# define MGB_ERR_DSIGTST_INVLD_GLBL_MSG           237  /* Invalid global message                     */
# define MGB_ERR_DSIGTST_INVLD_CTL_REQ            238  /* Invalid control code to dsigtst            */
# define MGB_ERR_DSIGTST_SIG_SPACE                239  /* Signal space not allocated                 */
# define MGB_ERR_DSIGTST_INVLD_BLK_COUNT          240  /* Invalid memory block count to dsigtst      */
# define MGB_ERR_DSIGTST_INVLD_CH_MSG             241  /* Invalid channel message                    */
# define MGB_ERR_DSIGTST_INVLD_CH                 242  /* Invalid channel ID                         */
# define MGB_ERR_DSIGTST_INVLD_CH_STATE           243  /* Message for closed dsig channel            */
# define MGB_ERR_DSIGTST_SIG_DELETE               244  /* Tried to delete a deleted signal           */


/* The following values are also defined in mgmicdsp.h */
#ifndef MGB_DSIG_ENABLE_RX_SM
#define MGB_DSIG_ENABLE_RX_SM         0x0020  /* Enable Rx state machine                  */
#endif

#ifndef MGB_DSIG_RESET_RX_SM
#define MGB_DSIG_RESET_RX_SM          0x0040  /* Reset Rx state machine                   */
#endif

#endif  /* _DSIGTAPI_H */



