/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/



#ifndef __IUUPAPI_H__
#define __IUUPAPI_H__

#include <ti/mas/iface/ifeco/mgdefs.h>

/* Dummy Define for a Micro Macro */
#ifndef TNI_ATTR_PACK
#define TNI_ATTR_PACK	 
#endif

/******************************************************************************
 * Name:       MGB_MDS_IU_ERR_CONFIG                                          *
 * Description: The Iu encapsulation error reporting configuration allows     *
 *              the DIM Client to specify how certain error types are         *
 *              to be reported to the host/DIM Client.                        *
 *                                                                            *
 *              The choices are:                                              *
 *                  reported 1:1                                              *
 *                  rate throttled                                            *
 *                                                                            *
 *              Reported 1:1 should be used for any errors that are needed to *
 *              drive higher level state-machines.  These errors are reported *
 *              by the rx_p2p report and contain information on the error     *
 *              cause and error distance.                                     *
 *                                                                            *
 *              Rate throttled will be used for other errors.  These errors   *
 *              are reported in a pkt_violation but not more often than a     *
 *              configured rate.  The pkt_violation data contains information *
 *              on the current error counters and the error types that        *
 *              have occured since the last report.                           *
 *                                                                            *
 *              According to the Iu protocol, errors are classified into the  *
 *              folowing types/classes:                                       *
 *                  syntax errors                                             *
 *                  semantic errors                                           *
 *                  other errors                                              *
 *                                                                            *
 *              A value of 1 defines all errors to be throttled.              *
 *              Setting a bit to '0' enables 1:1 reporting for that error     *
 *              type.                                                         *
 *                                                                            *
 ******************************************************************************/
struct MGB_MDS_IU_ERR_CONFIG {
  tuint  syntax_err_mask;     /* Syntax error reporting mask */
#define MGB_IU_ERR_HEADER_CRC               0x0000
#define MGB_IU_ERR_PAYLOAD_CRC              0x0002
#define MGB_IU_ERR_UNEXP_FRAME_NUM          0x0004
#define MGB_IU_ERR_FRAME_LOSS               0x0008
#define MGB_IU_ERR_UNKOWN_PDU_TYPE          0x0010
#define MGB_IU_ERR_UNKNOWN_PROC             0x0020
#define MGB_IU_ERR_UNKNOWN_RESRV_VALUE      0x0040
#define MGB_IU_ERR_UNKNOWN_FIELD            0x0080
#define MGB_IU_ERR_FRAME_TOO_SHORT          0x0100
#define MGB_IU_ERR_MISSING_FIELDS           0x0200  
  tuint  symantic_err_mask;   /* Semantic error reporting mask */
#define MGB_IU_ERR_UNEXP_PDU_TYPE           0x0000
#define MGB_IU_ERR_SEMANTIC_SPARE_A         0x0002
#define MGB_IU_ERR_UNEXP_PROC               0x0004
#define MGB_IU_ERR_UNEXP_RFCI               0x0008
#define MGB_IU_ERR_UNEXP_VALUE              0x0010
  tuint  other_err_mask;      /* Other error reporting mask */
#define MGB_IU_ERR_INIT_FAILURE1            0x0000
#define MGB_IU_ERR_INIT_FAILURE2            0x0002
#define MGB_IU_ERR_INIT_FAILURE3            0x0004
#define MGB_IU_ERR_RATECNTL_FAILURE         0x0008
#define MGB_IU_ERR_ERREVENT_FAILURE         0x0010
#define MGB_IU_ERR_TIMEALIGN_NOT_SUPP       0x0020
#define MGB_IU_ERR_TIMEALIGN_NOT_POSSIBLE   0x0040
#define MGB_IU_ERR_VERSION_NOT_SUPPORTED    0x0080
} TNI_ATTR_PACK;

/******************************************************************************
 * Name:       MGB_ENCAP_IU_PROCEDURE                                         *
 * Description: The IU encapsulation procedure configuration specifies        *
 *              paramaters for each of the different procedure types in Iu.   *
 *                                                                            *
 *              Each procedure has its own acknowledgement timeout value and  *
 *              its own retry limit.                                          *
 *                                                                            *
 * Note:        Currently NO_AUTO_ACK is only supported for INIT procedure    *
 *              and RATE CONTROL procedure.                                   *
 *                                                                            *
 ******************************************************************************/
#define MGB_IU_MAX_PROCEDURES           3       /* this maximum number of procedures */

#define MGB_IU_PROCEDURE_INIT           0       /* init procedures */
#define MGB_IU_PROCEDURE_RATE_CTRL      1       /* rate control procedures */
#define MGB_IU_PROCEDURE_TIME_ALIGN     2       /* time alignment procedures */

#define MGB_IU_PROC_OPT_NO_AUTO_ACK     0x01    /* Do not automaticly acknowledge */
                                                /* these procedures. */
/******************************************************************************
 * Name:        MGB_MDS_ENCAP_IU_PARAMS                                              *
 * Description: The encapsulation configuration for Iu related protocols.     *
 *              This structure contains the configuration that is used for the* 
 *              IuFP protocol layer and indepedent of the various codings and *
 *              coding rates.                                                 *
 ******************************************************************************/
struct MGB_MDS_ENCAP_IU_PARAMS
{
  tuint  valid_params;       
#define MGB_IU_ENCAP_VALID_MODE         0x0001  /* mode is valid */
#define MGB_IU_ENCAP_VALID_FLAGS        0x0002  /* flags is valid */
#define MGB_IU_ENCAP_VALID_VER          0x0004  /* config_version & version_mask */
#define MGB_IU_ENCAP_VALID_ERR          0x0008  /* error configuration */
#define MGB_IU_ENCAP_VALID_PROCS        0x0010  /* procedures are valid */
  tuint  flags;              
#define MGB_IU_ENABLE_ERR_SDU           0x0010  /* deliver errored SDUs */
#define MGB_IU_ENABLE_TIME_ALIGN        0x0020  /* enable Time Alignment support */
  tuint  modes_config_version;                 
    /* modes and config version*/                            
    /* Bits 0-7  : Config version currently must be 1 or 2 */
    /* Bits 8-15 : operational mode                        */
#define MGB_IU_VERSION_MASK             0x00FF
#define MGB_IU_VERSION_SHIFT            0
#define MGB_IU_MODE_MASK                0xFF00
#define MGB_IU_MODE_SHIFT               8

  tuint  version_mask;                         /* bitmap of supported versions */

  /* See above for bit definitions. */
  tuint  syntax_err_mask;     /* Syntax error reporting mask */
  tuint  symantic_err_mask;   /* Semantic error reporting mask */
  tuint  other_err_mask;      /* Other error reporting mask */

  tuint ack_timeout[MGB_IU_MAX_PROCEDURES];
  tuint opts_max_retries[MGB_IU_MAX_PROCEDURES];
#define MGB_IU_OPTION_MASK             0xFF00
#define MGB_IU_OPTION_SHIFT            8
#define MGB_IU_MAX_RETRIES_MASK        0x00FF
#define MGB_IU_MAX_RETRIES_SHIFT       0
} TNI_ATTR_PACK;

#define MGB_MAX_IU_CODEC_RATES         16
/******************************************************************************
 * Name:        MGB_MDS_CODEC_PROFILE_IU                                      *
 * Description: Current coding codec type information                         *
 *              Current maximun rate in each direction and a collection of    * 
 *              codec profile records At present codec type must be           *
 *              TSG_CODING_GSM_AMR, TSG_CODING_PCM_MU or TSG_CODING_PCM_A     *
 * This structure is exactly the contents of the MGB_MDS_PROFILE member of    *
 * the profile define message.                                                *
 ******************************************************************************/
typedef struct 
{
  tuint  valid_params;
#define MGB_IU_CP_VALID_CODING_TYPE     0x0001
#define MGB_IU_CP_VALID_MAX_TX_RATE     0x0002
#define MGB_IU_CP_VALID_MAX_RX_RATE     0x0004
#define MGB_IU_CP_VALID_NEG_VERSION     0x0008
#define MGB_IU_CP_VALID_INIT_STATE      0x0010
#define MGB_IU_CP_VALID_OPTIONS         0x0020
#define MGB_IU_CP_VALID_NUM_RATES       0x0040
  
  tuint  coding_type;
  tuint  max_txrx_rate_index;
  /*Specifies the current Tx and Rx indexes which are zero based*/
  /*Bits 0-7 : Rx rate index                                    */
  /*Bits 8-15: Tx rate index                                    */
#define MGB_IU_MAX_RX_RATE_INDEX_MASK      0x00FF
#define MGB_IU_MAX_RX_RATE_INDEX_SHIFT     0
#define MGB_IU_MAX_TX_RATE_INDEX_MASK      0xFF00
#define MGB_IU_MAX_TX_RATE_INDEX_SHIFT     8
  
  tuint  neg_version_init_state;
  /*Specifies negotiated version and init state in packed manner */
  /*Bits 0-7 : init state                                        */
  /*Bits 8-15: negotiated version. Not used unless init state is */
  /*           inited                                            */
#define MGB_IU_INIT_STATE_MASK      0x00FF
#define MGB_IU_INIT_STATE_SHIFT     0
#define MGB_IU_NEG_VER_MASK         0xFF00
#define MGB_IU_NEG_VER_SHIFT        8
#define MGB_IU_INIT_STATE_NONE      0
#define MGB_IU_INIT_STATE_INITING   1
#define MGB_IU_INIT_STATE_READY     2
  
  tuint  options_num_rates;
  /* Specifies codec profile options and number of rates in packed manner
   * Bits 0-7 : number of rates
   * Bits 8-15: options
   * Bit 8: PDU type 0 or 1
   * Bit 9: 1 for A-Law 0 for mu-law
   * Bits 10-12: VAD TYPE:
    *  0 - RESERVED
    *  1 - RESERVED
    *  2 - RESERVED
    *  3 - MGB_VAD_INTERNAL1
    *  4 - MGB_VAD_INTERNAL2
    * Bit 13: Post filtering
    *  0 - Disable
    *  1 - Enable             */
#define MGB_IU_NUM_RATES_MASK       0x00FF
#define MGB_IU_NUM_RATES_SHIFT      0
#define MGB_IU_OPTIONS_MASK         0xFF00
#define MGB_IU_OPTIONS_SHIFT        8
#define MGB_IU_PDU_TYPE_1_MASK      0x0100 
#define MGB_IU_PDU_TYPE_1_SHIFT     8      
#define MGB_IU_PCM_TYPE_LAW_MASK    0x0200
#define MGB_IU_PCM_TYPE_LAW_SHIFT   9
  
  tuint rab_data[MGB_MAX_IU_CODEC_RATES];
  /*Specifies RFCI and size                                      */
  /*Bits 0-5 : RFCI                                              */
  /*Bits 6-15: size                                              */
#define MGB_IU_RFCI_MASK       0x003F
#define MGB_IU_RFCI_SHIFT      0
#define MGB_IU_SIZE_MASK       0xFFC0
#define MGB_IU_SIZE_SHIFT      6
} TNI_ATTR_PACK MGB_MDS_CODEC_PROFILE_IU;


/******************************************************************************
 * Name: MGB_IU_VIOLATION_ERRORS_T
 * Description: Consolodation of error events structure. Give count of total
 * errors since last report and a bitmask set to 1 for each error type seen.
 ******************************************************************************/
typedef struct MGB_IU_VIOLATION_ERROR_STATS_tag {
  tuint  num_errors;             /* Error count since last report */
  
    /* The following use the same form of bitmasks as the 
    ** error report configuration.  Each 1 bit below represents that one or
    ** more errors of the specified type has occured since the last report */
  tuint  syntax_errors_seen;
  tuint  symantic_errors_seen;
  tuint  other_errors_seen;
} TNI_ATTR_PACK MGB_IU_VIOLATION_ERROR_STATS_T;

/* Error Cause Subtypes */
#define MGB_IU_VIOL_ERRORS_CAUSE_SUBTYPE_RATECTRL_NACKS     1
#define MGB_IU_VIOL_ERRORS_CAUSE_SUBTYPE_RATECTRL_TIMEOUT   2
#define MGB_IU_VIOL_ERRORS_CAUSE_SUBTYPE_PROCEDURE_EXPIRED  3

typedef struct MGB_IU_VIOLATION_ERRORS_tag
{
  tuint  latest_error_dist_code;  /* latest error event code and distance */
#define MGB_IU_VIOL_CODE_MASK       0x00FF
#define MGB_IU_VIOL_CODE_SHIFT      0
#define MGB_IU_VIOL_DIST_MASK       0xFF00
#define MGB_IU_VIOL_DIST_SHIFT      8

  tuint  valid_params;
#define MGB_IU_VIOL_ERR_VALID_NUM_ERRORS  0x0001  /* number of errors cnt */
#define MGB_IU_VIOL_ERR_VALID_NUM_SEEN    0x0002  /* error_seen bitmaps */
#define MGB_IU_VIOL_ERR_VALID_CAUSE_SUBTYPE  0x0004  /* error cause subtype */

  MGB_IU_VIOLATION_ERROR_STATS_T stats;

  tuint  error_cause_subtype;  /* Only used for DSP to Host error reporting */
} TNI_ATTR_PACK MGB_IU_VIOLATION_ERRORS_T;

/******************************************************************************
 * Name: MGB_DMS_VIOLATION_IU_BODY
 * Description: Union of IU violation report types. For now only Error Events
 * that are filtered are reported. Filtered means consolodated and throttled
 * as opposed to reported individually.
 ******************************************************************************/
union MGB_DMS_VIOLATION_IU_BODY {
  MGB_IU_VIOLATION_ERRORS_T errors;
} TNI_ATTR_PACK;

/******************************************************************************
 * Name: MGB_DMS_VIOLATION_IU
 * Description: IU violation report types.
 ******************************************************************************/
struct MGB_DMS_VIOLATION_IU {
  tuint type;
#define MGB_IU_VIOLATION_TYPE_ERRORS    0x0001
  union MGB_DMS_VIOLATION_IU_BODY body;
} TNI_ATTR_PACK;


/******************************************************************************
 * Name:        MGB_MDS_RATECTRL_IU                                           *
 * Description: Current rate control information                              *
 *              Current maximun rate in each direction and a collection of    * 
 *              codec profile records At present codec type must be           *
 *              TSG_CODING_GSM_AMR, TSG_CODING_PCM_MU or TSG_CODING_PCM_A     *
 ******************************************************************************/
typedef struct 
{
  tuint  max_rate_index;
  /*Specifies the current Tx or Rx indexes which are zero based*/
  /*When from host -> DSP  it specifies Rx rate index          */
  /*When from DSP -> host it specifies Tx rate index           */
  tint   frame_num; 
  /* When from DSP -> host, specifies received RATE CONTROL frame number */
  /* When from host -> DSP, specifies RATECTRL frame number to be ACK-ed */
} TNI_ATTR_PACK MGB_MDS_RATECTRL_IU;

/******************************************************************************
 * Name:        MGB_MDS_TIMEALIGN_IU                                          *
 * Description: Time alignment information                                    *
 ******************************************************************************/
typedef struct 
{
  tuint  valid_params;
#define MGB_IU_TIME_ALIGN_VALID_DELTA   0x0001
#define MGB_IU_TIME_ALIGN_VALID_TOTAL   0x0002
  tuint  req_delta_alignment;
   /*0.5 msec units +/- offset requested in this message*/
  tuint  current_total_alignment;
   /*0.5 msec units total offset from base alignment point. Currently this    */
   /*will never be negative                                                   */
} TNI_ATTR_PACK MGB_MDS_TIMEALIGN_IU;

/******************************************************************************
 * Name:        MGB_MDS_ERRINFO_IU                                            *
 * Description: Error  information                                            *
 ******************************************************************************/
/* Error Cause Subtypes */
#define MGB_IU_ERROR_INFO_CAUSE_SUBTYPE_RATECTRL_NACKS     \
                          MGB_IU_VIOL_ERRORS_CAUSE_SUBTYPE_RATECTRL_NACKS
#define MGB_IU_ERROR_INFO_CAUSE_SUBTYPE_RATECTRL_TIMEOUT   \
                          MGB_IU_VIOL_ERRORS_CAUSE_SUBTYPE_RATECTRL_TIMEOUT
#define MGB_IU_ERROR_INFO_CAUSE_SUBTYPE_PROCEDURE_EXPIRED  \
                          MGB_IU_VIOL_ERRORS_CAUSE_SUBTYPE_PROCEDURE_EXPIRED
typedef struct 
{
  tuint  valid_params;
#define MGB_IU_ERROR_INFO_VALID_DISTANCE    0x0001
#define MGB_IU_ERROR_INFO_VALID_CAUSE       0x0002
#define MGB_IU_ERROR_INFO_VALID_CAUSE_SUBTYPE  0x0004
#define MGB_IU_ERROR_INFO_VALID_FRAME_NUM      0x0008
  tuint  error_distance;
  tuint  error_cause;
  tuint  error_cause_subtype; /* Only used for DSP to Host error reporting */

  tint   frame_num; 
  /* Currently only used when NO_AUTO_ACK for RATE CONTROL procedure is set */
  /* Specifies frame number of RATECTRL procedure to be NACK-ed */
} TNI_ATTR_PACK MGB_MDS_ERRINFO_IU;

typedef struct {
  tuint  bitmap;                      /* Control Bitmap                      */ 
#define MGB_SEND_IU_SET_INIT         0x0001 /* Set codec profile and generate */
                                            /* Init packet                    */
#define MGB_SEND_IU_GEN_INIT         0x0002 /* Set codec profile and generate */
                                            /* Init packet                    */
#define MGB_SEND_IU_GEN_INIT_ACK     0x0008 /* Generate acknoledgement packet */
#define MGB_SEND_IU_GEN_INIT_NACK    0x0010 /* Generate negack packet         */
#define MGB_SEND_IU_GEN_RATE_CTRL    0x0020 /* Generate rate control packet   */
#define MGB_SEND_IU_GEN_ERROR        0x0040 /* Generate error packet          */
#define MGB_SEND_IU_GEN_RATE_CTRL_ACK  0x0080 /* Generate rate control ack packet  */
#define MGB_SEND_IU_GEN_RATE_CTRL_NACK 0x0100 /* Generate rate control negack packet  */
  tuint  payload_len;                      /* playload length no. of tuint  */
  tuint  reserved;             
  tuint  reserved1;             
  tuint  reserved2;
} TNI_ATTR_PACK MGB_MDS_SEND_PACKET_PROTO_IU;

typedef struct {
  tuint  bitmap;                        /* Control Bitmap                    */ 
#define MGB_RECEIVE_IU_IND_INIT          0x0001 /* Got Init packet            */
#define MGB_RECEIVE_IU_IND_INIT_ACK      0x0002 /* Got Init ack               */
#define MGB_RECEIVE_IU_IND_RATE_CTRL     0x0004 /* Got Rate control           */
#define MGB_RECEIVE_IU_IND_RATE_CTRL_ACK 0x0008 /* Got Rate control ack       */
#define MGB_RECEIVE_IU_IND_TIME_ALIGN    0x0010 /* Got time alignment         */
#define MGB_RECEIVE_IU_IND_ERROR         0x0040 /* Got error indication       */
  tuint  payload_len;                   /* playload length no. of tuint     */
  tuint  reserved;             
  tuint  reserved1;             
  tuint  reserved2;
} TNI_ATTR_PACK MGB_MDS_RECEIVE_PACKET_PROTO_IU;

struct MGB_DMS_IUUP_STAT {
  tuint  nRxCntlPdus;         /* Number of control PDUs received. */
  tuint  nTxCntlPdus;         /* Number of control PDUs transmitted. */
  tuint  maxRxRateIndex;      /* Index of max RX rate RFCI in profile. */
  tuint  maxTxRateIndex;      /* Index of max TX rate RFCI in profile. */
  tuint  negotiatedVersion;   /* Version negotiated in last INIT. */
  tuint  initState;           /* 0 - NONE, 1 - INITing, 2 - READY */
  tuint  nInitNacks;          /* Number of NACKs of INIT procedure. */
  tuint  nRateCntlNacks;      /* Number of NACKs of RATE CNTL procedure. */
  tuint  nTimeAlignNacks;     /* Number of NACKs of TIME ALIGN procedure. */
  tuint  nInitRetries;        /* Number of INIT procedure sent retries. */
  tuint  nRateCntlRetries;    /* Number of RATE CNTL procedure sent retries. */
} TNI_ATTR_PACK;

#endif  /* __IUUPAPI_H__ */

/* nothing past this point */
