#ifndef _MICDSPDP_INCLUDED
#define _MICDSPDP_INCLUDED
/******************************************************************************
 * FILE PURPOSE: MIC-DSP dual-port RAM (DPR) definitions.
 ******************************************************************************
 * FILE NAME:  micdspdp.h 
 *
 * DESCRIPTION: Defines the partition of the MIC-DSP DPR during and after
 *              DSP download.  
 *
 *              NOTE 1: Both DSP and MIC use this file.
 *              NOTE 2: All sizes are in 16-bit words.
 *              NOTE 3: "dprword" must be defined by each the DSP and 
 *                      MIC in a header file named: micdspsh.h
 *
 * REQUIRES:  micdspsh.h - Both DSP and MIC have their own separate file which 
 *                         defines data types shared between MIC and DSP.
 *                         Must define the following data type:
 *                              dprword: an unsigned 16-bit integer
 *
 * TABS: NONE
 *
 * REVISION HISTORY:
 *
 * DATE          AUTHOR        CHANGE DESCRIPTION
 * ====          ======        ==================
 * 05-Apr-1999   ffruth        Created initial version
 * 18-Jun-1999   ffruth        Modified DPR map for modem.  Added Bulk EC delay
 *                             buffers.
 * 22-Nov-1999   ffruth        Added new 7.0 Channel FIFO DPR map definition.
 *
 *
 *
 * Started with:
 *
 * Revision 2.1  1998/12/08 09:28:01  ZORAN
 * Added  clock init. through the bootload. Builds on 2.0.1.3
 * Revision 2.0.1.3  1998/09/25 10:03:17  qsu
 * - For high density test. Option of linear or circular buffering in DPR added.
 * Revision 2.0.1.2  1998/09/22 18:10:12  qsu
 * - For high density test. Memory partition changed in HPI.
 * Revision 2.0.1.1  1998/09/21 08:59:10  qsu
 * - For high density test. New HPI.
 * Revision 2.0  1997/06/14 15:43:31  ORENE
 * Revision 1.3  1997/05/11  07:05:44  wchung
 * Made local and offnet call.
 *
 * Revision 1.4  1997/03/25 15:29:36  RICKF
 * Name change.
 * Revision 1.3  1997/03/24 22:50:04  ffruth
 * - added overlay structures.
 * Revision 1.1  1997/03/20 17:05:59  ffruth
 * Initial revision
 *
 * (C) Copyright 1997, 1998, 1999, 2000 Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
 /* MUST define drpword as unsigned 16-bit integer */
#include <ti/mas/iface/ifeco/mgdefs.h>

/* Dummy Define for a Micro Macro */
#ifndef TNI_ATTR_PACK
#define TNI_ATTR_PACK
#endif


/*****************************************************************************
 * General DPR defintions:
 *****************************************************************************/
#define DPR_SIZE                2048   /* 16 bit words */


/*****************************************************************************
 * Message and Signaling buffer defintions:
 *
 * - Message and signaling buffers between host and DSP are the same size
 * - Message and signaling buffers are identical for ALL DPR maps
 *****************************************************************************/

#define DPR_MSGBUF_SIZE         64
#define DPR_SIGBUF_SIZE         64
#define DPR_MSG_BUFFERS         2  /* one buffer DSP2MIC and one buffer MIC2DSP */
#define DPR_SIG_BUFFERS         2  /* one buffer DSP2MIC and one buffer MIC2DSP */


/*****************************************************************************
 * Download DPR mapping definitions
 *****************************************************************************/

#define DPR_DOWNLOAD_DATA_SIZE  2047  /* in 16-bit words */

typedef struct dprInitialConfig_stc {
  dprword    hpi_sa_msw;                      /* HPI Start Address (MSW) */
  dprword    hpi_sa_lsw;                      /* HPI Start Address (LSW) */

  dprword    hpi_options_word;                /* HPI Options Word */ 
#define DPR_HPI_OPTIONS_FIFO_OWNER_MASK     0x0006
#define DPR_HPI_OPTIONS_FIFO_OWNER_SHIFT    1
#define DPR_HPI_OPTIONS_FIFO_OWNER_ALL_DSP  0 /* DSP owns HPI in both directions */
#define DPR_HPI_OPTIONS_FIFO_OWNER_D2M_MIC  1 /* Micro owns the DSP-to-Micro direction */
#define DPR_HPI_OPTIONS_FIFO_OWNER_M2D_MIC  2 /* Micro owns the Micro-to_DSP direction */
#define DPR_HPI_OPTIONS_FIFO_OWNER_ALL_MIC  3 /* Micro owns both directions */

  dprword    hpi_version_id;                  /* HPI Version ID */
#define DPR_HPI_VERSION_ID_DEFAULT          0x0000
#define DPR_HPI_VERSION_ID_LEGACY           0x0001
#define DPR_HPI_VERSION_ID_HPI_VER_C        0x0100

  dprword    boot_memory_test;                /* Do memory test or not */  
  dprword    boot_clock;					  /* DSP clock multiplier */
  dprword    boot_status;					  /* Boot status field */
} TNI_ATTR_PACK dprInitialConfig_t;

/* HPI 10.0 : Partition of DPR during DSP download at intialization */
typedef struct dprDownloadConfig_stc {
  dprword               data[DPR_DOWNLOAD_DATA_SIZE-6];  /* Download area */
  dprInitialConfig_t    config;
} TNI_ATTR_PACK dprDownloadConfig_t;


/* Partition of DPR during DSP download at intialization */
struct dprDownload_init {
  dprword    data[DPR_DOWNLOAD_DATA_SIZE-2];  /* Download area */
  dprword    boot_memory_test;                /* do memory test or not */  
  dprword    boot_clock;					  /* DSP clock multiplier */
  dprword    boot_status;					  /* Boot status field */
} TNI_ATTR_PACK;

/* Partition of DPR during DSP download */
struct dprDownload_stc {
  dprword    data[DPR_DOWNLOAD_DATA_SIZE];
  dprword    boot_status;					  /* Download area */
} TNI_ATTR_PACK;							  /* boot status field */


/*****************************************************************************
 * Overlay DPR mapping definitions
 *****************************************************************************/

#define DPR_OVERLAY_DATA_SIZE   1920  /* in 16-bit words */

/* Partition of DPR when DSP overlays are downloaded */
struct dprOverlay_stc {
  dprword mic2dspMSG[DPR_MSGBUF_SIZE];    /* MIC to DSP regular message buffer */
  dprword dsp2micMSG[DPR_MSGBUF_SIZE];	  /* DSP to MIC regular message buffer */
  dprword data[DPR_OVERLAY_DATA_SIZE];	  /* Download area */
} TNI_ATTR_PACK;

/* Error check for overlay DPR map */
#if (DPR_OVERLAY_DATA_SIZE + DPR_MSG_BUFFERS*DPR_MSGBUF_SIZE > DPR_SIZE)
# error Invalid DPR_OVERLAY_DATA_SIZE
#endif


/*****************************************************************************
 * Fixed-slot DPR mapping definitions:
 *
 * - consists of a dedicated data buffer for each channel in each direction.
 *****************************************************************************/

/* Size of the voice data buffers (same for all channels and directions) */
#define DPR_VOXBUF_SIZE         192

/* 4 buffers in each direction */
#define DPR_MIC2DSP_VOXBUFFERS  4
#define DPR_DSP2MIC_VOXBUFFERS  4

/* Calculate unused DPR area */
#define DPR_RESERVED_SIZE       (DPR_SIZE - (DPR_MSGBUF_SIZE * DPR_MSG_BUFFERS + \
            DPR_SIGBUF_SIZE * DPR_SIG_BUFFERS + (DPR_MIC2DSP_VOXBUFFERS +        \
            DPR_DSP2MIC_VOXBUFFERS) * DPR_VOXBUF_SIZE))

/* Error checking for Fixed slot DPR map */
#if (DPR_RESERVED_SIZE < 0)
# error Not enough room in DPR
#endif

/* Fixed-slot Data Buffer Structure definition: */
struct dprVoxDataBuf_stc {
  dprword    data[DPR_VOXBUF_SIZE];
} TNI_ATTR_PACK;
typedef struct dprVoxDataBuf_stc dprVoxDataBuf_t;

struct dprLINPayload_stc {
  dprVoxDataBuf_t mic2dspVOX[DPR_MIC2DSP_VOXBUFFERS];
  dprVoxDataBuf_t dsp2micVOX[DPR_DSP2MIC_VOXBUFFERS];
  dprword         reserved[DPR_RESERVED_SIZE];
} TNI_ATTR_PACK;


/*****************************************************************************
 * Shared FIFO DPR mapping definitions:
 *
 * - consists of one FIFO in each direction.
 *****************************************************************************/
#define DPR_VOXBUF_SIZE_FIFO    894	  /* Size of FIFO in DPR for one direction */
#define DPR_VOXBUFFERS_FIFO   	1  	  /* 1 FIFO in each direction */

/* FIFO Voice Data Buffer Structure */
struct dprVoxFIFO_stc {
  dprword in_index;                     /* IN index  */
  dprword out_index;                    /* OUT index */
  dprword data[DPR_VOXBUF_SIZE_FIFO];	/* FIFO data area */
} TNI_ATTR_PACK;

typedef struct dprVoxFIFO_stc dprVoxFIFO_t;

struct dprFIFOPayload_stc {
  dprVoxFIFO_t    mic2dspFIFO;
  dprVoxFIFO_t    dsp2micFIFO;
} TNI_ATTR_PACK;


/*****************************************************************************
 * Channel FIFO mapping definitions:
 *
 * - consists of a dedicated data FIFO and bulk delay line FIFO	for each
 *   channel in each direction
 *****************************************************************************/

#define DPR_MIC2DSP_CFIFO         4    /* 4 Buffers in MIC to DSP direction */
#define DPR_DSP2MIC_CFIFO         4    /* 4 Buffers in DSP to MIC direction */

/* Total size of channel FIFO buffer */
#define DPR_CFIFO_DATBUF_SIZE     224

/* Data buffer FIFO size when bulk delay support is NOT enabled */
#define DPR_CFIFO_DATBUF_SIZE_WOBULKDELAY       DPR_CFIFO_DATBUF_SIZE
#define DPR_CFIFO_DATABUF_HEADER_SIZE           2

/* Data buffer FIFO size when bulk delay support is enabled */
#define DPR_CFIFO_DATBUF_SIZE_WBULKDELAY        96

/* Bulk delay FIFO size when enabled */                                                     
#define DPR_CFIFO_BULKDELAYBUF_SIZE             128
#define DPR_CFIFO_BULKDELAYBUF_HEADER_SIZE        3
#define DPR_CFIFO_BULKDELAYBUF_BODY_SIZE        (DPR_CFIFO_BULKDELAYBUF_SIZE - \
                                              DPR_CFIFO_BULKDELAYBUF_HEADER_SIZE)

#define DPR_CFIFO_BULKDELAY_CONTROL_BIT		0x8000   /* MSB bit in dprword */

/* Size checking -- */
#if ((DPR_CFIFO_DATBUF_SIZE != (DPR_CFIFO_DATBUF_SIZE_WBULKDELAY + \
     DPR_CFIFO_BULKDELAYBUF_SIZE)) || (DPR_CFIFO_DATBUF_SIZE != \
     DPR_CFIFO_DATBUF_SIZE_WOBULKDELAY))
  #error Buffer size error for Channel FIFO
#endif

/* Check available and required DPR size */
#define DPR_CFIFO_RESERVED_SIZE (DPR_SIZE - \
          (DPR_MSGBUF_SIZE * DPR_MSG_BUFFERS + \
           DPR_SIGBUF_SIZE * DPR_SIG_BUFFERS + \
		   DPR_CFIFO_DATBUF_SIZE * (DPR_MIC2DSP_CFIFO + DPR_DSP2MIC_CFIFO)))

#if (DPR_CFIFO_RESERVED_SIZE < 0)
  #error Not enough room in DPR
#endif

/* Data buffer layout -- when bulk delay support is enabled, available
   FIFO size is reduced for data exchange */
struct dprCFIFODataBuffer_stc {
  dprword in_index;                         /* IN index  */
  dprword out_index;                        /* OUT index */
  dprword data[DPR_CFIFO_DATBUF_SIZE_WOBULKDELAY-DPR_CFIFO_DATABUF_HEADER_SIZE];
} TNI_ATTR_PACK;
typedef struct dprCFIFODataBuffer_stc dprCFIFODataBuffer_t;

/* Bulk delay buffer layout -- when used overlays bottom of data buffer area */
struct dprCFIFOBulkDelayBuffer_stc {
  dprword reserved[DPR_CFIFO_DATBUF_SIZE_WBULKDELAY];  /* Overlay with data buffer */
  dprword in_index;                            
  dprword out_index;                               
  dprword control;							      
  dprword sampledata[DPR_CFIFO_BULKDELAYBUF_BODY_SIZE];
} TNI_ATTR_PACK;
typedef struct dprCFIFOBulkDelayBuffer_stc dprCFIFOBulkDelayBuffer_t;

/* Channel FIFO buffer description */
union dprCFIFOBuffer_u {
     dprCFIFODataBuffer_t      databuffer;
     dprCFIFOBulkDelayBuffer_t bulkdelay;
  };

/* Channel FIFO payload map */
struct dprCFIFOPayload_stc {
  union dprCFIFOBuffer_u mic2dspBuffer[DPR_MIC2DSP_CFIFO];
  union dprCFIFOBuffer_u dsp2micBuffer[DPR_DSP2MIC_CFIFO];
#if (DPR_CFIFO_RESERVED_SIZE > 0)
  dprword reserved[DPR_CFIFO_RESERVED_SIZE];
#endif
} TNI_ATTR_PACK;
typedef struct dprCFIFOPayload_stc dprCFIFOPayload_t;


/*****************************************************************************
 * Data Payload union definition for normal DPR mapping
 *****************************************************************************/

union dprPayload_u {
  struct dprFIFOPayload_stc  FIFO;		 /* Shared FIFO mapping */
  struct dprLINPayload_stc   LIN;		 /* Fixed-slot mapping */
  struct dprCFIFOPayload_stc CFIFO;		 /* Channel FIFO mapping */
};

/*****************************************************************************
 * Normal DPR mapping
 *****************************************************************************/

struct dprNormal_stc {
  dprword         mic2dspMSG[DPR_MSGBUF_SIZE];
  dprword         dsp2micMSG[DPR_MSGBUF_SIZE];
  dprword         mic2dspSIG[DPR_SIGBUF_SIZE];
  dprword         dsp2micSIG[DPR_SIGBUF_SIZE];
  union dprPayload_u Data;
} TNI_ATTR_PACK;



/*****************   HPI 10.0: Implementation begins here   *****************/

/*****************************************************************************
 * HPI Descriptor
 *****************************************************************************/

/* See download structure definition for Version ID values */

/* HPI Alignment Options */
#define DPR_HPI_ALIGNMENT_TYPE_MASK     0x00FF
#define DPR_HPI_ALIGNMENT_8_BIT         0x0001
#define DPR_HPI_ALIGNMENT_16_BIT        0x0002
#define DPR_HPI_ALIGNMENT_32_BIT        0x0004
#define DPR_HPI_ALIGNMENT_OPTION_MASK   0xFF00
#define DPR_HPI_ALIGNMENT_INDEX         0x0100

typedef struct dprHPIDesc_stc {
    dprword             hpi_size;       /* Size of HPI descriptor in words */
    dprword             version_id;     /* Version ID */
    dprword             dsp2mic_fifos;  /* Number of DSP to Micro FIFOs */
    dprword             mic2dsp_fifos;  /* Number of Micro to DSP FIFOs */
    dprword             acs_sa_msw;     /* ACS Start Address MSW */
    dprword             acs_sa_lsw;     /* ACS Start Address LSW */
    dprword             alignment;      /* Alignment mode */
} TNI_ATTR_PACK dprHPIDesc_t;


/*****************************************************************************
 * FIFO Descriptor
 *****************************************************************************/

/* FIFO Types */
#define DPR_FIFO_TYPE_GENERAL                       0
#define DPR_FIFO_TYPE_FLOW_CONTROL                  1
#define DPR_FIFO_TYPE_CHANNELIZED                   2
#define DPR_FIFO_TYPE_MAILBOX                       3
#define DPR_FIFO_TYPE_INDEX                         4

/* Packet Types */
#define DPR_PKT_HIGH_PRIORITY_MSG                   0
#define DPR_PKT_NORMAL_PRIORITY_MSG                 1
#define DPR_PKT_LOW_PRIORITY_MSG                    2
#define DPR_PKT_ANNOUNCEMENTS                       3
#define DPR_PKT_BULK_DELAY                          8
#define DPR_PKT_DEBUG                               9
#define DPR_PKT_FILL_PKT                            10
#define DPR_PKT_FEC                                 11
#define DPR_PKT_VFD                                 15

/* Packet Classes */
#define DPR_PKT_CLASS_HIGH_PRIORITY_MSG             (1 << DPR_PKT_HIGH_PRIORITY_MSG)
#define DPR_PKT_CLASS_NORMAL_PRIORITY_MSG           (1 << DPR_PKT_NORMAL_PRIORITY_MSG)
#define DPR_PKT_CLASS_LOW_PRIORITY_MSG              (1 << DPR_PKT_LOW_PRIORITY_MSG)
#define DPR_PKT_CLASS_ANNOUNCEMENTS                 (1 << DPR_PKT_ANNOUNCEMENTS)
#define DPR_PKT_CLASS_BULK_DELAY                    (1 << DPR_PKT_BULK_DELAY)
#define DPR_PKT_CLASS_DEBUG                         (1 << DPR_PKT_DEBUG)
#define DPR_PKT_CLASS_FEC                           (1 << DPR_PKT_FEC)
#define DPR_PKT_CLASS_VFD                           (1 << DPR_PKT_VFD)
#define DPR_PKT_CLASS_MAX                           15

/* FIFO Options */
#define DPR_FIFO_OPTIONS_NONE                       0x0000
#define DPR_FIFO_OPTIONS_CHECKSUM                   0x0001
#define DPR_FIFO_OPTIONS_FILL_PKT                   0x0002

/* Number of config fields per FIFO Type */
#define DPR_FIFO_NUM_CONFIG_FIELDS_GENERAL          0
#define DPR_FIFO_NUM_CONFIG_FIELDS_FLOW_CONTROL     1
#define DPR_FIFO_NUM_CONFIG_FIELDS_CHANNELIZED      1
#define DPR_FIFO_NUM_CONFIG_FIELDS_MAILBOX          0
#define DPR_FIFO_NUM_CONFIG_FIELDS_INDEX            0

typedef struct dprFIFODesc_stc {
    dprword             fifo_type;      /* FIFO Type */
    dprword             pkt_class;      /* Allowed Packet Classes */
    dprword             sa_msw;         /* FIFO Start Address MSW */
    dprword             sa_lsw;         /* FIFO Start Address LSW */
    dprword             fifo_size;      /* FIFO Size in words */
    dprword             fifo_options;   /* FIFO Option bits */
} TNI_ATTR_PACK dprFIFODesc_t;

/*****************************************************************************
 * ACS (Active Communication Section)
 *****************************************************************************/

#define DPR_ACS_FLOW_CONTROL_SIZE   2   /* The required number of words for
                                         * bit-map representation of all channels 
                                         */
#define DPR_NUM_ACS_FIELDS                  2
#define DPR_NUM_ACS_FIELDS_GENERAL          DPR_NUM_ACS_FIELDS
#define DPR_NUM_ACS_FIELDS_CHANNELIZED      DPR_NUM_ACS_FIELDS
#define DPR_NUM_ACS_FIELDS_FLOW_CONTROL     DPR_NUM_ACS_FIELDS + \
                                            DPR_ACS_FLOW_CONTROL_SIZE
#define DPR_NUM_ACS_FIELDS_MAILBOX          0
#define DPR_NUM_ACS_FIELDS_INDEX            0

typedef struct dprACS_stc {
    dprword             heart_beat;     /* Version ID */
    void*               data;
} TNI_ATTR_PACK dprACS_t;

/*****************************************************************************
 * Packet Wrapper for all packet types
 *****************************************************************************/

typedef struct dprPktWrap_stc {
    dprword             pkt_length;         /* Packet Length in bytes */
    dprword             pkt_type_chnum;     /* Packet Type and Logical Channel ID */
} TNI_ATTR_PACK dprPktWrap_t;

/*****************************************************************************
 * DPR mapping data structure definition
 *****************************************************************************/

union dpr_u {
  struct dprDownload_init       Download_init;
  struct dprDownload_stc        Download;
  struct dprNormal_stc          Normal;
  struct dprDownloadConfig_stc  Download_config;
};

#endif /* _MICDSPDP_INCLUDED */
/* nothing past this point */

