/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/



#ifndef _MSU_CFG_H
#define _MSU_CFG_H
/******************************************************************************
 * FILE PURPOSE: Generic MSU configuration and message file.
 ******************************************************************************
 * FILE NAME:   msucfg.h
 *
 * DESCRIPTION: 1. Contains Micro-DSP messages
 *              2. Contains #defines related with DSP and Micro
 *
 *              NOTE: This file may be used by both Micro and DSP software.
 *              NOTE: These messages are on per channel basis.
 *
******************************************************************************/

#include <ti/mas/iface/ifeco/mgdefs.h>

/* Dummy Define for a Micro Macro */
#ifndef TNI_ATTR_PACK
#define TNI_ATTR_PACK
#endif

#define MSU_MAX_KEY_SIZE_IN_TUINT 8
#define MSU_MAX_SALT_SIZE_IN_TUINT 7

/*****************************************************************************/
/* SECURITY PROFILE TABLES                                                   */
/*****************************************************************************/
/* Note: All sizes are in byte length.*/
/* Assumption is that the sizes will always a multiple of 8 */
typedef struct {
  /* master key =8 msb / master salt =8 lsb */
  tuint  master_key_salt_size;
#define MSU_SRTP_MASTER_KEY_SIZE_MASK   0xFF00
#define MSU_SRTP_MASTER_KEY_SIZE_SHIFT  8
#define MSU_SRTP_MASTER_SALT_SIZE_MASK  0x00FF
#define MSU_SRTP_MASTER_SALT_SIZE_SHIFT 0
  /* session key =8 msb / session salt =8 lsb */
  tuint  session_key_salt_size;
#define MSU_SRTP_SESSION_KEY_SIZE_MASK   0xFF00
#define MSU_SRTP_SESSION_KEY_SIZE_SHIFT  8
#define MSU_SRTP_SESSION_SALT_SIZE_MASK  0x00FF
#define MSU_SRTP_SESSION_SALT_SIZE_SHIFT 0
  /* session mac key =8 msb / authentication tag size =8 lsb */
  tuint  session_mac_key_tag_size;
#define MSU_SRTP_SESSION_MAC_KEY_SIZE_MASK  0xFF00
#define MSU_SRTP_SESSION_MAC_KEY_SIZE_SHIFT 8
#define MSU_SRTP_SESSION_MAC_TAG_SIZE_MASK  0x00FF
#define MSU_SRTP_SESSION_MAC_TAG_SIZE_SHIFT 0
} TNI_ATTR_PACK msu_srtp_key;

#define SRTP_MASTER_KEY_SIZE(a) (((((msu_srtp_key *)a)->master_key_salt_size) & \
                   MSU_SRTP_MASTER_KEY_SIZE_MASK) >> MSU_SRTP_MASTER_KEY_SIZE_SHIFT)
#define SRTP_MASTER_SALT_SIZE(a) (((((msu_srtp_key *)a)->master_key_salt_size) &  \
                   MSU_SRTP_MASTER_SALT_SIZE_MASK) >> MSU_SRTP_MASTER_SALT_SIZE_SHIFT)
#define SRTP_SESSION_KEY_SIZE(a) (((((msu_srtp_key *)a)->session_key_salt_size) &  \
                   MSU_SRTP_SESSION_KEY_SIZE_MASK) >> MSU_SRTP_SESSION_KEY_SIZE_SHIFT)
#define SRTP_SESSION_SALT_SIZE(a) (((((msu_srtp_key *)a)->session_key_salt_size) & \
                   MSU_SRTP_SESSION_SALT_SIZE_MASK) >> MSU_SRTP_SESSION_SALT_SIZE_SHIFT)
#define SRTP_SESSION_MAC_KEY_SIZE(a) (((((msu_srtp_key *)a)->session_mac_key_tag_size) & \
                   MSU_SRTP_SESSION_MAC_KEY_SIZE_MASK) >> MSU_SRTP_SESSION_MAC_KEY_SIZE_SHIFT)
#define SRTP_SESSION_MAC_TAG_SIZE(a) (((((msu_srtp_key *)a)->session_mac_key_tag_size) & \
                   MSU_SRTP_SESSION_MAC_TAG_SIZE_MASK) >> MSU_SRTP_SESSION_MAC_TAG_SIZE_SHIFT)
typedef struct {
  /* session key =7 msb / session mac key size =9 lsb */
  tuint  session_key_mac_key_size;
#define MSU_PKTCBL_SESSION_KEY_SIZE_MASK   0xFE00
#define MSU_PKTCBL_SESSION_KEY_SIZE_SHIFT  9
#define MSU_PKTCBL_SESSION_MAC_KEY_SIZE_MASK  0x01FF
#define MSU_PKTCBL_SESSION_MAC_KEY_SIZE_SHIFT 0
  /* session iv =8 msb / authentication tag size =8 lsb */
  tuint  iv_mac_tag_size;
#define MSU_PKTCBL_IV_SIZE_MASK        0xFF00
#define MSU_PKTCBL_IV_SIZE_SHIFT       8
#define MSU_PKTCBL_AUTH_TAG_SIZE_MASK  0x00FF
#define MSU_PKTCBL_AUTH_TAG_SIZE_SHIFT 0
} TNI_ATTR_PACK msu_pktcbl_key;

#define PKT_KEY_SIZE(a) (((((msu_pktcbl_key *)a)->session_key_mac_key_size) & \
                   MSU_PKTCBL_SESSION_KEY_SIZE_MASK) >> MSU_PKTCBL_SESSION_KEY_SIZE_SHIFT)
#define PKT_MAC_KEY_SIZE(a) (((((msu_pktcbl_key *)a)->session_key_mac_key_size) & \
                   MSU_PKTCBL_SESSION_MAC_KEY_SIZE_MASK) >> MSU_PKTCBL_SESSION_MAC_KEY_SIZE_SHIFT)
#define PKT_IV_SIZE(a) (((((msu_pktcbl_key *)a)->iv_mac_tag_size) & \
                   MSU_PKTCBL_IV_SIZE_MASK) >> MSU_PKTCBL_IV_SIZE_SHIFT)
#define PKT_AUTH_TAG_SIZE(a) (((((msu_pktcbl_key *)a)->iv_mac_tag_size) & \
                   MSU_PKTCBL_AUTH_TAG_SIZE_MASK) >> MSU_PKTCBL_AUTH_TAG_SIZE_SHIFT)

#define  MSU_AES_NULL     0
#define  MSU_AES_CTR      1
#define  MSU_AES_f8       2
#define  MSU_AES_CBC_RBT  3

#define  MSU_MAC_NULL     0
#define  MSU_HMAC_SHA1    1
#define  MSU_MMH          2
#define  MSU_HMAC_MD5     3

typedef struct {
  /* Cipher Mode 8 msb / Mac 8 lsb */
  tuint  cipher_mode_mac;
#define MSU_CIPHER_MODE_MASK   0xFF00
#define MSU_CIPHER_MODE_SHIFT  8
#define MSU_MAC_TYPE_MASK      0x00FF
#define MSU_MAC_TYPE_SHIFT     0
} TNI_ATTR_PACK msu_cipher_mac;

/*****************************************************************************/
/* SECURITY PROFILE                                                          */
/*****************************************************************************/
struct MGB_MDS_SECURITY_PROF   /* both TX and RX will have same profile type */
{
  tuint prof_type_bitmap;
#define MGB_SRTP_PROF                 0x0001 /*   Secure RTP */
#define MGB_SRTCP_PROF                0x0002 /*   Secure RTCP */
#define MGB_PKTCBL_RTP_PROF           0x0004 /*   PacketCable RTP*/
#define MGB_PKTCBL_RTCP_PROF          0x0008 /*   PacketCable RTCP*/
} TNI_ATTR_PACK;

/*****************************************************************************/
/* SECURITY PROFILE CONTROL                                                  */
/*****************************************************************************/
typedef struct 
{
  tuint  cipher_mac_index;
  tuint  key_index;
} TNI_ATTR_PACK MGB_MDS_CIPHER_MAC_KEY_CTRL;

typedef struct 
{
  tuint  bitmap;
#define MGB_MDS_TX_RX_CTRL            0x0001 /* TX/RX Enable Disable */
#define MGB_MDS_TX_PROF_CTRL          0x0002 /* TX cipher mode ctrl */
#define MGB_MDS_RX_PROF_CTRL          0x0004 /* RX cipher mode ctrl */

  tuint  tx_rx_bitmap;
#define MGB_MSU_TX_RX_CLOSED          0x0000 /*   TX RX Disable */
#define MGB_MSU_TX_OPEN               0x0001 /*   TX Enable */
#define MGB_MSU_RX_OPEN               0x0002 /*   RX Enable */
#define MGB_MSU_TX_RX_MASK            0x0003 /*   mask bits */
#define MGB_MSU_TX_RX_RESET           0x0004 /* TX/RX Security Instance/State is  */ 
                                             /* CLEANED: everything is cleared */
                                             /* except the Profile info */

  MGB_MDS_CIPHER_MAC_KEY_CTRL  tx_ctrl;
  MGB_MDS_CIPHER_MAC_KEY_CTRL  rx_ctrl;
} TNI_ATTR_PACK MGB_MDS_TX_RX_PROF_CTRL;

/* MSU Control Message Structure */
struct MGB_MDS_SECURITY_CTRL 
{
  tuint bitmap;
#define MGB_MDS_MSU_RTP_CTRL          0x0001 /*   RTP Ctrl */
#define MGB_MDS_MSU_RTCP_CTRL         0x0002 /*   RTCP Ctrl */

  MGB_MDS_TX_RX_PROF_CTRL   rtp_ctrl;
  MGB_MDS_TX_RX_PROF_CTRL   rtcp_ctrl;
} TNI_ATTR_PACK ;

/*****************************************************************************/
/* SECURITY KEY CTRL & SETUP                                                 */
/*****************************************************************************/

/* Secure RTP/RTCP Key control */
typedef struct 
{
  tuint  mki_length; /* in bytes */
  tuint  mki_value;  /* With tuint mki_value, mki_length can be at most 2 */
} TNI_ATTR_PACK MGB_MDS_MSU_MKI;

typedef struct 
{
  tuint  from[3];
  tuint  to[3];
} TNI_ATTR_PACK MGB_MDS_MSU_FROM_TO;

typedef struct MGB_MDS_MSU_SRTP_KEY
{
  tuint bitmap;
#define   MGB_MDS_MASTER_KEY       0x0001
#define   MGB_MDS_MASTER_SALT      0x0002
#define   MGB_MDS_KEY_DERIVE_RATE  0x0004 
#define   MGB_MDS_MAX_KEY_LIFETIME 0x0008
#define   MGB_MDS_ROC              0x0010
#define   MGB_MDS_MKI              0x0020 
#define   MGB_MDS_FROM_TO          0x0040
  /* Key sizes should be derived from the key profile */
  tuint  master_key[MSU_MAX_KEY_SIZE_IN_TUINT];
  tuint  master_salt[MSU_MAX_SALT_SIZE_IN_TUINT];

  tuint  key_derivation_rate; /* n of 2^n format (where 0 <= n <= 24)
                                  if not set then use default value, 
                                  i.e no key derivation */
#define   MGB_MDS_MAX_KEY_DERIVATION_RATE       24

  tuint  max_key_lifetime[3]; /* for SRTP it will be < 2^48 (0 - (2^48)-1) 
                                  and for SRTCP < 2^31 (0 - (2^31) -1)*/
  tuint  roc_msw; /*  When the party is joining an ongoing call then must be set */
  tuint  roc_lsw; /*  When the party is joining an ongoing call then must be set */

  union
  {
    MGB_MDS_MSU_FROM_TO from_to;
    MGB_MDS_MSU_MKI     mki;
  } TNI_ATTR_PACK u;  
} TNI_ATTR_PACK MGB_MDS_MSU_KEY_SRTP;

/* PacketCable Key control */
typedef struct  
{
  tuint ctrl_type;
#define   MGB_MDS_KEY_DEF_AES      0x0001
#define   MGB_MDS_KEY_DEF_MAC      0x0002
#define   MGB_MDS_KEY_DEF_IV       0x0003
  
  tuint   key_total_size;
  tuint   current_key_size;
  tuint   iter_num;
#define MGB_MDS_MSU_PKTCBL_KEY_SIZE_PER_MSG     53      /* Max key size per msg in words.*/
  tuint   keybuf[MGB_MDS_MSU_PKTCBL_KEY_SIZE_PER_MSG];
} TNI_ATTR_PACK MGB_MDS_MSU_KEY_PKTCBL;

/* Choose Secure RTP/RTCP or PacketCable RTP/RTCP Security */
typedef struct
{
  tuint   tx_rx;
#define   MGB_MSU_TX_KEY            0x0001
#define   MGB_MSU_RX_KEY            0x0002

  union
  {
    MGB_MDS_MSU_KEY_SRTP    srtp_key;
    MGB_MDS_MSU_KEY_PKTCBL  pktcbl_key;
  } TNI_ATTR_PACK u;
} TNI_ATTR_PACK MGB_MDS_MSU_KEY;

struct MGB_MDS_SECURITY_KEY_CTRL 
{
  tuint    pkt_type;
#define   MGB_MSU_TYPE_RTP              1
#define   MGB_MSU_TYPE_RTCP             2

  MGB_MDS_MSU_KEY key;
} TNI_ATTR_PACK;

/*****************************************************************************/
#define MGB_MSU_CONTROL   0
#define MGB_MSU_KEY_SETUP 1

#define MGB_MSU_CTRL_RES1   256 /* Reserved for internal use */
#define MGB_MSU_CTRL_RES2   257 /* Reserved for internal use */


/*****************************************************************************/
/* SECURITY STATS for RTP/RTCP                                               */
/*****************************************************************************/
typedef struct {
  tuint   replay;        /* Number of Replays */
  tuint   auth_fail;     /* Number of Authentication failures */
  /* Present TX Roll-Over Counter (32 bits) */
  tuint   tx_roc_msw;    /* Upper 16 bits */
  tuint   tx_roc_lsw;    /* Lower 16 bits */
  /* Present RX Roll-Over Counter (32 bits) */
  tuint   rx_roc_msw;    /* Upper 16 bits */
  tuint   rx_roc_lsw;    /* Lower 16 bits */
  tuint   tx_key;        /* Number of times keying happened in Tx */
  tuint   rx_key;        /* Number of times keying happened in Rx */
  /* Total Number of Packets encrypted with this key (32 bits) */
  tuint   pkt_enc_msw;   /* Upper 16 bits */
  tuint   pkt_enc_lsw;   /* Lower 16 bits */
  /* Total Number of Packets decrypted with this key (32 bits) */
  tuint   pkt_dec_msw;   /* Upper 16 bits */
  tuint   pkt_dec_lsw;   /* Lower 16 bits */
} TNI_ATTR_PACK MGB_DMS_SECURITY_STATS_SRTP;

typedef struct {
  tuint   replay;        /* Number of Replays */
  tuint   auth_fail;     /* Number of Authentication failures */
  tuint   tx_nwrap;      /* Number of Times wraparound of TS in TX*/
  tuint   rx_nwrap;      /* Number of Times wraparound of TS in RX*/
  /* Total Number of Packets encrypted with this key (32 bits) */
  tuint   pkt_enc_msw;   /* Upper 16 bits */
  tuint   pkt_enc_lsw;   /* Lower 16 bits */
  /* Total Number of Packets decrypted with this key (32 bits) */
  tuint   pkt_dec_msw;   /* Upper 16 bits */
  tuint   pkt_dec_lsw;   /* Lower 16 bits */
} TNI_ATTR_PACK MGB_DMS_SECURITY_STATS_PKTCBL;

typedef union
{
  MGB_DMS_SECURITY_STATS_SRTP    srtp_stats;
  MGB_DMS_SECURITY_STATS_PKTCBL  pktcbl_stats;
} TNI_ATTR_PACK MGB_DMS_MSU_STATS;

struct MGB_DMS_SECURITY_GET_STATS {
  tuint pkt_type;  /* not a bitmap */
#define   MGB_DMS_RTP              0x0001
#define   MGB_DMS_RTCP             0x0002
  
  MGB_DMS_MSU_STATS  stats;
} TNI_ATTR_PACK;

/*****************************************************************************/
/* SECURITY KEY REQUEST  <message from MSU to Host>                          */
/*****************************************************************************/
/* Description: This structure is used to tell the upper layer that          */
/*              the security key is expiring, or it needs Master Key in MKI  */
/*****************************************************************************/

/* Defines when the key expiry notice will be sent by MSU before the key expires */
/* NOTE: these values are just an example, may be changed later */
#define RTP_KEY_LIFE_MARGIN       10
#define RTCP_KEY_LIFE_MARGIN      1

typedef struct {
  tuint    dir;
#define MGB_DM_KEY_LIFE_LEFT_TX         MGB_MSU_TX_KEY
#define MGB_DM_KEY_LIFE_LEFT_RX         MGB_MSU_RX_KEY

  tuint  key_life_left;      /* # of Packets left before master key expires */
} TNI_ATTR_PACK MGB_DMS_KEY_LIFE_LEFT;

struct MGB_DMS_SECURITY_KEY_REQUEST {
  tuint                  pkt_type;  /* not a bitmap */

  tuint                  select;
#define MGB_DM_KEY_REQUEST_SELECT_KEY_LIFE      1
#define MGB_DM_KEY_REQUEST_SELECT_MKI_INDEX     2

  union
  {
      MGB_DMS_KEY_LIFE_LEFT   key;
      tuint                  MKI_index;
  } TNI_ATTR_PACK u;
} TNI_ATTR_PACK;

/*****************************************************************************/
/* List of error indication messages used by SECURITY package:               */
/*****************************************************************************/
/* Invalid security profile */ 
#define MGB_ERR_INVD_SECURITY_PROF               261
/* Invalid security control message */
#define MGB_ERR_INVD_SECURITY_CTRL               262
/* Invalid security key control message */
#define MGB_ERR_INVD_SECURITY_KEY_CTRL           263
/* RTP security module does not exist */
#define MGB_ERR_INVD_SECURITY_RTP_NULL           264
/* RTCP security module does not exist */
#define MGB_ERR_INVD_SECURITY_RTCP_NULL          265
/* Invalid call to KEY REQUEST report */ 
#define MGB_ERR_INVD_SECURITY_STATS_BAD_MSG_TYPE 266

/* Below Errors are reported by SIU while setting DSP configuration
 * for MSU instance
 */
/* MGB_MD_SECURITY_CTRL is not received for the channel */ 
#define MGB_ERR_MSU_SET_DSP_CFG_INVD_CHAN        276
/* No MSU Instance available for setting the states */ 
#define MGB_ERR_MSU_SET_DSP_CFG_NULL_INST        277
/* Invalid message for setting the DSP state information */ 
#define MGB_ERR_MSU_SET_DSP_CFG_INVD_CTRL        278

#endif /* _MSU_CFG_H */
/* nothing past this point */
