#ifndef __NETAPI_H__
#define __NETAPI_H__
/******************************************************************************
 * FILE PURPOSE: Structures shared between DSP and host for PVP, RCU, ACU, NEU
 ******************************************************************************
 * FILE NAME:   net.h
 *
 * DESCRIPTION: Defines shared structures for the network modules of the DSP.
 *              This structures are used by the DSP and HOST, and also
 *              between the DSP modules.
 *
 * TABS: NONE
 *
 * $Id: $
 *
 * REVISION HISTORY:
 *
 * $Log: $
 *
 * (C) Copyright 2002 Texas Instruments Incorporated. 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#include <ti/mas/iface/ifeco/mgdefs.h>
#include <ti/mas/iface/ifeco/micdspdp.h>
#include <ti/mas/iface/ifeco/iuupapi.h>
#include <ti/mas/iface/ifeco/tfoapi.h>

#include <ti/mas/iface/ifpkt/ifpkt.h>
#include <ti/mas/iface/ifneu/ifneu.h>
#include <ti/mas/iface/ifrcu/ifrcu.h>

/* Dummy Define for a Micro Macro */
#ifndef TNI_ATTR_PACK
#define TNI_ATTR_PACK
#endif

/***********************************************************************
 * Start of data structures definition for ENCAP_CONFIG message        *
 ***********************************************************************/
/***********************************************************************
 * Name: MGB_MD_ENCAP_RTP                                              *
 * Description: Definition of RTP encapsulation parameters             *
 *       Other relevant parameters are specified in via the profiles   *
 *       or via parameters in the MGB_MD_ENCODE structure              *
 *       Peer-to-peer packets have their own configuration for some    *
 *       some of the parameters                                        *
 * There are two bit fields per direction: change bit field and actual *
 * value of bit field.                                                 *
 * Note: Same functionality as in <R9.2 Protocol Modify                *
 ***********************************************************************/
struct MGB_MDS_ENCAP_RTP {
#define MGB_ENCAP_CONFIG_VALID_RTP_TX_SSRC             0x0002
#define MGB_ENCAP_CONFIG_VALID_RTP_TX_TIMESTAMP        0x0004
#define MGB_ENCAP_CONFIG_VALID_RTP_RX_SSRC             0x0008
#define MGB_ENCAP_CONFIG_VALID_RTP_RX_TIMESTAMP        0x0010
#define MGB_ENCAP_CONFIG_VALID_RTP_RX_SSRC_CTRL        0x0020
#define MGB_ENCAP_CONFIG_VALID_RTP_TX_FIRST_OCTET      0x0040
#define MGB_ENCAP_CONFIG_VALID_RTP_TX_SEQ_NUMBER       0x0080
  tuint valid_params;
  /* Tx synchronization source (32 bits) */
  tuint  tx_ssrc_msw;                          /* Upper 16 bits */
  tuint  tx_ssrc_lsw;                          /* Lower 16 bits */
  /* Tx initial timestamp (32 bits) */
  tuint  tx_start_timestamp_msw;               /* Upper 16 bits */
  tuint  tx_start_timestamp_lsw;               /* Lower 16 bits */
  /* Tx first octet (8 bits) */
  tuint  tx_first_octet;                       /* Low 8 bits used */
  /* Rx synchronization source (32 bits) */
  tuint  rx_ssrc_msw;                          /* Upper 16 bits */
  tuint  rx_ssrc_lsw;                          /* Lower 16 bits */
  /* Rx initial timestamp (32 bits) */
  tuint  rx_start_timestamp_msw;               /* Upper 16 bits */
  tuint  rx_start_timestamp_lsw;               /* Lower 16 bits */
  /* Type of action if ssrc from the received path is violated:  *
   * 1 - accepts all packets                                     *
   * 0 - create packet violation                                 */
  tuint  rx_ssrc_ctrl;
  tuint  tx_start_seq_number;                  /* Start SEQ number*/  
} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MD_ENCAP_AAL2                                             *
 * Description: Definition of AAL2 encapsulation parameters            *
 * Note: Already supported in PROTOCOL_MODIFY                          *
 ***********************************************************************/
struct MGB_MDS_ENCAP_AAL2 {
#define MGB_ENCAP_CONFIG_VALID_AAL2_CID                0x0002
   tuint valid_params;
   tuint cid;          /* AAL2 channel ID */
} TNI_ATTR_PACK;



/* FIXME: <R9.2 legacy */
#define MGB_AAL2_MODIFY_ALL            0x0002

/***********************************************************************
 * Name: MGB_MD_ENCAP_PROTO_NULL                                       *
 * Description: Definition of PROTO_NULL encapsulation parameters      *
 * Note: Placeholder                                                   *
 ***********************************************************************/
struct MGB_MDS_ENCAP_PROTO_NULL {
   tuint dummy;        /* no parameters yet specified - placeholder */
} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MD_ENCAP_AAL1                                             *
 * Description: Support for reconfigurable AAL1 VIF                    *
 * DIM ACTION PROPOSAL: Convert VIF size to bits                       *
 ***********************************************************************/
struct MGB_MDS_ENCAP_AAL1 {
#define MGB_ENCAP_CONFIG_VALID_AAL1_PARTIAL_FILL       0x0002
#define MGB_ENCAP_CONFIG_VALID_AAL1_NET_CODING_TYPE    0x0004
   tuint valid_params;
   tuint partial_fill; /* Partiall fill of ATM cell in bytes: 40, 44, 47 */
   tuint net_coding_type; /* Network side companding: G711 a-law & mu-law */
} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MD_ENCAP_IU_RTP_PARAMS                                    *
 * Description: Iu RTP encap parameters.                               *
 ***********************************************************************/
struct MGB_MDS_ENCAP_IU_RTP_PARAMS
{
#define MGB_ENCAP_IU_RTP_PARAMS_VALID_PAYLOAD_TYPE      0x0001
#define MGB_ENCAP_IU_RTP_PARAMS_VALID_RTP               0x0002
  tuint valid_params;
  tuint tx_payload_type;
  tuint rx_payload_type;
  struct MGB_MDS_ENCAP_RTP rtp;
} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MD_ENCAP_IU                                               *
 * Description: Normal Iu parameters plus RTP encap parameters.        *
 ***********************************************************************/
struct MGB_MDS_ENCAP_IU {
#define MGB_ENCAP_IU_TRANSPORT_MODE_ATM     0
#define MGB_ENCAP_IU_TRANSPORT_MODE_RTP     1
  tuint transport_mode;
  struct MGB_MDS_ENCAP_IU_PARAMS      iu_params;
  struct MGB_MDS_ENCAP_IU_RTP_PARAMS  rtp_params;   /* only for RTP mode */
} TNI_ATTR_PACK;

/***************** MGB_MD_VIDEO_TFTP_CFG ******************************************/
/*******************************************************************************
 * Description: MGB_MD_VIDEO_TFTP_CFG message definition. 
 ******************************************************************************/
struct MGB_MDS_ENCAP_TFTP {
  tuint             valid_params;
#define     MGB_MDS_TFTP_CFG_VALID_FILE_NAME       0x0001
#define     MGB_MDS_TFTP_CFG_VALID_ACTION          0x0002
  tuint             action;
#define     MGB_MDS_TFTP_CFG_ACTION_START               1
#define     MGB_MDS_TFTP_CFG_ACTION_STOP                0

  tuint             file_name[64];
} TNI_ATTR_PACK;

/***********************************************************************
 * Name: MGB_MD_ENCAP_PARAMS                                           *
 * Description: Simple union of each encapsulation parameters          *
 ***********************************************************************/
union MGB_MDS_ENCAP_PARAMS {
   struct MGB_MDS_ENCAP_RTP  rtp;
   struct MGB_MDS_ENCAP_AAL2 aal2;
   struct MGB_MDS_ENCAP_PROTO_NULL proto_null;
   struct MGB_MDS_ENCAP_AAL1 aal1;
   struct MGB_MDS_ENCAP_IU   iu;
   struct MGB_MDS_ENCAP_TFTP tftp;
} TNI_ATTR_PACK;
/***********************************************************************
 * Name: MGB_MD_ENCAP_CONFIG                                           *
 * Description: Definition of encapsulation type and its initial       *
 *              configuration                                          *
 *    This functionality mostly exists in <R9.2 PROTOCOL_MODIFY        *
 *    message. SID retransmission needs to be added. In <R9.2          *
 *    software this parameter was passed in OPEN_CHANNEL.              *
 *    Also, an enhancement of AAL1 implementation is needed in order   *
 *    to support reconfigurable VIF size.                              *
 * DIM ACTION PROPOSAL:                                                *
 *    Pass parameters received from DIM client.                        *
 ***********************************************************************/
struct MGB_MDS_ENCAP_CONFIG {
#define MGB_ENCAP_CONFIG_VALID_ENCAPSULATION           0x0001
  /* KJ: Is this valid encap really needed? Why not always set the encap? */
   tuint valid_params;
/* Encapsulation protocols. */
#define MGB_PT_NONE   99
#define MGB_PT_RTP    3 
#define MGB_PT_AAL2   4 
#define MGB_PT_AAL1   6     
#define MGB_PT_IU     7 
#define MGB_PT_TFTP   8

/* Note other 0, 1, 2, and 5 are reserved values */
   tuint encapsulation;
   union MGB_MDS_ENCAP_PARAMS encap_params;
} TNI_ATTR_PACK;

/******************************************************************************
 * Begin SEND PACKET definition
 *****************************************************************************/
/* 
 * Define Layer type to be used by struct MGB_MDS_SEND_PACKET, 
 * MGB_DMS_RECEIVE_PACKET and MGB_DMS_RECEIVE_PACKET_VIOLATION
 */
/* protocol MGB_MDS_LAYER_ values are also used  as MGB_PT_ values */
#define MGB_MDS_LAYER_RTP    MGB_PT_RTP
#define MGB_MDS_LAYER_AAL2   MGB_PT_AAL2
#define MGB_MDS_LAYER_AAL1   MGB_PT_AAL1     
#define MGB_MDS_LAYER_IU     MGB_PT_IU
/* Note other 0, 1, 2, and 5 are reserved values */
#define MGB_MDS_LAYER_NET   100       /* Network encapsulation of pkts */
#define MGB_MDS_LAYER_RTCP  101       /* RTCP encapsulation (only for RTCP) */
#define MGB_MDS_LAYER_CMR   102       /* CMR for wireless codecs */
#define MGB_MDS_LAYER_TFO   103       /* TFO for wireless codecs */
#define MGB_MDS_LAYER_HDLC  104       /* HDLC/MTP2 */ 

#define MGB_MDS_SEND_PACKET_COMMON   ifpktSendPktCommon_t

#define MGB_SEND_COMMON_INIT_RPTS           ifpkt_SEND_COMMON_INIT_RPTS  /* init rtps in header valid */
#define MGB_SEND_COMMON_INIT_RPTS_INTERVAL  ifpkt_SEND_COMMON_INIT_RPTS_INTERVAL  /* init repeat interval in header valid */
#define MGB_SEND_COMMON_KEEP_ALIVE_INTERVAL ifpkt_SEND_COMMON_KEEP_ALIVE_INTERVAL  /* keep alive interval in header valid */

#define MGB_SEND_AAL2_TYPE3TS               ifpkt_SEND_AAL2_TYPE3TS /* Type 3 timestamp already valid      */
                                                                    /* masks 0x0100, 0x0200 and 0x0400 are reserved */
#define MGB_MDS_SEND_PACKET_PROTO_AAL2      ifpktSendPktProtoAAL2_t

#define MGB_SEND_RTP_VX         ifpkt_SEND_RTP_VX /* V, X bits in header valid           */
#define MGB_SEND_RTP_P          ifpkt_SEND_RTP_P /* P bit in header valid               */
#define MGB_SEND_RTP_M          ifpkt_SEND_RTP_M /* M bit in header valid               */
#define MGB_SEND_RTP_CSRC       ifpkt_SEND_RTP_CSRC /* CC field in header + CSRC's valid   */
#define MGB_SEND_RTP_SEQN       ifpkt_SEND_RTP_SEQN /* Sequence Number in header valid     */
#define MGB_SEND_RTP_TS         ifpkt_SEND_RTP_TS /* Timestamp in header valid           */
#define MGB_SEND_RTP_SSRC       ifpkt_SEND_RTP_SSRC /* SSRC in header valid                */
#define MGB_SEND_RTP_PT         ifpkt_SEND_RTP_PT /* payload type in header valid        */

#define MGB_SEND_RTP_MBIT_PKT_D0_BIT   ifpkt_SEND_RTP_MBIT_PKT_D0_BIT
#define MGB_SEND_RTP_MBIT_PKT_D0       0x0100  /* set:init dur. of M-bit pkt=0ms 
                                                  reset:init dur. of M-bit pkt=30ms  */   
#define MGB_SEND_RTP_MBIT_PKT_T0  0x0200  /* set:TS of digit pkt = T0 
                                           reset:TS of digit pkt = T1         */
                                          /* T0: actual time of the beginning
                                                 of a tone; 
                                             T1: the time required by TDU to 
                                                 detect a tone. For more 									  
                                             detail see QRS4543 of T0 and T1*/
#define MGB_SEND_RTP_RESERVED   0x0400 /* reserved - See MGB_SEND_COMMON_KEEP_ALIVE_INTERVAL */
#define MGB_SEND_RTP_PKTSQUELCH ifpkt_SEND_RTP_PKTSQUELCH /* Enable squelching of individual pkts*/
#define MGB_SEND_RTP_VSQUELCH   ifpkt_SEND_RTP_VSQUELCH /* Enable squelching of voice for event*/
#define MGB_SEND_RTP_INCSEQN    ifpkt_SEND_RTP_INCSEQN /* Increment sequence number for rpts  */
#define MGB_SEND_RTP_INCTS      ifpkt_SEND_RTP_INCTS /* Increment timestamp for rpts        */
#define MGB_SEND_RTP_SET_EBIT   ifpkt_SEND_RTP_SET_EBIT /* if set, DSP will set E bit in the packet,
                                        * this is intended to be used to end an 
                                        * event with E bit set in the packet.
                                        * Host application should send appropriate, initial
                                        * _repeat_count to end the event
                                        * if not set, DSP ignores this bit
                                        */

#define MGB_MDS_SEND_PACKET_PROTO_RTP ifpktSendPktProtoRTP_t


typedef union {
  MGB_MDS_SEND_PACKET_PROTO_AAL2 aal2;
  MGB_MDS_SEND_PACKET_PROTO_RTP  rtp;
  MGB_MDS_SEND_PACKET_PROTO_IU   iu;
} TNI_ATTR_PACK MGB_MDS_SEND_PACKET_PROTO;

typedef struct {
  MGB_MDS_SEND_PACKET_COMMON common;
  MGB_MDS_SEND_PACKET_PROTO  proto;
} TNI_ATTR_PACK MGB_MDS_SEND_PACKET_HEADER;

struct MGB_MDS_SEND_PACKET {
  MGB_MDS_SEND_PACKET_HEADER hdr;
  
  /* Note: AAL2 Payload will have space for cps header to be filled by dsp */
  /* 3bytes CPS header + payload */
  tuint payload[MGB_MSG_MAXNPAR-10]; 
} TNI_ATTR_PACK;

/******************************************************************************
 * End SEND PACKET definition
 *****************************************************************************/
 
/******************************************************************************
 * Begin P2P_STM_RX_CONFIG definition
 *****************************************************************************/
/* These bits are used to select which state machines are being configured */ 
#define MGB_MDS_RX_STM_GENERAL    ifproto_in_RX_STM_GENERAL    /* general config valid */
#define MGB_MDS_RX_STM_ALARM      ifproto_in_RX_STM_ALARM      /* alarm config valid */
#define MGB_MDS_RX_STM_DTMF       ifproto_in_RX_STM_DTMF       /* dtmf config valid */
#define MGB_MDS_RX_STM_CAS        ifproto_in_RX_STM_CAS        /* cas config valid */
#define MGB_MDS_RX_STM_STATESONLY ifproto_in_RX_STM_STATESONLY /* All states config valid */
#define MGB_MDS_RX_STM_2198ENCAP  ifproto_in_RX_STM_2198ENCAP  /* Enable 2833 over 2198 and 
                                                                * RFC 2198 payload type field valid */
#define MGB_MDS_RX_STM_ALL        ifproto_in_RX_STM_ALL        /* all configs valid */

/* Protocol-independant state machine specific parameters */
/***************************************************************************/
/* Note that acuRxStmCfg_t and rcuRxStmCfg_t must match these definitions! */
/***************************************************************************/
#define MGB_MDS_P2P_STM_RX_TIMEOUT_MASK  ifproto_in_P2P_STM_RX_TIMEOUT_MASK
#define MGB_MDS_P2P_STM_RX_TIMEOUT_SHIFT ifproto_in_P2P_STM_RX_TIMEOUT_SHIFT
#define MGB_MDS_P2P_STM_RX_RESET_MASK    ifproto_in_P2P_STM_RX_RESET_MASK
#define MGB_MDS_P2P_STM_RX_RESET_SHIFT   ifproto_in_P2P_STM_RX_RESET_SHIFT
#define MGB_MDS_P2P_STM_RX_COMMON        ifproto_in_P2P_STM_RX_COMMON_t

#define MGB_P2P_STM_RX_SSRC_LOCKED       ifproto_P2P_STM_RX_SSRC_LOCKED
#define MGB_MDS_P2P_STM_RX_RTP           ifproto_in_P2P_STM_RX_RTP_t

#define MGB_MDS_P2P_STM_RX_AAL2          ifproto_in_P2P_STM_RX_AAL2_t

#define MGB_MDS_P2P_STM_RX_ROUTE_PR      ifproto_in_P2P_STM_RX_ROUTE_PR
#define MGB_MDS_P2P_STM_RX_ROUTE_HOST    ifproto_in_P2P_STM_RX_ROUTE_HOST
#define MGB_MDS_P2P_STM_RX_ROUTE_DSP     ifproto_in_P2P_STM_RX_ROUTE_DSP
#define MGB_MDS_P2P_STM_RX_ROUTE_MASK    ifproto_in_P2P_STM_RX_ROUTE_MASK
#define MGB_MDS_P2P_STM_RX_ROUTE_SHIFT   ifproto_in_P2P_STM_RX_ROUTE_SHIFT
#define MGB_MDS_P2P_STM_RX_EXTRA         ifproto_in_P2P_STM_RX_EXTRA_t

/* Protocol/stm dependant components */
#define MGB_MDS_P2P_STM_RX_PROTO         ifproto_in_P2P_STM_RX_PROTO_t

#define MGB_MDS_P2P_STM_RX_PROTO_COMEXT  ifproto_in_P2P_STM_RX_PROTO_COMEXT_t

#define MGB_MDS_P2P_STM_RX               ifproto_in_P2P_STM_RX_t

/* Per-state machine configuration */
#define MGB_P2P_RX_STM_CONFIG_VPBT1 ifproto_P2P_RX_STM_CONFIG_VPBT1
#define MGB_P2P_RX_STM_CONFIG_VPBT2 ifproto_P2P_RX_STM_CONFIG_VPBT2
#define MGB_MDS_P2P_RX_STM_CONFIG   ifproto_in_P2P_RX_STM_CONFIG

/******************************************************************************
 * End P2P_STM_RX_CONFIG definition
 *****************************************************************************/
 
/******************************************************************************
 *  Start of P2P_PKT_CONFIG
 *****************************************************************************/
#define MGB_MDS_P2P_TO_PKT_RTP_2198ENCAP ifproto_in_P2P_TO_PKT_RTP_2198ENCAP /* Turns on 2833 over 2198 */
#define MGB_MDS_P2P_TO_PKT_RTP_PT        ifproto_in_P2P_TO_PKT_RTP_PT        /* Mask for payload type field*/
#define MGB_MDS_P2P_TO_PKT_RTP_RECORD    ifproto_in_P2P_TO_PKT_RTP_RECORD

#define MGB_MDS_P2P_TO_PKT_AAL2_RECORD   ifproto_in_P2P_TO_PKT_AAL2_RECORD

#define MGB_MDS_P2P_TO_PKT_VIF_INTERVAL  ifproto_in_P2P_TO_PKT_VIF_INTERVAL 

#define MGB_MDS_P2P_TO_PKT_COMMON_RECORD ifproto_in_P2P_TO_PKT_COMMON_RECORD

#define MGB_MDS_P2P_TO_PKT_RECORD        ifproto_in_P2P_TO_PKT_RECORD

/* These bits are used to select which state machines are being configured */ 
#define MGB_MDS_TX_STM_GENERAL    ifproto_in_TX_STM_GENERAL     /* general config valid */
#define MGB_MDS_TX_STM_ALARM      ifproto_in_TX_STM_ALARM       /* alarm config valid */
#define MGB_MDS_TX_STM_DTMF       ifproto_in_TX_STM_DTMF        /* dtmf config valid */
#define MGB_MDS_TX_STM_CAS        ifproto_in_TX_STM_CAS         /* cas config valid */
#define MGB_MDS_TX_STM_STATESONLY ifproto_in_TX_STM_STATESONLY  /* All states config is valid */
#define MGB_MDS_TX_STM_2198ENCAP  ifproto_in_TX_STM_2198ENCAP   /* TX 2198 payload type field valid */
#define MGB_MDS_TX_STM_ALL        ifproto_in_TX_STM_ALL         /* all configs valid */
    
#define MGB_P2P_TO_PKT_CONFIG_VPBT1  ifproto_P2P_TO_PKT_CONFIG_VPBT1
#define MGB_P2P_TO_PKT_CONFIG_VPBT2  ifproto_P2P_TO_PKT_CONFIG_VPBT2

#define MGB_MDS_TX_STM_GENERAL_INDX  ifproto_in_TX_STM_GENERAL_INDX
#define MGB_MDS_TX_STM_ALARM_INDX    ifproto_in_TX_STM_ALARM_INDX
#define MGB_MDS_TX_STM_DTMF_INDX     ifproto_in_TX_STM_DTMF_INDX
#define MGB_MDS_TX_STM_CAS_INDX      ifproto_in_TX_STM_CAS_INDX
#define MGB_MDS_TX_NUM_STMS          ifproto_in_TX_NUM_STMS


#define MGB_MDS_P2P_TO_PKT_CONFIG    ifproto_in_P2P_TO_PKT_CONFIG

/******************************************************************************
 *  End of P2P_TX_PKT_CONFIG
 *****************************************************************************/

/******************************************************************************
 * Begin RECEIVE PACKET Definitions 
 *****************************************************************************/
#define MGB_DMS_RECEIVE_PACKET_COMMON      ifpktReceivePktCommon_t

#define MGB_DMS_RECEIVE_PACKET_PROTO_AAL2  ifpktReceivePktProtoAAL2_t

#define MGB_DMS_RECEIVE_PACKET_PROTO_RTP   ifpktReceivePktProtoRTP_t

typedef struct {
  tuint  packetType;                   /* Packet Type */ 
#define MGB_RECEIVE_RTCPXR_TX      0x0001 /* XR report sent */
#define MGB_RECEIVE_RTCPXR_RX      0x0002 /* XR report received */
#define MGB_RECEIVE_RTCPAPP_PKT    0x0004 /* APP packet received */
#define MGB_RECEIVE_RTCP_SR_RR_RX  0x0008 /* Regular RTCP packet received */
#define MGB_RECEIVE_RTCP_SR_RR_TX  0x0010 /* Regular RTCP packet transmitted */
#define MGB_RECEIVE_RTCPXR_PIQUA_RX 0x0020 /* XR PIQUA block received */
#define MGB_RECEIVE_RTCPRTPFB_PKT   0x0040 /* RTP Feedback block received */
#define MGB_RECEIVE_RTCPPSFB_PKT    0x0080 /* Payload specific feedback received */

  tuint  packetLength;                 /* packet length in bytes  */
  tuint  reserved[3];

} TNI_ATTR_PACK MGB_DMS_RECEIVE_PACKET_PROTO_RTCP;


typedef union {
  MGB_DMS_RECEIVE_PACKET_PROTO_AAL2 aal2;
  MGB_DMS_RECEIVE_PACKET_PROTO_RTP   rtp;
  MGB_MDS_RECEIVE_PACKET_PROTO_IU     iu;
  MGB_DMS_RECEIVE_PACKET_PROTO_TFO   tfo; 
  MGB_DMS_RECEIVE_PACKET_PROTO_RTCP  rtcp;
} TNI_ATTR_PACK MGB_DMS_RECEIVE_PACKET_PROTO;

typedef struct {
  MGB_DMS_RECEIVE_PACKET_COMMON common;
  MGB_DMS_RECEIVE_PACKET_PROTO  proto;
} TNI_ATTR_PACK MGB_DMS_RECEIVE_PACKET_HEADER;

struct MGB_DMS_RECEIVE_PACKET {
  tuint layer;   
  MGB_DMS_RECEIVE_PACKET_HEADER hdr;
  tuint payload[MGB_MSG_MAXNPAR-11];
} TNI_ATTR_PACK;
/******************************************************************************
 * END RECEIVE PACKET Definitions 
 *****************************************************************************/
 
/******************************************************************************
 * Begin Codec/Voice/Peer to Peer Profile Definitions 
 *****************************************************************************/
typedef struct {                                                
  tuint encapsulationSpec; 
     /* Specifies the encapsulation parameters/options                        *
      * Bits 0-2:  Payload format:                                            *
      * 0 - Linear                                                            *
      * 1 - RFC 3267  (For AMR)                                               *
      * 2 - RFC 2658  (For Qcelp)                                             *
      * 3 - RFC 3558  (For EVRC and SMV)                                      *
      * Bit  3-4:      Packing Options                                        *
      *             0 - Bandwidth Efficient                                   *
      *             1 - Octet Aligned                                         *
      * Bit  5:     CRC 0-Disabled, 1-Enabled                                 *
      *                                                                       *
      * Bit  6:     Suppress continous CMR reports with the same value to host*
      * Bits 7-15:  Reserved:FEC, interleaving,                               *
      *             can be added over here in future                          * 
      * (options=-h-v3)                                                       */
#define MGB_PROFILE_CPREC_RTP_PAYLDFRMT_MASK         0x0007                                               
#define MGB_PROFILE_CPREC_RTP_PAYLDFRMT_SHIFT        0                                                  
#define MGB_PROFILE_CPREC_RTP_PACKOPTS_MASK          0x0018                                                 
#define MGB_PROFILE_CPREC_RTP_PACKOPTS_SHIFT         3                                                  
#define MGB_PROFILE_CPREC_RTP_CRC_MASK               0x0020                                                 
#define MGB_PROFILE_CPREC_RTP_CRC_SHIFT              5 
#define MGB_PROFILE_CPREC_RTP_SUPP_SAME_CMR_MASK     0x0040                                                 
#define MGB_PROFILE_CPREC_RTP_SUPP_SAME_CMR_SHIFT    6
#define MGB_PROFILE_CPREC_RTP_PACKING_CTRL_MASK      (MGB_PROFILE_CPREC_RTP_PACKOPTS_MASK | \
                                                      MGB_PROFILE_CPREC_RTP_CRC_MASK | \
                                                      MGB_PROFILE_CPREC_RTP_SUPP_SAME_CMR_MASK) 
#define MGB_PROFILE_CPREC_RTP_PACKING_CTRL_SHIFT     MGB_PROFILE_CPREC_RTP_PACKOPTS_SHIFT                                                    

  tuint encoderRateInfo;           
   /* Encoder rate information  (Only applicable for multirate codecs)        *
    * Bits 0-3:  Specify the encoder default rate / rate reduction mode       *
    *      EVRC(B)/Qcelp13    SMV                            GSM-AMR          *
    *   0 - Mode 0       0 - Premium Mode                 0 - Rate Index - 0  *
    *   1 - Mode 1       1 - Standard Mode                1 - Rate Index - 1  *
    *   2 - Mode 2       2 - Economy Mode                  .................  *
    *   3 - Mode 3       3 - Capacity Saving Mode         7 - Rate Index - 7  *
    *   4 - Mode 4       4 - Premium Mode (Half rate Max)         .           *
    *   5 - Mode 5
    *   6 - Mode 6
    *   7 - mODE 7
    *   ---              5 - Standard Mode (Half Rate max)                    *
    *   10  Fixed Rate EVRC FULL   (Please see note below)
    *   11  Fixed Rate EVRC HALF   (Please see note below)
    *   EVRCB   
    * Mode	Target average (channel) encoding rate (kbps)
    *  0	        9.3                                                       *
    *  1	        8.5                                                       *
    *  2	        7.5                                                       *
    *  3	        7.0                                                       *
    *  4	        6.6                                                       *
    *  5	        6.2                                                       *
    *  6	        5.8                                                       *
    *  7	        4.8                                                       *
    *  8          2.4                                                       *
    *  9         RESERVED                                                     *
    *  10        Fixed Rate EVRCB-FULL (Please see note below)                *
    *  11        Fixed Rate EVRCB-HALF (Please see note below)                *
    *
    *     G.723                                                               *
    *  0 - 5.3 kbps                                                           *
    *  1 - 6.3 kbps
    *
    *        GSM AMR-WB                                                       *
    *    0 - Rate Index - 0                                                   *     
    *    1 - Rate Index - 1                                                   *      
    *    2 - Rate Index - 2                                                   *      
    *    3 - Rate Index - 3                                                   *      
    *    --------------------                                                 *
    *    8 - Rate Index - 8                                                   *
    *  Bits 4-15:  Valid encoder rate mask.                                   *
    *  Used to enable/disable specific encoder rates in a session             *
    *                                                                         *
    * Note: The values of EVRCB FULL/HALF rate should not change from 10      *
    *       and 11 as they constitute the last two bits of the 12 bit rate    *
    *       mask.                                                             */
   /* (options=-v3-h)                                                         */
#define MGB_PROFILE_CPREC_RTP_START_RATEMODE_MASK     0x000F                                                 
#define MGB_PROFILE_CPREC_RTP_START_RATEMODE_SHIFT    0                                                  
#define MGB_PROFILE_CPREC_RTP_VALID_RATEMODE_MASK     0xFFF0                                                 
#define MGB_PROFILE_CPREC_RTP_VALID_RATEMODE_SHIFT    4   
  
#define MGB_PROFILE_CPREC_RTP_RATEMODE_VCRATE_FULL    10
#define MGB_PROFILE_CPREC_RTP_RATEMODE_VCRATE_HALF    11

  tuint codecOptions;
   /* Codec Options                                                           *
    * Bits 0-2: VAD TYPE:                                                     *
    *           0 - RESERVED                                                  *
    *           1 - RESERVED                                                  *
    *           2 - RESERVED                                                  *
    *           3 - MGB_VAD_INTERNAL1 Applicable for SMV,AMR,AMR-WB;N/A for other *
    *           4 - MGB_VAD_INTERNAL2 Applicable for SMV,AMR; N/A for other   *
    * Bit 3: Post filtering                                                   *
    *           0 - Disable  Applicable for EVRC(B); N/A for other               *
    *           1 - Enable   Applicable for EVRC(B); N/A for other               *
    * Bits 4-5: Codec output format                                           *
    *           0 - IF1 no CRC  Applicable for AMR, AMR-WB;N/A for other codecs*
    *           1 - IF1 with CRC                                              *
    *           2 - IF2                                                       *
    * Bits 6-7:Noise Suppression:Applicable for EVRC(B)/SMV, N/A for other codecs*
    *      EVRC(B)               SMV                                             *
    *    0 - Disable        0 - N/A for SMV                                   *
    *    1 - Enable         1 - Enable NSA                                    *                                   
    *    2 - N/A For EVRC   2 - Enable NSB                                    *
    *                                                                         *
    * Bits 8-9: TTY/TDD.Applicable for EVRC(B)/SMV/Qcelp13.N/A for other codecs  *
    *           0 - Disable                                                   *
    *           1 - Enable 45.45 baud rate                                    *
    *           2 - Enable 50 baud rate                                       *
    * Bits 10:  DTMF                                                       
    *           0 - Disable                                                   *
    *           1 - Enable                                                    *
    * Bits 11-12:  mode_change_period ;Applicable for AMR/AMR-WB              *
    *           Number of 20ms frames between consecutive mode changes        *
    *           i.e mode changes are allowed every (mode-change-period+1) 20ms*
    *           intervals. Please refer to RFC3267 for details                *
    * Bits 13-15: Reserved                                                    *
    * (options=-h-v3)                                                         * 
    * Warning: This bitfield must be a superset of                            *
    * mgmicdsp.h:MGB_MDS_ENCODE_CONFIG_CODEC_OPTIONS.codec_options            */
#define MGB_PROFILE_CPREC_RTP_VADTYPE_MASK          0x0007                                                
#define MGB_PROFILE_CPREC_RTP_VADTYPE_SHIFT         0                                                  
#define MGB_PROFILE_CPREC_RTP_POSTFILTER_MASK       0x0008                                                
#define MGB_PROFILE_CPREC_RTP_POSTFILTER_SHIFT      3                                                  
#define MGB_PROFILE_CPREC_RTP_CODEC_FRAMEFRMT_MASK  0x0030                                                
#define MGB_PROFILE_CPREC_RTP_CODEC_FRAMEFRMT_SHIFT 4                                                  
#define MGB_PROFILE_CPREC_RTP_NOISE_SUPP_MASK       0x00C0                                                
#define MGB_PROFILE_CPREC_RTP_NOISE_SUPP_SHIFT      6                                                  
#define MGB_PROFILE_CPREC_RTP_TTYTTD_MASK           0x0300                                                
#define MGB_PROFILE_CPREC_RTP_TTYTTD_SHIFT          8                                                  

#define MGB_PROFILE_CPREC_RTP_DTMF_MASK                 0x0400
#define MGB_PROFILE_CPREC_RTP_DTMF_SHIFT                10
#define MGB_PROFILE_CPREC_RTP_MODE_CHANGE_PERIOD_MASK   0x1800
#define MGB_PROFILE_CPREC_RTP_MODE_CHANGE_PERIOD_SHIFT  11

} TNI_ATTR_PACK MGB_MDS_CODEC_PROFILE_RTP;

#define MGB_MAX_CODEC_PROFILES  5

typedef struct {
  tuint codingTypeSeqInfo;     
    /* Bits 00-03: Sequence Number Base (0-15)                                * 
     * Bits 04-07: Sequence Number Increment per Vif                          * 
     * Bits 08-15: Coding Type                                                */
    /* (options=-v1-h)                                                        */
#define MGB_PROFILE_VPREC_AAL2_SEQBASE_MASK  0x000F
#define MGB_PROFILE_VPREC_AAL2_SEQBASE_SHIFT 0
#define MGB_PROFILE_VPREC_AAL2_SEQINCR_MASK  0x00F0
#define MGB_PROFILE_VPREC_AAL2_SEQINCR_SHIFT 4
#define MGB_PROFILE_VPREC_AAL2_CODTYPE_MASK  0xFF00
#define MGB_PROFILE_VPREC_AAL2_CODTYPE_SHIFT 8
  tuint vifSize ;             /* TX or RX Voice payload in bits.             *  
                                * if MSB set, dummy entry                     */
                               /* (options=-v1)                               */
#define MGB_PROFILE_VPREC_AAL2_DUMMY_PROFILE 0x8000

} TNI_ATTR_PACK MGB_MDS_PROFILE_RECORD_AAL2_VOICE;


typedef struct {
  tuint codingPayloadType; 
    /* Coding Type and RTP Payload Type                                       * 
     * Bits 00-06 : Payload Type (0-127)                                      * 
     * Bit     07 : Reserved                                                  * 
     * Bits 08-15 : Coding type                                               */
    /* (RX value for switchover; Default value for TX unless overridden       *
     * in encode/channel modify                                               */
    /* (options=-v1-h)                                                        */
#define MGB_PROFILE_VPREC_RTP_PT_MASK       0x007F
#define MGB_PROFILE_VPREC_RTP_PT_SHIFT      0
#define MGB_PROFILE_VPREC_RTP_CODTYPE_MASK  0xFF00
#define MGB_PROFILE_VPREC_RTP_CODTYPE_SHIFT 8
  tuint codingProfileInfo;           
    /* Coding Profile Information                                   * 
     * Bits 00-01 : indicates frame size                            * 
     *              0 - 5ms                                         * 
     *              1 - 5.5ms                                       *
     *              2 - 10ms                                        * 
     * Bits 02-05 : Transmit VIF size in codeword segments          * 
     * For sample based codecs, this indicates the number of        * 
     *    segments in terms of  5 or 5.5 ms (indicated by Bits 0-1) * 
     * For other codecs this indicates the number of codeword       * 
     *    segments (corresponds to natural frame size of the codec) * 
     *    0 - Dummy profile                                         * 
     * Bits 06-08 : Codec Profile Index - Specifies the associated  * 
     *              codec profile to be used.                       * 
     * Bits 09-15 : Reserved                                        */
     /* (options=-v1-h)                                             */
#define MGB_PROFILE_VPREC_RTP_FSIZE_MASK         0x0003                                                 
#define MGB_PROFILE_VPREC_RTP_FSIZE_SHIFT        0                                                  
#define MGB_PROFILE_VPREC_RTP_VIFSIZE_MASK       0x003C                                                 
#define MGB_PROFILE_VPREC_RTP_VIFSIZE_SHIFT      2                                                  
#define MGB_PROFILE_VPREC_RTP_DUMMY_PROFILE      0                                                  
#define MGB_PROFILE_VPREC_RTP_CPROF_INDEX_MASK   0x01C0                                                 
#define MGB_PROFILE_VPREC_RTP_CPROF_INDEX_SHIFT  6
} TNI_ATTR_PACK MGB_MDS_PROFILE_RECORD_RTP_VOICE;

typedef struct {
  tuint bitmap1;               
     /* See mgmicdsp.h for more info                                          */
     /* (options=-h-v1)                                                       */
     /* Bits 00-04 : First Rx UUI                                             */
     /* Bits 05-06 : Reserved                                                 */
     /* Bit     07 : (FLT) 1 - range of UUIs defined                          */
     /*                    0 - range of type 3 subtypes defined               */
     /* Bits 08-13: Number of UUIs selected (if FLT = 1) or                   */
     /*             number of subtype values (if FLT = 0)                     */
     /*             0 -> select all UUIs or subtypes                          */
     /* Bits 14-15: Reserved                                                  */
#define MGB_PROFILE_PPREC_AAL2_STARTUUI_MASK  0x001F
#define MGB_PROFILE_PPREC_AAL2_STARTUUI_SHIFT 0
#define MGB_PROFILE_PPREC_AAL2_BIT_FLT        0x0080
#define MGB_PROFILE_PPREC_AAL2_RANGE_MASK     0x3F00
#define MGB_PROFILE_PPREC_AAL2_RANGE_SHIFT    8
  tuint bitmap2;
     /* See mgmicdsp.h for more info                                          */
     /* (options=-h-v1)                                                       */
     /* Bits 00-03 : State machine number/routing code                        */
     /*                 0: Reserved                                           */
     /*                 1: General Purpose (user data)                        */
     /*                 2: Alarms                                             */
     /*                 3: DTMF/MF relay, FAX data                            */
     /*                 4: CAS relay                                          */
     /*              5-11: Reserved                                           */
     /*                12: Packet violation                                   */
     /*                13: Send to host via RECEIVE_PACKET (throttled)        */
     /*                14: Send to host via RECEIVE_PACKET (unthrottled)      */
     /*                15: Drop the packet                                    */
     /* Bits 04-07 : Reserved                                                 */
     /* Bits 08-13 : Starting UUI subtype (if FLT = 0)                        */
     /*              Reserved (if FLT = 1)                                    */
     /* Bits 14-15 : Reserved                                                 */
#define MGB_PROFILE_PPREC_AAL2_STMCODE_MASK   0x000F
#define MGB_PROFILE_PPREC_AAL2_STMCODE_SHIFT  0
#define MGB_PROFILE_PPREC_AAL2_STARTSUB_MASK  0x3F00
#define MGB_PROFILE_PPREC_AAL2_STARTSUB_SHIFT 8
} TNI_ATTR_PACK MGB_MDS_PROFILE_RECORD_AAL2_P2P;

typedef struct {
  tuint bitmap1 ;
     /* See mgmicdsp.h for more info                                          */
     /* (options=-h-v1)                                                       */
     /* Bits 00-06 : First Rx payload type (0 - 127)                          */
     /* Bit     07 | (FLT) : 1 - range of payload types defined               */
     /*                      0 - range of NSE subtypes defined                */
     /* Bits 08-13 : Number of payload types (if FLT = 1) or                  */
     /*              number of NSE subtypes (if FLT = 0)                      */
     /*              (0 -> select all events or payload types)                */
     /* Bits 14-15 : Reserved                                                 */
#define MGB_PROFILE_PPREC_RTP_STARTPT_MASK  0x007F
#define MGB_PROFILE_PPREC_RTP_STARTPT_SHIFT 0
#define MGB_PROFILE_PPREC_RTP_BIT_FLT       0x0080
#define MGB_PROFILE_PPREC_RTP_RANGE_MASK    0x3F00
#define MGB_PROFILE_PPREC_RTP_RANGE_SHIFT   8
  tuint bitmap2 ;
     /* See mgmicdsp.h for more info                                          */
     /* (options=-h-v1)                                                       */
     /* Bits 00-03 : State machine number/routing code                        */
     /*                 0: Voice profile                                      */
     /*                 1: General Purpose (user data)                        */
     /*                 2: Alarms                                             */
     /*                 3: DTMF/MF relay, FAX data                            */
     /*                 4: CAS relay                                          */
     /*              5-11: Reserved                                           */
     /*                12: Packet violation                                   */
     /*                13: Send to host via RECEIVE_PACKET (throttled)        */
     /*                14: Send to host via RECEIVE_PACKET (unthrottled)      */
     /*                15: Drop the packet                                    */
     /* Bits 04-07 : Reserved                                                 */
     /* Bits 08-15 : Starting NSE subtype (if FLT = 0)                        */
     /*              Reserved (if FLT = 1)                                    */
#define MGB_PROFILE_PPREC_RTP_STMCODE_MASK   0x000F
#define MGB_PROFILE_PPREC_RTP_STMCODE_SHIFT  0
#define MGB_PROFILE_PPREC_RTP_STARTNSE_MASK  0xFF00
#define MGB_PROFILE_PPREC_RTP_STARTNSE_SHIFT 8
} TNI_ATTR_PACK MGB_MDS_PROFILE_RECORD_RTP_P2P;

/* The following recs are used to hold the variable number of vp (voice profile),
 * pp (peer to peer profile) records.
 * Since the number of profiles is variable their respective structures cannot be used
 * below. The recs need to be interpreted according to the values in numRecs field
 * of MGB_MDS_PROFILE_DEFINE.
 * The records are in the following order: first vp, then pp (if num_pp != 0), 
 */
typedef union {
  tuint recs[MGB_MSG_MAXNPAR-8];
} TNI_ATTR_PACK MGB_MDS_PROFILE_RECORDS_AAL2;

/* The following recs are used to hold the variable number of vp (voice profile),
 * pp (peer to peer profile) and cp (codec profile) records.
 * Since the number of profiles is variable their respective structures cannot be used
 * below. The recs need to be interpreted according to the values in numRecs field
 * of MGB_MDS_PROFILE_DEFINE.
 * The records are in the following order: first vp, then pp (if num pp != 0), 
 * then cp (if num cp != 0).
 */
typedef union {
  tuint recs[MGB_MSG_MAXNPAR-8];
} TNI_ATTR_PACK MGB_MDS_PROFILE_RECORDS_RTP;

typedef struct {
  tuint sequenceNumberMask;   /* 0xF, 0x7, 0x3, 0x1, 0x0 - divides uui into
                                * subranges */
                               /* (options=-h-v1)                             */
  tuint interopMode;          /* 1=VMOA, 2=CopperCom, 3=JetStream            */
                               /* (options=-v3)                               */
#define MGB_INTEROP_MODE_VMOA                   0x0001
#define MGB_INTEROP_MODE_COPPERCOM              0x0002
#define MGB_INTEROP_MODE_JETSTREAM              0x0003
} TNI_ATTR_PACK MGB_MDS_PROFILE_COMMON_AAL2;

typedef struct {
  tuint reserved1;            /* (private)                                   */
  tuint reserved2;            /* (private)                                   */
} TNI_ATTR_PACK MGB_MDS_PROFILE_COMMON_RTP;


typedef struct {
  MGB_MDS_PROFILE_COMMON_AAL2 common;
  MGB_MDS_PROFILE_RECORDS_AAL2  recs;
} TNI_ATTR_PACK MGB_MDS_PROFILE_AAL2;

typedef struct {
  MGB_MDS_PROFILE_COMMON_RTP common;
  MGB_MDS_PROFILE_RECORDS_RTP recs;
} TNI_ATTR_PACK MGB_MDS_PROFILE_RTP;

typedef union {
  MGB_MDS_PROFILE_RTP  rtp;
  MGB_MDS_PROFILE_AAL2 aal2;
  /* KJ: For now, Iu UP gets rid of unnecessary levels of indirection. RTP
   * and AAL2 could be reworked to use this method. */
  /* IU only supports VPRecs=0,PPRecs=0,CPRecs=1, where iuRecs is casted to
   * iuupapi.h:MGB_MDS_CODEC_PROFILE_IU   
   */
  tuint iuRecs[MGB_MSG_MAXNPAR-7];
} TNI_ATTR_PACK MGB_MDS_PROFILE;

struct MGB_MDS_PROFILE_DEFINE {
  tuint numRecs;
     /* Bits 00-04: Number of Voice Profile Records (VP)                      */                     
     /* Bits 05-09: Number of Peer to Peer Profile Record (PP)                */                     
     /* Bits 10-12: Number of Codec Profile Records                           */                     
     /* Bits 13-15: Reserved                                                  */                     
#define MGB_PROFILE_DEFINE_NUMVPRECS_MASK  0x001F                       
#define MGB_PROFILE_DEFINE_NUMVPRECS_SHIFT 0                        
#define MGB_PROFILE_DEFINE_NUMPPRECS_MASK  0x03E0                       
#define MGB_PROFILE_DEFINE_NUMPPRECS_SHIFT 5
#define MGB_PROFILE_DEFINE_NUMCPRECS_MASK  0x1C00                       
#define MGB_PROFILE_DEFINE_NUMCPRECS_SHIFT 10                       
  tuint initCodecs; 
     /* Bits 00-07: Initial TX codec (between 0 and VP-1)                     */
     /* Bits 08-15: Initial RX codec (between 0 and VP-1)                     */
#define MGB_PROFILE_DEFINE_INITTX_MASK  0x00FF
#define MGB_PROFILE_DEFINE_INITTX_SHIFT 0
#define MGB_PROFILE_DEFINE_INITRX_MASK  0xFF00
#define MGB_PROFILE_DEFINE_INITRX_SHIFT 8
  tuint pktViolationFilter;  
     /* Bit     00 : If set : If AAL2 : Switching coding types, ex SID        */
     /*                       If RTP  : Switching lines in profile ex SID     */
     /* Bit     01 : If set : If AAL2 : Switching VIF sizes of same codec     */
     /*                       If RTP  : Not used                              */
     /* Bit     02 : If set : Both    : Not found in profile                  */
     /* Bit     04 : If set : If RTP:RFC3267: rate change in received payload */
     /* Bits 05-07 : Reserved                                                 */
     /* Bit     08 : If set : If AAL2 : CID changed                           */
     /* Bit     08 : If set : If RTP  : SSRC changed                          */
     /* Bits 09-11 : Reserved                                                 */
     /* Bit     12 : If set  : The remote side DTX capability is changed      */
     /* Bit     13 : If set  : if RTP:RFC3267: received packet contains a mode*/
     /*                        which is not in the mode-set                   */
     /* Bit     14 : If set  : if RTP:RFC3267: received payload is invalid    */
     /* Bit     15 : If set    Block asymmetric LBRC if build supports variable*/
     /*                        numbers of channels for each type of codec     */

#define MGB_PROFILE_DEFINE_PKTVIOL_PROFLINE ifproto_PROFILE_DEFINE_PKTVIOL_PROFLINE /* Profile line chg, ex sid*/
                                                   /* if aal2, coding type chg*/
#define MGB_PROFILE_DEFINE_PKTVIOL_PROFVIF  0x0002 /* vif size changed (AAL2) */
#define MGB_PROFILE_DEFINE_PKTVIOL_PROFMISS ifproto_PROFILE_DEFINE_PKTVIOL_PROFMISS /* not in profile          */
#define MGB_PROFILE_DEFINE_PKTVIOL_HOSTREQ  ifproto_PROFILE_DEFINE_PKTVIOL_HOSTREQ /* State machine 12        */

#define MGB_PROFILE_DEFINE_PKTVIOL_RTP_SSRC ifproto_PROFILE_DEFINE_PKTVIOL_RTP_SSRC /* SSRC changed            */
#define MGB_PROFILE_DEFINE_PKTVIOL_AAL2_CID 0x0100 /* CID changed             */
#define MGB_PROFILE_DEFINE_PKTVIOL_UDPTL    ifproto_PROFILE_DEFINE_PKTVIOL_UDPTL /* UDP-TL Packet detected  */


#define MGB_PROFILE_DEFINE_PKTVIOL_AAL2_DEF 0x0107 /* Default at open chan    */
#define MGB_PROFILE_DEFINE_PKTVIOL_AAL2_ALL 0x010F /* Nothing blocked         */
#define MGB_PROFILE_DEFINE_PKTVIOL_RTP_DEF  0x0107 /* Default at open chan    */
#define MGB_PROFILE_DEFINE_PKTVIOL_RTP_ALL  0x030F /* Nothing blocked         */

/* The bit indicates packet violation for reporting that remote side 
 * capability for DTX support is changed, either it was disabled and now it is
 * enabled or vice versa
 */
#define MGB_PROFILE_DEFINE_PKTVIOL_DTX_CHANGE         0x1000

/* If set, block all asymmetric LBRC if build supports variable numbers
 * of channels for each type of codec.
 */
#define MGB_PROFILE_DEFINE_PKTVIOL_BLOCK_ASYM        0x8000

/* The bit indicates packet violation for packet being received 
 * with invalid payload example CRC failures.Currently the bit is valid 
 * only in the case of RFC3267 using RTP encapsulation
 */
#define MGB_PROFILE_DEFINE_PKTVIOL_PLOAD_INV         0x4000

/* The bit indicates packet violation for packet being received 
 * with rate mode different from the configured rate mode set in 
 * MGB_MDS_CODEC_PROFILE_RTP->encoderRateInfo.Currently the bit is valid 
 * only in the case of RFC3267 using RTP encapsulation
 */
#define MGB_PROFILE_DEFINE_PKTVIOL_RATEMODE_MISS      0x2000

/* Report packet violation upon receiving payload containing rate mode   
 * matching the mode request in outgoing packet. This is reported once 
 * for every CMR set operation. Bit valid only in the case of RFC3267 
 * using RTP encapsulation. Bit reserved in all other cases and will 
 * have no effect.   */ 
#define MGB_PROFILE_DEFINE_PKTVIOL_RATEMATCH_CMR      0x0010 

  tuint pktViolThrottle;
   /* Bits 14-15: Reserved
    * Bits 00-13: Minimum interpacket time in 8ms units
    *             0: packet violations shut off
    *             >0: Minimum interpacket time
    */
#define MGB_PROFILE_DEFINE_PKTVIOL_THROTTLE_MASK 0x3FFF
  tuint hostPktThrottle;
   /* Bits 14-15: Reserved
    * Bits 00-13: Minimum interpacket time in 8ms units
    *             0: Route-to-host-throttled packets shut off
    *             >0: Minimum interpacket for route-to-host-throttled pkts
    *             Note: This parameter does not affect any packets routed
    *             to host other than those through STM #13
    */
#define MGB_PROFILE_DEFINE_HOSTPKT_THROTTLE_MASK 0x3FFF
  tuint bitfield;
  /* Bits 05-15: Reserved
   * Bit  04   : Codec Bypass Mode Selection
   * Bits 00-03: Packet violation burst integration limit
   *             0: Reserved
   *             1-15(N).  N packet violations can be received before any throttle
   *             kicks in.  Once N packet violations are received within 
   *             pktViolThrottle, all packet violations are dropped.
   *
   * For example: Assume the integration value is 4 and pktViolThrottle is 1000ms.
   * From the empty state, lets assume we get 5 violations 20ms apart.  
   * T=   0: viol # 1 reported; timer=1000, count=1
   * T=  20: viol # 2 reported; timer=980, count=2
   * T=  40: viol # 3 reported; timer=960, count=3
   * T=  60: viol # 4 reported; timer=940, count=4
   * T=  80: viol # 5 dropped since count=4; timer=920, count=4
   * T=1000: timer reset to 1000, count = 3
   * T=1100: viol # 6 reported; timer=900, count=4
   * T=2000: timer reset to 1000, count = 3
   * T=3000: timer reset to 1000, count = 2
   * T=4000: timer reset to 1000, count = 1
   * T=5000: timer empty, count = 0
   */
#define MGB_PROFILE_DEFINE_PKTVIOL_BURST_INTEGRATION_MASK 0x000F

#define MGB_PROFILE_DEFINE_VOCODER_BYPASS_MODE_MASK       0x10
#define MGB_PROFILE_DEFINE_VOCODER_BYPASS_MODE_SHIFT      4
#define MGB_PROFILE_DEFINE_VOCODER_BYPASS_MODE_ENBL       0x10

#define MGB_PROFILE_PAYLDFRMT_LINEAR  0
#define MGB_PROFILE_PAYLDFRMT_RFC3267 1
#define MGB_PROFILE_PAYLDFRMT_RFC2658 2
#define MGB_PROFILE_PAYLDFRMT_RFC3558 3

#define  MGB_PROFILE_PAYLD_PACKING_TYPE_RFC3558_INTERLEAVED   0
#define  MGB_PROFILE_PAYLD_PACKING_TYPE_RFC3558_CMPCT_HDRFREE 1
#define  MGB_PROFILE_PAYLD_PACKING_TYPE_RFC3558_HDRFREE 	  2

  MGB_MDS_PROFILE profile;
} TNI_ATTR_PACK;

/******************************************************************************
 * End Voice/Peer to Peer Profile Definitions 
 *****************************************************************************/
 
/******************************************************************************
 * Begin Packet Violation Definitions 
 *****************************************************************************/
struct MGB_DMS_AAL2_NOTIFICATION {
  tuint  cid;                         /* Channel ID                       */
  tuint  li;                          /* LI - Length Indication           */
  tuint  uui;                         /* UUI - User-to-User Indication    */
  tuint  msg_id;                      /* Type 3 message ID                */
  tuint  reserved;                    /* align with RTP                   */
  tuint  stm;                         /* State machine pkt viol (0=voice) */
  tuint  pktViolReason;               /* MGB_PROFILE_DEFINE_PKTVIOL_*     */
} TNI_ATTR_PACK;

#define MGB_DMS_RECEIVE_PACKET_VIOLATION_RTP  ifrcu_out_RECEIVE_PACKET_VIOLATION_RTP

#define MGB_IP_ADDR_MAX_NUM_WORDS             ifneu_IP_ADDR_MAX_NUM_WORDS
#define MGB_ETH_ADDR_MAX_NUM_WORDS            ifneu_ETH_ADDR_MAX_NUM_WORDS

#define MGB_NET_ENCAP_VIO_VALID_SRC_ID        ifneu_NET_ENCAP_VIO_VALID_SRC_ID
#define MGB_NET_ENCAP_VIO_VALID_LAYER_ID      ifneu_NET_ENCAP_VIO_VALID_LAYER_ID
#define MGB_NET_ENCAP_VIO_VALID_PROTO         ifneu_NET_ENCAP_VIO_VALID_PROTO
#define MGB_NET_ENCAP_VIO_VALID_LOC_PORT      ifneu_NET_ENCAP_VIO_VALID_LOC_PORT
#define MGB_NET_ENCAP_VIO_VALID_LOC_ADDR      ifneu_NET_ENCAP_VIO_VALID_LOC_ADDR
#define MGB_NET_ENCAP_VIO_VALID_REM_PORT      ifneu_NET_ENCAP_VIO_VALID_REM_PORT
#define MGB_NET_ENCAP_VIO_VALID_REM_ADDR      ifneu_NET_ENCAP_VIO_VALID_REM_ADDR
#define MGB_NET_ENCAP_VIO_VALID_LOC_ETH_ADDR  ifneu_NET_ENCAP_VIO_VALID_LOC_ETH_ADDR
#define MGB_NET_ENCAP_VIO_VALID_REM_ETH_ADDR  ifneu_NET_ENCAP_VIO_VALID_REM_ETH_ADDR

#define MGB_DMS_NEU_RX_PACKET_VIOLATION       ifneu_out_NEU_RX_PACKET_VIOLATION

union MGB_DMS_RECEIVE_PACKET_VIOLATION_LAYER_INFO {
  struct MGB_DMS_RECEIVE_PACKET_VIOLATION_RTP rtp;
  struct MGB_DMS_AAL2_NOTIFICATION            aal2;
  struct MGB_DMS_NEU_RX_PACKET_VIOLATION      neu;
  struct MGB_DMS_VIOLATION_IU                 iu;
  struct MGB_DMS_VIOLATION_TFO                tfo;
} TNI_ATTR_PACK;

struct MGB_DMS_RECEIVE_PACKET_VIOLATION {
  tuint layer;
  union MGB_DMS_RECEIVE_PACKET_VIOLATION_LAYER_INFO layer_info;    
} TNI_ATTR_PACK;
 
/******************************************************************************
 * End Packet Violation Definitions 
 *****************************************************************************/

/****************************************************************************** 
 *                                                                            *
 ******************************************************************************/
/*********************************************************************** 
 * Name: MGB_MD_CHANNEL_STATE                                          *
 * Description: Determination of packet data flow, both to network and *
 *              telephony, for all kind of packets:                    *
 *            - application information (voice/fax)                    *
 *            - user data (peer-to-peer)                               *
 *            - alarm peer-to-peer                                     *
 *            - DTMF/MF & other tone relay                             *
 *            - CAS signalling                                         *
 * There are two bit fields per direction: change bit field and actual *
 * value of bit field.                                                 *
 * Additional functionality is required in RCU/ACU modules - these are *
 * only modules with necessary information available.                  *
 * DSP IMPACT AREA: SIU, ACU/RCU                                       *
 * MGB ACTION PLAN: On Rx side, provide proper configuration for RX_STM*
 *              On Tx side, pass parameters as provided by MGB client  *
 ***********************************************************************/
 /* Voice/Fax/Modem packet flow */
#define MGB_CHANNEL_STATE_APP        ifproto_CHANNEL_STATE_APP
 /* Peer-to-peer packet flow */
#define MGB_CHANNEL_STATE_USER_DATA  ifproto_CHANNEL_STATE_USER_DATA
 /* Alarm events packet flow */
#define MGB_CHANNEL_STATE_ALARM      ifproto_CHANNEL_STATE_ALARM
 /* Tone relay events packet flow */
#define MGB_CHANNEL_STATE_TONE       ifproto_CHANNEL_STATE_TONE
 /* CAS signalling packet flow */
#define MGB_CHANNEL_STATE_CAS        ifproto_CHANNEL_STATE_CAS

 /* RTCP packet flow */
#define MGB_CHANNEL_STATE_RTCP       ifproto_CHANNEL_STATE_RTCP
 
 /* MUTE - Signal data replaced with silence */
#define MGB_CHANNEL_STATE_MUTE       ifproto_CHANNEL_STATE_MUTE
 /* TX delay may be activated to eliminate any DTMF leakage */
 #define MGB_CHANNEL_STATE_DELAY      0x0200
 /* Put channel in standby mode - can appear in tele2pkt bitfield ! */
#define MGB_CHANNEL_STATE_STANDBY    ifproto_CHANNEL_STATE_STANDBY

#define MGB_MDS_CHANNEL_STATE        ifproto_in_CHANNEL_STATE

/*******************************************************************************
 * SID_OPTIONS
 *
 * NONE: Suppress voice, but send no SID packets.  This will cause receiver to 
 *       interpret silence as packet loss or late packet arrival until next 
 *       voice packet arrives.  Cannot be combined with ONE, MAX_RATE, 
 *       or MIN_RATE.
 * ONE:  Send exactly one SID per silence interval.  Codecs with internal VADs 
 *       will have extra SIDs suppressed which will break codec bit exactness 
 *       compliance.  Cannot be combined with NONE or MIN_RATE.
 * MAX_RATE: Send SIDs no faster than packetization rate.  This will break 
 *           codec bit exactness compliance when extra SIDs are suppressed.
 *           Cannot be combined with NONE.
 * MIN_RATE: Specifies minimum SID refresh rate.  0 indicates none required.
 *           SID retransmission for LBRCs will break codec bit exactness.
 *           
 * In order preserve codec bit exactness, the sid_options bitmap should be
 * 0, and the MIN_RATE should be 0.  
 ******************************************************************************/
struct MGB_MDS_SID_OPTIONS {
#define MGB_SID_SET_OPTIONS    0x0001
#define MGB_SID_SET_MINTXRATE  0x0002
#define MGB_SID_OPTIONS_VPBT   0x00000011
   tuint  valid_params;      /* bitfield for sid_options and/or min_tx_rate */

   tuint  sid_options;       /* SID transmission options bitfield.            * 
                               * 0x0001: Send no sids;                         *
                               * 0x0002: Send exactly one sid;                 *
                               * 0x0004: Max sid rate limited to vif rate.     */
#define MGB_SID_SEND_NONE  0x0001
#define MGB_SID_SEND_ONE   0x0002
#define MGB_SID_MAX_RATE   0x0004
   tuint  sid_min_tx_rate;   /* minimum rate/maximume time in ms between sids */
} TNI_ATTR_PACK;


/*******************************************************************************
 * Description: These structures defines the statistics for the common Network 
 *              Encapsulation function.
 ******************************************************************************/
#define MGB_DMS_GET_NETENCAP_STAT_LAYER     ifneu_out_GET_NETENCAP_STAT_LAYER

/*******************************************************************************
 * Description: These structures defines the statistics for the packets dropped
 *              because of traffic policing in NEU module 
 *               
 ******************************************************************************/
#define MGB_DMS_GET_NETENCAP_STAT_TC_LAYER  ifneu_out_GET_NETENCAP_STAT_TC_LAYER

/*******************************************************************************
 * Description: These structures defines the statistics for the packets dropped
 *              because of media policing (RTP payload type) in NEU module 
 *               
 ******************************************************************************/
#define MGB_DMS_GET_NETENCAP_STAT_MP_LAYER  ifneu_out_GET_NETENCAP_STAT_MP_LAYER

/* Layer 2 Error Types */

/* L2 General Errors:  These bits are not L2 protocol specific
** The specific reasons these bits are set may differ based on source protocol
** If so this is noted */
#define MGB_NETENCAP_ERR_L2_PROTO       ifneu_NETENCAP_ERR_L2_PROTO  /* Invalid next Protocol Type
                                                ** ALL: not == 0x0800 */
#define MGB_NETENCAP_ERR_L2_CTRL_FIELD  ifneu_NETENCAP_ERR_L2_CTRL_FIELD  /* Invalid control field
                                                ** ETH/SNAP: LLC & SNAP Org code 
                                                **           not == 0xAAAA03000000
                                                ** PPP: addr/ctrl != 0xFF03 */
#define MGB_NETENCAP_ERR_L2_LOC_ADDR    ifneu_NETENCAP_ERR_L2_LOC_ADDR  /* Dest Addr is unicast and does 
                                                ** not match local addr
                                                ** ETH only */
#define MGB_NETENCAP_ERR_L2_BCAST       ifneu_NETENCAP_ERR_L2_BCAST  /* Dest addr is Broadcast
                                                ** ETH only */
#define MGB_NETENCAP_ERR_L2_MCAST       ifneu_NETENCAP_ERR_L2_MCAST  /* Dest addr is Multicast
                                                ** ETH only */
#define MGB_NETENCAP_ERR_L2_REM_ADDR    ifneu_NETENCAP_ERR_L2_REM_ADDR  /* Dest addr is Multicast
                                                ** ETH only */
#define MGB_NETENCAP_ERR_L2_TOO_LONG    ifneu_NETENCAP_ERR_L2_TOO_LONG  /* Layer 2 Length was too big */

#define MGB_NETENCAP_ERR_L2_IFACE_PORT  ifneu_NETENCAP_ERR_L2_IFACE_PORT  /* Layer 2 Packet received from incorrect
                                                      * port of the device
                                                      */

/* SAR level. AAL5 and others possible in future (SSAR/SSTED, IPoAAL2 SAR) */
#define MGB_NETENCAP_ERR_L2_SAR_CRC         ifneu_NETENCAP_ERR_L2_SAR_CRC  /* SAR layer CRC error */
#define MGB_NETENCAP_ERR_L2_SAR_LEN         ifneu_NETENCAP_ERR_L2_SAR_LEN  /* SAR encoded length does not match 
                                                ** expected length range determined from
                                                ** cell count / packet total size */
#if 0 /* Not Yet Implemented */
#define MGB_NETENCAP_ERR_L2_SAR_TIMEOUT     ifneu_NETENCAP_ERR_L2_SAR_TIMEOUT   /* Reassembly timeout */
#endif
#define MGB_NETENCAP_ERR_L2_SAR_HDR_SYNTAX  ifneu_NETENCAP_ERR_L2_SAR_HDR_SYNTAX   /* Header syntax err (bad CID or UUI). */

/* L2 Error Source: Since the DSP can accept multiple L2 protocols concurently
** these bits help isolate the source L2 protocol of the errors.  These bits
** will be set if any of the general errors above were due to the specified
** protocol */
#define MGB_NETENCAP_ERR_L2_ETH             ifneu_NETENCAP_ERR_L2_ETH  /* Any error from Ethernet */
#define MGB_NETENCAP_ERR_L2_PPP             ifneu_NETENCAP_ERR_L2_PPP  /* Any error from PPP */
#define MGB_NETENCAP_ERR_L2_SNAPLLC         ifneu_NETENCAP_ERR_L2_SNAPLLC  /* Any error from SNAP/LLC */

/* Layer 3 Error Types */
#define MGB_NETENCAP_ERR_L3_IP_PROTO        ifneu_NETENCAP_ERR_L3_IP_PROTO  /* Protocol field not UDP */
#define MGB_NETENCAP_ERR_L3_IP_VER          ifneu_NETENCAP_ERR_L3_IP_VER  /* Is not V4 */
#define MGB_NETENCAP_ERR_L3_IP_LOC_ADDR     ifneu_NETENCAP_ERR_L3_IP_LOC_ADDR  /* Destination (Local) IP addr does not match */
#define MGB_NETENCAP_ERR_L3_IP_BCAST        ifneu_NETENCAP_ERR_L3_IP_BCAST  /* Destination IP addr is broadcast */
#define MGB_NETENCAP_ERR_L3_IP_MCAST        ifneu_NETENCAP_ERR_L3_IP_MCAST  /* Destination IP addr is multicast */
#define MGB_NETENCAP_ERR_L3_IP_REM_ADDR     ifneu_NETENCAP_ERR_L3_IP_REM_ADDR  /* Source (Remote) IP addr does not match */
#define MGB_NETENCAP_ERR_L3_IP_LEN          ifneu_NETENCAP_ERR_L3_IP_LEN  /* IP length is > length passed from Layer 2 */
#define MGB_NETENCAP_ERR_L3_IP_HDR_CSUM     ifneu_NETENCAP_ERR_L3_IP_HDR_CSUM  /* Header Checksum incorrect */
#define MGB_NETENCAP_ERR_L3_IP_FRAG         ifneu_NETENCAP_ERR_L3_IP_FRAG  /* Fragmented IP packet */

/* Layer 4 Error Types */
#define MGB_NETENCAP_ERR_L4_UDP_LOC_PORT    ifneu_NETENCAP_ERR_L4_UDP_LOC_PORT  /* Destination (Local) UDP Port does not match */
#define MGB_NETENCAP_ERR_L4_UDP_REM_PORT    ifneu_NETENCAP_ERR_L4_UDP_REM_PORT  /* Source (Remote) UDP port does not match */
#define MGB_NETENCAP_ERR_L4_UDP_LEN         ifneu_NETENCAP_ERR_L4_UDP_LEN  /* UDP length is > length passed from Layer 3 */
#define MGB_NETENCAP_ERR_L4_UDP_CSUM        ifneu_NETENCAP_ERR_L4_UDP_CSUM  /* Checksum incorrect */
#define MGB_NETENCAP_ERR_L4_PDT_PROT        ifneu_NETENCAP_ERR_L4_PDT_PROT  /* Prot diff tag errors */

#define MGB_DMS_GET_NETENCAP_STAT           ifneu_out_GET_NETENCAP_STAT

/*******************************************************************************
 * Description: These structures defines the GLOBAL Packet statistics gathered
 *              by the Network Encapsulation Unit.
 ******************************************************************************/
struct MGB_DMS_GET_GLOBAL_PKT_STAT {
  tuint txPkts_MSW;       /* # of packets sent succesfully by DSP per core */
  tuint txPkts_LSW;
  tuint rxPkts_MSW;       /* # of packets received by DSP per core, includes */
  tuint rxPkts_LSW;       /*  packets with errors as well. */
  tuint rxErrPkts_L2_MSW; /* # of packets received by DSP per core with */
  tuint rxErrPkts_L2_LSW; /*  level 2 errors. */
  tuint rxErrPkts_L3_MSW; /* # of packets received by DSP per core with */
  tuint rxErrPkts_L3_LSW; /*  level 3 errors. */
  tuint rxErrPkts_L4_MSW; /* # of packets received by DSP per core with */
  tuint rxErrPkts_L4_LSW; /*  level 4 errors. */
  tuint rxErrPkts_TC_MSW; /* # of packets received by DSP per core with */
  tuint rxErrPkts_TC_LSW; /*  Bandwidth Control QoS errors. */
  tuint rxErrPktsMP_MSW; /* # of packets received by DSP per core with Media    */
  tuint rxErrPktsMP_LSW; /*  Policing errors  */
  tuint txBufOverflow_MSW; /* # of packets dropped with Transmit buffer overflow  */
  tuint txBufOverflow_LSW; /*  errors  */
  tuint txErrPkts;        /* Used to reflect dropped packets exceeding handled size */
  tuint numNvpPackets;    /* # of channel independent packets forwarded to Host
                              (presently Non Voice Packets) */
  tuint numNvpPktsThrEx;  /* # of channel independent packets dropped due to exceeding the 
                             throttle */
  tuint rxChIndErrPkts;   /* # of channel independent packets (presently Non Voice Packets
                            dropped due to errors */
  tuint numNvpPacketsSend; /* # of channel independent send packets 
                               (presently Non Voice Packets) */
  tuint numTIDcontrol;    /* TID control testing value */
  tuint numTIDcontrol2;   /* TID control testing value */
  tuint numTIDcontrol3;   /* TID control testing value */
};

/*******************************************************************************
 * Bandwidth Control
 *******************************************************************************/
#define MGB_BWC_VALID_CMD              ifneu_BWC_VALID_CMD
#define MGB_BWC_VALID_PEAK_RATE        ifneu_BWC_VALID_PEAK_RATE
#define MGB_BWC_VALID_PEAK_BKTSZE      ifneu_BWC_VALID_PEAK_BKTSZE
#define MGB_BWC_VALID_MAX_PKTSZE       ifneu_BWC_VALID_MAX_PKTSZE

#define MGB_BWC_CMD_MASK               ifneu_BWC_CMD_MASK
#define MGB_BWC_CMD_ENABLE             ifneu_BWC_CMD_ENABLE
#define MGB_BWC_CMD_DISABLE            ifneu_BWC_CMD_DISABLE
                                /* In the case if host sets the parameter 
                                 * to MGB_BWC_CMD_ENABLE and if below config
                                 * parameters are not provided during API call 
                                 * then the parameters would be set as default which is
                                 * the maximum supported value.
                                 */
#define MGB_BWC_CMD_MEDIA_CHK_ENABLE   ifneu_BWC_CMD_MEDIA_CHK_ENABLE 
                                /* The configuration is valid only for NEU transparency 
                                 * mode. When enabled media policing is enabled. In that case
                                 * payload will be terminated at VPU and once validated it will
                                 * go through NEU tranaparent routing
                                 */ 
#define MGB_BWC_CMD_MEDIA_CHK_DISABLE  ifneu_BWC_CMD_MEDIA_CHK_DISABLE
                                /* The configuration is valid only for NEU transparency 
                                 * mode and is the default mode. When disabled the media policing 
                                 * check will be disabled. In that case
                                 * payload go through NEU tranaparent routing without any additional
                                 * check for the media
                                 */ 
#define MGB_MAX_PEAK_RATE_BPS          ifneu_MAX_PEAK_RATE_BPS
#define MGB_MAX_PEAK_BKT_SIZE          ifneu_MAX_PEAK_BKT_SIZE
#define MGB_MAX_ALLOWED_PKT_SIZE       ifneu_MAX_ALLOWED_PKT_SIZE

#define MGB_MDS_BWC_CONFIG             ifneu_in_BWC_CONFIG
#define MGB_MDS_BWC_CONFIG_T           ifneu_in_BWC_CONFIG_t

/*******************************************************************************
 * ANNOUNCEMENTS
 *******************************************************************************/
 
struct MGB_ANNOUNCE_SEG_INFO {
  tuint  addr_msw;     /* Double word address, MSW */
  tuint  addr_lsw;     /* Double word address, LSW */
  tuint  seg_length;   /* Segment length, bytes    */
  tuint  silence;      /* Silence period, ms       */
} TNI_ATTR_PACK;
  
struct MGB_MDS_ANNOUNCE_BEGIN {
  tuint  coding_type;  /* voice coding type (0-255, VCT_xxx from vctypes.h) */
  tuint  direction;    /* tone direction mask, same as for TONE_ON */
   /* MGB_TDIR_RX   0x1      */ /* toward PCM */
   /* MGB_TDIR_TX   0x2      */ /* toward HPI */
   /* MGB_TDIR_RXTX (MGB_TDIR_RX | MGB_TDIR_TX) */
  tuint  vif_size;     /* voice information field size in bits (1-65535) */
  tuint  play_count;   /* Loop count: value of 1 = play once             */
  tuint  nsegs;        /* Total number of playout segments               */
#define MGB_ANU_BEGIN_MAX_SEGS    12  /* Max segments downloadable to DSP */
#define MGB_ANU_BEGIN_SEG_OFFSET   5  /* Message body offset to segment info */
  struct MGB_ANNOUNCE_SEG_INFO segs[MGB_ANU_BEGIN_MAX_SEGS]; /* Seg info     */
} TNI_ATTR_PACK;

struct MGB_MDS_ANNOUNCE_SEG {
  struct MGB_ANNOUNCE_SEG_INFO segs[MGB_ANU_BEGIN_MAX_SEGS];  /* Segment info */
} TNI_ATTR_PACK;


/*********** ECO specific Error Indications for PVP RCU ACU VPU etc ************/
/*                                                                             */
/* RESERVED ERROR CODES: 72 to 99                                              */
/*                                                                             */
/* Please look at structure MGB_DMS_ERROR_INDICATION in mgmicdsp.h             */
/*                                                                             */
/*                                                                             */
/* These are special error codes which will be used the ECOs directly to       */
/* indicate eco specific error messages to the Micro                           */
/*                                                                             */
/* Please keep this section synched up with mgmicdsp.h  ECO specific Error code*/
/* comment section, created for ease of error code search/access               */
/*                                                                             */
/*                                                                             */
/*                                                                             */
/* PVP specific Error codes                                                    */
#define MGB_ERR_PVP_INVALID_PAYLDFRMT          72
#define MGB_ERR_PVP_PAYLDFRMT_NOT_SUPPORTED    73
/* following is valid only for packet cable */
#define MGB_ERR_PVP_DTMF_TX_QUEUE_FULL         74

/*******************************************************************************
 * Quality Alert (QA)
 ******************************************************************************/
/* valid bit map values for QA parameters */
#define MGB_QA_RTCPXR_ENDSYSDLY ifrcu_QA_RTCPXR_ENDSYSDLY /* End System Delay  */
#define MGB_QA_RTCPXR_RTDLY     ifrcu_QA_RTCPXR_RTDLY /* Round Trip Delay  */
#define MGB_QA_RTCPXR_LOSS      ifrcu_QA_RTCPXR_LOSS /* Loss Rate         */
#define MGB_QA_RTCPXR_DISC      ifrcu_QA_RTCPXR_DISC /* Discard Rate      */
#define MGB_QA_RTCPXR_JITTER    ifrcu_QA_RTCPXR_JITTER /* Jitter            */
#define MGB_QA_RTCPXR_MOSLQ     ifrcu_QA_RTCPXR_MOSLQ /* MOS-LQ            */
#define MGB_QA_RTCPXR_MOSCQ     ifrcu_QA_RTCPXR_MOSCQ /* MOS-CQ            */
#define MGB_QA_RTCPXR_RFAC      ifrcu_QA_RTCPXR_RFAC /* R Factor          */
#define MGB_QA_RTCPXR_JBNOM     ifrcu_QA_RTCPXR_JBNOM /* JB Nominal        */
#define MGB_QA_RTCPXR_RERL      ifrcu_QA_RTCPXR_RERL /* RERL              */
#define MGB_QA_RTCPXR_RLQ       ifrcu_QA_RTCPXR_RLQ /* Listening R-Factor */

#define MGB_QA_NUM_MON_PARAMS   ifrcu_QA_NUM_MON_PARAMS   /* MAXIMUM Number of QA parameters that
                                     * could be monitored at the same time */
#define MGB_QA_NUM_PARAMS       ifrcu_QA_NUM_PARAMS  /* Total number of parameters */

#define MGB_QA_DISABLE          ifrcu_QA_DISABLE
#define MGB_QA_ENABLE           ifrcu_QA_ENABLE  

#define MGB_MDS_QA_CONFIG       ifrcu_in_QA_CONFIG

#define MGB_QA_TYPE_CESSATION   0
#define MGB_QA_TYPE_ALERT       1   

#define MGB_DMS_QA_REPORT       ifrcu_out_QA_REPORT


#endif  /* __NETAPI_H__ */

/* nothing past this point */
