/*******************************************************************************
 * FILE PURPOSE: Structures shared between DSP and host for Nx64
 *******************************************************************************
 * FILE NAME:   nx64api.h
 *
 * DESCRIPTION: Defines shared structures for the Nx64 operation of the DSP.
 *              This structures are used by the DSP and HOST, and also
 *              between the DSP modules.
 *
 * TABS: NONE
 *
 * $Id: $
 *
 * REVISION HISTORY:
 *
 * $Log: $
 *
 * (C) Copyright 2005 Texas Instruments Incorporated. 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#ifndef __NX64API_H__
#define __NX64API_H__

#include <ti/mas/iface/ifeco/mgdefs.h>

/* Dummy Define for a Micro Macro */
#ifndef TNI_ATTR_PACK
#define TNI_ATTR_PACK
#endif

/*******************************************************************************
 * Name:        MGB_NX64_ENCAP_
 * Description: Definition of the Nx64 Encapsulation modes
 ******************************************************************************/
#define MGB_NX64_ENCAP_RAW          0    /* Raw Mode (wo/ 32-bit control word) */
#define MGB_NX64_ENCAP_STRUCT_LOCK  1    /* Structure-Locked Encap Mode */
#define MGB_NX64_ENCAP_STRUCT_IND   2    /* Structure-Indicated Encap Mode */
#define MGB_NX64_ENCAP_HDLC         3    /* HDLC Mode */

/* Only the fisrt two encapsulation mode is supported */
#define MGB_NX64_VALID_ENCAP_LAST   MGB_NX64_ENCAP_STRUCT_LOCK 


/*******************************************************************************
 * Name:        MGB_NX64_COMPANDING_
 * Description: Definition of various companding operations for NX64 operation                        
 ******************************************************************************/
#define MGB_NX64_COMPANDING_TRANSPARENT 0   /* Transparent mode (default) */
#define MGB_NX64_COMPANDING_MU2A        1   /* Mu-law to A-law conversion 
                                               in TDM to Network direction */
#define MGB_NX64_COMPANDING_A2MU        2   /* A-law to Mu-law conversion in 
                                               TDM to Network direction */

/*******************************************************************************
 * Name: MGB_NX64_TDM_CONFIG
 * Description: Definition of TDM configuration of NX64 channels
 ******************************************************************************/
struct MGB_NX64_TDM_CONFIG {
   tint  timeslot;         /* TDM timeslot */
  tuint  fill_pattern;     /* specify pattern generated when packet lost is 
                               indicated from network to TDM direction 
                               For HDLC mode, valid values are
                               0x7E: HDLC Flags
                               0xFF: HDLC Idle
                               For non-HDLC mode, any 8-bit pattern may be 
                               specified */
} TNI_ATTR_PACK;

/*******************************************************************************
 * NX64 Configuration Message
 * Description: Definition of configuration and operational options for
 *              NX64 operation
 ******************************************************************************/
struct MGB_MDS_NX64_CONFIG {
  tuint  valid_params;
#define MGB_NX64_CONFIG_VALID_ENCAP_MODE      0x0001  /* encap_mode is valid */
#define MGB_NX64_CONFIG_VALID_COMPANDING      0x0002  /* companding is valid */
#define MGB_NX64_CONFIG_VALID_PKT_THRESH      0x0004  /* pkt_loss_thresh, pkt_recov_thresh  */
#define MGB_NX64_CONFIG_VALID_ALARM_MASK      0x0008  /* alram_report_mask                  */
#define MGB_NX64_CONFIG_VALID_FRAME_LENGTH    0x0010  /* frame length */
#define MGB_NX64_CONFIG_VALID_TSLOTS          0x0020  /* time slot configuration */

  tuint  encap_mode;                       /* Encapsulation mode (see above) */
  tuint  companding;                       /* Companding conversion mode (see above) */
  tuint  pkt_loss_thresh;                  /* number of consecutive lost packets to enter
                                               packet loss state */
  tuint  pkt_recov_thresh;                 /* number of consecutive packets received to exit
                                               packet loss state */   
  tuint  alarm_report_mask;                /* Indicate which alram should be reported to host
                                             * when transition occurs 
                                             * Bit 0: Local packet loss condition
                                             * Bit 1: Remote packet loss condition
                                             * Bit 2: Remote RDI condication 
                                             * Bit 3: Remote AIS condition
                                             */
  tuint  frame_length;                     /* number of PCM samples (frames) per time slot
                                               in an Nx64 data packet (M) */
  tuint  num_tslots;                       /* number of time slots in a PCM frame (N) */
  struct MGB_NX64_TDM_CONFIG tdm_cfg[1];    /* TDM configuration data (variable size array) */
} TNI_ATTR_PACK;

/*******************************************************************************
 * NX64 Alarm Status definitions
 * Description: NX64 Alarm Status Definition definitions. DSP uses these types 
 *              to set and report Nx64 Alram Status
 ******************************************************************************/
struct MGB_MDS_SET_NX64_ALARM {
  tuint  alarm_mask;           /* Indicates which bits are valids */
  tuint  alarm_bitmap;         /* Alarm Bit Map: 
                                 * Bit 0: RDI condication 
                                 * Bit 1: AIS condition
                                 */
#define MGB_MDS_SET_NX64_LOCAL_RDI            0x0001
#define MGB_MDS_SET_NX64_LOCAL_AIS            0x0002
} TNI_ATTR_PACK;

struct MGB_DMS_REPORT_NX64_ALARM {
  tuint  alarm_mask;           /* Indicates which bits have been changed since
                                   the last report in NXU to Control module;
                                   Indicate which bits are valid in control
                                   modle to NXU */
  tuint  alarm_bitmap;         /* Alarm Bit Map: 
                                 * Bit 0: Local PSN enters packet loss state
                                 * Bit 1: Remote PSN enters packet loss state
                                 * Bit 2: RDI condication 
                                 * Bit 3: AIS condition
                                 */
#define MGB_DMS_REPORT_NX64_LOCAL_PKT_LOSS    0x0001
#define MGB_DMS_REPORT_NX64_REMOTE_PKT_LOSS   0x0002
#define MGB_DMS_REPORT_NX64_REMOTE_RDI        0x0004
#define MGB_DMS_REPORT_NX64_REMOTE_AIS        0x0008
} TNI_ATTR_PACK;

/******************************************************************************
 * NX64 statistics
 *****************************************************************************/
struct MGB_DMS_NX64_STAT {
  
  /* number of sequence number discontinuity detected   */
  tuint  seqnum_discont_cnt;
 
  /* number of times the remote side enters packet loss state */
  tuint  remote_pkt_loss;
 
  /* number of times the local side enters packet loss state */
  tuint  local_pkt_loss;
                     
  /* number of times the remote side enters RDI condition */                               
  tuint  remote_rdi;
 
  /* number of times the local side enters RDI condition */         
  tuint  local_rdi;
 
  /* number of times the remote side enters AIS condition */                               
  tuint  remote_ais;
 
  /* number of times the local side enters AIS condition */         
  tuint  local_ais;
 
  /* number of packets received from network with invalid payload size */
  tuint  invalid_payload_size;
 
} TNI_ATTR_PACK;

#endif  /* __NX64API_H__ */

/* nothing past this point */
