/******************************************************************************
 * FILE PURPOSE: Structures shared between DSP and host for TFO
 ******************************************************************************
 * FILE NAME:   tfoapi.h
 *
 * DESCRIPTION: Defines shared structures for the TFO modules of the DSP.
 *              This structures are used by the DSP and HOST, and also
 *              between the DSP modules.
 *
 * TABS: NONE
 *
 * $Id: $
 *
 * REVISION HISTORY:
 *
 * $Log: $
 *
 * (C) Copyright 2004 Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#ifndef __TFOAPI_H__
#define __TFOAPI_H__

#include <ti/mas/iface/ifeco/mgdefs.h>

/* Dummy Define for a Micro Macro */
#ifndef TNI_ATTR_PACK
#define TNI_ATTR_PACK
#endif

/*****************************************************************************
 * TFO Opcodes (bit map)
 * Description: Definition of various opcodes for TFO unit                        
 *****************************************************************************/
 #define MGB_TFO_OPCODE_DISABLE    0x00  
 #define MGB_TFO_OPCODE_ENABLE     0x01  

/*****************************************************************************
 * TFO States (bit map)
 * Description: Definition of various states for TFO unit                        
 *****************************************************************************/
 #define MGB_TFO_STATE_NORMAL    0x00  /* In Normal(non-TFO) state */
 #define MGB_TFO_STATE_PRESYNCH  0x01  /* in negotiation phase */
 #define MGB_TFO_STATE_TRANS     0x02  /* In TFO state */
 
 
/*****************************************************************************
 * TFO Direction  (bit map) 
 *****************************************************************************/
 #define MGB_TFO_DIR_TOTELE      0x01 
 #define MGB_TFO_DIR_TONET       0x02 
 
 
/*****************************************************************************
 * TFO Type  (bit map) 
 *****************************************************************************/
#define MGB_TFO_TYPE_NONE        0x00
#define MGB_TFO_TYPE_INPATH      0x01
#define MGB_TFO_TYPE_ACTIVE      0x02
#define MGB_TFO_TYPE_INPATH_IP   0x03  /* TFO-over-IP 
                                        * The pre-requisite for this mode is 
                                        * that conferencing should not be
                                        * active for the channel
                                        */
/******************************************************************************
 * TFO IPE and TFO over IP standard terminology 
 *
 * IS_Message - In-band Signaling Message
 * IS_Header  - Header of the IS_Message
 * KOI        - Keep Open Indication (indicates continued TFO connection)
 * IS_REQ     - IS_REQ message
 * IS_ACK     - IS_ACK message
 * IS_TRANS   - IS_IPE message with request for Transparent path (indicates 
 *              start of TFO connection
 * IS_NORMAL  - IS_IPE message with indicates termination of TFO connection
 * IS_FILL    - IS_FILL message
 * IS_DUP     - IS_DUP message
 * IS_SYL     - IS_SYL message
 * SYNC_LOST  - IS_Message sync lost
 * KOI_SYN_LOST - Keep Open indication sync lost
 *****************************************************************************/
 
typedef struct MGB_TFO_CODEC_DESC {
#define MGB_TFO_CODEC_DESC_VALID_CODECTYPE_MASK  0x0001
#define MGB_TFO_CODEC_DESC_VALID_ACS_MASK        0x0002
#define MGB_TFO_CODEC_DESC_VALID_SCS_MASK 	     0x0004
#define MGB_TFO_CODEC_DESC_VALID_PREF_CFG_MASK   0x0008
#define MGB_TFO_CODEC_DESC_VALID_MACS_MASK       0x0010
#define MGB_TFO_CODEC_DESC_VALID_OM_MASK         0x0020

#define MGB_TFO_CODEC_DESC_VALID_CODECTYPE_SHIFT  0
#define MGB_TFO_CODEC_DESC_VALID_ACS_SHIFT        1
#define MGB_TFO_CODEC_DESC_VALID_SCS_SHIFT	  2
#define MGB_TFO_CODEC_DESC_VALID_PREF_CFG_SHIFT   3
#define MGB_TFO_CODEC_DESC_VALID_MACS_SHIFT       4
#define MGB_TFO_CODEC_DESC_VALID_OM_SHIFT         5



  tuint valid_params;


#define MGB_TFO_GSM_FR       0
#define MGB_TFO_GSM_HR       1
#define MGB_TFO_GSM_EFR      2
#define MGB_TFO_FR_AMR       3
#define MGB_TFO_HR_AMR       4
#define MGB_TFO_UMTS_AMR     5
#define MGB_TFO_UMTS_AMR_2   6
#define MGB_TFO_FR_AMR_WB    9
#define MGB_TFO_UMTS_AMR_WB  10
#define MGB_TFO_OHR_AMR      11
#define MGB_TFO_OFR_AMR_WB   12

#define MGB_TFO_CDMA_Q8        0
#define MGB_TFO_CDMA_EVRC      1
#define MGB_TFO_CDMA_Q13       2
#define MGB_TFO_CDMA_SMV       3

  tuint codec_type;
     /* 4bits Coding type values(above) as per 3GPP 26.103[6.3] */
  tuint acs;          /* 9 bits Active codec set, defined for NB-AMR, 
                                      * 8 bits in length(LSB) coding as per 
                                      * 3GPP 28.062 section 7.11.3.1.1,
                                      */
  tuint scs;         /* 9 bits Supported Codec Set, defined for NB-AMR,
                                      * coding same as ACS 
                                      */
  tuint pref_cfg;     /* 4 bits Preferred codec configuration,
                                      *  For NB-AMR values are {0-15}, 
                                      * Please refer to 3GPP 28.062 [7.11.3.1.3-2] 
                                      * For WB-AMR values are {0-5},
                                      * Please refer to 3GPP 26.103 [Table 5.7-1] 
                                      */
  tuint om;           /*1 bit 1-Optimization of ACS allowed, 0- Not Allowed.
                                      * Please refer to 3GPP 26.103 [5.4] for details 
                                      */
  tuint macs;          /* 4 bits Maximum allowed codec mode in the active codec set. 
                                      * For NB-AMR the values are {0-8},AMR-WB{0-4} 
                                      */
} MGB_TFO_CODEC_DESC_T;

#define MGB_TFO_NUM_LOCAL_CODECS 1
typedef struct  MGB_TFO_LOCAL_CODEC_CONFIG {
  tuint num_codecs;
  MGB_TFO_CODEC_DESC_T codec_descs[MGB_TFO_NUM_LOCAL_CODECS];
}MGB_TFO_LOCAL_CODEC_CONFIG_T;

/*****************************************************************************
 * TFO Configuration Message                                
 *                                                         
 * Description: Definition of configuration and operational options for
 *              Tandem Free Operation[TFO] unit                        
 *                                                                     
 *****************************************************************************/
struct MGB_MDS_TFO_CONFIG {

/*
 *  TFO operation control
 *
 *   15  14  13  12  11  10   9   8   7   6   5   4   3   2   1   0
 *  +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
 *  |        Reserved       |  DIR  |     STATE     |  TYPE | OPCODE|
 *  +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
 *
 *   OPCODE: Enable/Disable control
 *            0 - Disable
 *            1 - Enable
 *
 *   TYPE  : IPE, Active or TFO-over-IP configuration See Above MGB_TFO_TYPE_T
 *
 *   STATE : Begin state (normally for a standby channel). DSP simply makes the 
 *           state machine to enter this specified state. If no state information 
 *           is provided, DSP starts the state machine and start the negotiation 
 *           process. See above MGB_TFO_STATE_T 
 *
 *   DIR: Direction. See above MGB_TFO_DIR_T
 */
 tuint   cntl_valid_param;
 tuint   cntl_bfield;
 
 /* valid bit masks */ 
#define MGB_TFO_CNTL_OPCODE_MASK        0x0003
#define MGB_TFO_CNTL_TYPE_MASK          0x000C
#define MGB_TFO_CNTL_STATE_MASK         0x00F0
#define MGB_TFO_CNTL_DIR_MASK           0x0300
#define MGB_TFO_CNTL_CODECCFG_MASK      0x0400

 /* cntrl bfields */
#define MGB_TFO_CNTL_OPCODE_DISABLE     0
#define MGB_TFO_CNTL_OPCODE_ENABLE      1

 
/*
 *  TFO options
 *  
 *   15  14  13  12  11  10   9   8   7   6   5   4   3   2   1   0
 *  +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
 *  |           Version             | Rsrvd | EVNTIP|  SMM  |STD|CMR|
 *  +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
 *
 *  TFO Options
 *  CMR: Codec Mismatch Resolution
 *            0 - Disable
 *            1 - Enable
 *  STD: Standard specification
 *            0 - GSM/UMTS
 *            1 - CDMA (Currently not supported)
 *
 *  SMM : Standalone Mode/Auto operate Mode options. DSP adjust required
 *      voice path params [gain and EC] while switching in-out of
 *      TFO operatio itself, without interaction with host. DSP can be confgured
 *      to optionaly indicate status, but Host does not take any action
 *      unless marked action requied (as in the case of non-standalone mode)
 *
 *  EVNTIP : Events reporting for TFO-over-IP. This enables/disables event reporting
 *           and of optional events that does not require action from Host. Note
 *           that currently event reporting should be mandatorily enabled, but 
 *           optional events could be disabled
 *
 *  Version: Version 
 *            b0-b3 : Subversion 
 *            b7-b4 : Version 
 *            0 - Latest version
 *          1-3 - Not supported for GSM/UMTS
 *           >3 - Specific version (for ex: 4)
 *
 *
 */
 tuint   option_valid_param;
 tuint   option_bfield;
 
 /* valid bit masks */ 
#define MGB_TFO_OPTION_CMR_MASK           0x0001
#define MGB_TFO_OPTION_STD_MASK           0x0002
#define MGB_TFO_OPTION_SM_MASK            0x000C
#define MGB_TFO_OPTION_EVNTIP_MASK        0x0030
#define MGB_TFO_OPTION_VER_MASK           0xFF00

 /* bit field values */
#define MGB_TFO_OPTION_CMR_DISABLE        0x0000   /* enable codec mismtach resole o/n */
#define MGB_TFO_OPTION_CMR_ENABLE         0x0001

#define MGB_TFO_OPTION_STD_GSMUMTS        0x0001   /* protocol state machine selection */
#define MGB_TFO_OPTION_STD_CDMA           0x0002
 
#define MGB_TFO_OPTION_SM_DISABLE         0x0000
#define MGB_TFO_OPTION_SM_NOTIFY_NONE     0x0001   /* SMM disable notifications */
#define MGB_TFO_OPTION_SM_NOTIFY_ALL      0x0002   /* SMM enable notifications */
 
/* Currently either MGB_TFO_OPTION_EVNTIP_MANDATORY or 
 * MGB_TFO_OPTION_EVNTIP_ALL has to be enabled, as DSP
 * relies on the Host to take some actions 
 */
#define MGB_TFO_OPTION_EVNTIP_MANDATORY     0x0000 /* Enable mandatory 
                                                    * event reporting (that 
                                                    * require action from the Host)
                                                    */  
#define MGB_TFO_OPTION_EVNTIP_ALL           0x0001  /* Enable all events being
                                                     * reported to host
                                                     */
#define MGB_TFO_OPTION_EVNTAE_NONE         0x0000 /* Basic/(ON/OFF) or TRANS/NORMAL */
/* Event reportign for AE */
#define MGB_TFO_OPTION_EVNTAE_BASIC         0x0001 /* Basic/(ON/OFF) or TRANS/NORMAL 
                                                    */  
#define MGB_TFO_OPTION_EVNTAE_MSGS          0x0002  /* Messages received from distant side
                                                     */
#define MGB_TFO_OPTION_EVNTAE_STATE_CHANGE  0x0003  /* State m/c Changes(GSM/CDMA) as per standard
                                                     */
 
 /* Usefull Macro to extract above bit fields */
#define MGB_TFO_CNTL_OPCODE_SHIFT        0
#define MGB_TFO_CNTL_TYPE_SHIFT          2
#define MGB_TFO_CNTL_STATE_SHIFT         4
#define MGB_TFO_CNTL_DIR_SHIFT           8

#define MGB_TFO_OPTION_CMR_SHIFT         0
#define MGB_TFO_OPTION_STD_SHIFT         1
#define MGB_TFO_OPTION_SMM_SHIFT         2
#define MGB_TFO_OPTION_EVNTIP_SHIFT      4
#define MGB_TFO_OPTION_VER_SHIFT         8

 MGB_TFO_LOCAL_CODEC_CONFIG_T local_codec_config;
} TNI_ATTR_PACK;




/******************************************************************************
 * TFO Peer-Peer packet definitions.                                 
 *                                                         
 * Description: TFO Peer-Peer packet definitions. DSP uses these types to 
 *              indicate TFO protocol status
 *                                                                     
 ******************************************************************************/
typedef struct {
  tuint  bitmap;
  
  tuint  reserved1;
  tuint  reserved2;
  tuint  reserved3;             
  tuint  reserved4;
} TNI_ATTR_PACK MGB_MDS_SEND_PACKET_PROTO_TFO;


typedef struct {
    tuint  event;                              /* Event indication */

/* MGB_TFO_OPTION_EVNTIP_MANDATORY event reports - Requires action from Host */
#define MGB_TFO_OPTION_EVNTIP_MANDATORY_BASE    0
#define MGB_TFO_IP_EVNT_IS_HEADER_DET    (MGB_TFO_OPTION_EVNTIP_MANDATORY_BASE + 1)     
                                               /* Event: First IS_Header detected
                                                * Expected Action:
                                                *  Configure the channel in
                                                *  clear channel (VBD) mode which would
                                                *  include upspeeding to PCM if 
                                                *  existing codec is LBRC,
                                                *  Disabling ECAN,DC Offset,
                                                *  gain etc.
                                                */

#define MGB_TFO_IP_EVNT_KOI_SYN_LOST_CNF  (MGB_TFO_OPTION_EVNTIP_MANDATORY_BASE + 2)    
                                               /* Event: Confirmed 
                                                * Keep_Open_Indication sync lost 
                                                * Expected Action:
                                                *   At this point application
                                                *   can configure channel back 
                                                *   from clear channel to 
                                                *   normal (pre-TFO_IP) mode
                                                */

#define MGB_TFO_IP_EVNT_TFO_TERM          (MGB_TFO_OPTION_EVNTIP_MANDATORY_BASE + 3)      
                                               /* Event: TFO connection 
                                                *        terminated 
                                                * Expected Action:
                                                *   At this point application
                                                *   can configure channel back 
                                                *   from clear channel to 
                                                *   normal (pre-TFO_IP) mode
                                                */

/* Optional event reports - No specific Action required from host
 * These are for information only. These events are
 * sent only if MGB_TFO_OPTION_EVNTIP_ALL is enabled 
 */
#define MGB_TFO_IP_EVNT_OPTIONAL_BASE   0x1000
#define MGB_TFO_IP_EVNT_IS_REQ          (MGB_TFO_IP_EVNT_OPTIONAL_BASE + 1)
                                            /* IS_REQ message received */

#define MGB_TFO_IP_EVNT_IS_ACK          (MGB_TFO_IP_EVNT_OPTIONAL_BASE + 2)
                                            /* IS_ACK message received */

#define MGB_TFO_IP_EVNT_IS_TRANS        (MGB_TFO_IP_EVNT_OPTIONAL_BASE + 3)
                                            /* IS_IPE TRANS message received */

#define MGB_TFO_IP_EVNT_IS_NORMAL       (MGB_TFO_IP_EVNT_OPTIONAL_BASE + 4)
                                            /* IS_IPE NORMAL message received */

#define MGB_TFO_IP_EVNT_IS_FILL         (MGB_TFO_IP_EVNT_OPTIONAL_BASE + 5)
                                            /* IS_FILL message received */

#define MGB_TFO_IP_EVNT_IS_DUP          (MGB_TFO_IP_EVNT_OPTIONAL_BASE + 6)
                                            /* IS_DUP message received */

#define MGB_TFO_IP_EVNT_IS_SYL          (MGB_TFO_IP_EVNT_OPTIONAL_BASE + 7)
                                            /* IS_SYL message received */

#define MGB_TFO_IP_EVNT_TFO_CONNECT     (MGB_TFO_IP_EVNT_OPTIONAL_BASE + 8)
                                            /* TFO Connection established */

#define MGB_TFO_IP_EVNT_KOI_SYN_LOST    (MGB_TFO_IP_EVNT_OPTIONAL_BASE + 9)  
                                            /* Keep_Open_Indication - sync loss
                                             * loss occurence found. 
                                             * DSP starts a timer 
                                             * and sends 
                                             * MGB_TFO_IP_EVNT_KOI_SYN_LOST_CNF
                                             * if sync loss continues until 
                                             * timeout occurence
                                             */

}MGB_DMS_RECEIVE_PACKET_PROTO_TFO_IP_BODY;

typedef struct {

  /* valid params */
#define MGB_TFO_AE_REP_VALID_BASIC      0x0001
#define MGB_TFO_AE_REP_VALID_MSG_EVENT  0x0002
#define MGB_TFO_AE_REP_VALID_STATE      0x0004

    tuint  valid_params;                              

#define MGB_TFO_AE_EVENT_TFO_DISCONNECTED        0 
#define MGB_TFO_AE_EVENT_TFO_ESTABLISHED         1
    tuint  event_basic;                              /* Message Event indication */


/* All the above messages are applicable to AE as well */
#define MGB_TFO_AE_EVNT_IS_HEADER_DET  MGB_TFO_IP_EVNT_IS_HEADER_DET
#define MGB_TFO_AE_EVNT_KOI_SYN_LOST_CNF  MGB_TFO_IP_EVNT_KOI_SYN_LOST_CNF    
#define MGB_TFO_AE_EVNT_TFO_TERM       MGB_TFO_IP_EVNT_TFO_TERM

#define MGB_TFO_AE_EVNT_MSGS_BASE   MGB_TFO_IP_EVNT_OPTIONAL_BASE
  
#define MGB_TFO_AE_EVNT_IS_REQ         MGB_TFO_IP_EVNT_IS_REQ
#define MGB_TFO_AE_EVNT_IS_ACK         MGB_TFO_IP_EVNT_IS_ACK 
#define MGB_TFO_AE_EVNT_IS_TRANS       MGB_TFO_IP_EVNT_IS_TRANS
#define MGB_TFO_AE_EVNT_IS_NORMAL	   MGB_TFO_IP_EVNT_IS_NORMAL
#define MGB_TFO_AE_EVNT_IS_FILL        MGB_TFO_IP_EVNT_IS_FILL
#define MGB_TFO_AE_EVNT_IS_DUP         MGB_TFO_IP_EVNT_IS_DUP
#define MGB_TFO_AE_EVNT_IS_SYL         MGB_TFO_IP_EVNT_IS_SYL
#define MGB_TFO_AE_EVNT_TFO_CONNECT    MGB_TFO_IP_EVNT_TFO_CONNECT 
#define MGB_TFO_AE_EVNT_KOI_SYN_LOST   MGB_TFO_IP_EVNT_KOI_SYN_LOST
   tuint  msg_event;                              /* Message Event indication */



#define MGB_TFO_AE_GSM_NAC       1   /* Not_Active */
#define MGB_TFO_AE_GSM_WAK       2   /* Wakeup */   
#define MGB_TFO_AE_GSM_FIT       3   /* First_Try */
#define MGB_TFO_AE_GSM_COR       4   /* Continuous_Retry */
#define MGB_TFO_AE_GSM_PER       5   /* Periodic_Retry */
#define MGB_TFO_AE_GSM_MON       6   /* Monitor */
#define MGB_TFO_AE_GSM_MIS       7   /* Mismatch */
#define MGB_TFO_AE_GSM_CON       8   /* Contact */
#define MGB_TFO_AE_GSM_FAT       9   /* Fast_Try */
#define MGB_TFO_AE_GSM_FAC      10   /* Fast_Contact */
#define MGB_TFO_AE_GSM_WRC      11   /* Wait_RC */
#define MGB_TFO_AE_GSM_KON      12   /* Konnect */
#define MGB_TFO_AE_GSM_REK      13   /* Re_Konnect */
#define MGB_TFO_AE_GSM_SOS      14   /* Sync_Lost */
#define MGB_TFO_AE_GSM_OPE      15   /* Operation */
#define MGB_TFO_AE_GSM_FAI      16   /* Failure */	
#define MGB_TFO_AE_GSM_TT       17   /* TFO_Term */
 tuint  state;                       /* New State */

}MGB_DMS_RECEIVE_PACKET_PROTO_TFO_AE_BODY;

union MGB_DMS_RECEIVE_PACKET_PROTO_TFO_BODY {
    MGB_DMS_RECEIVE_PACKET_PROTO_TFO_IP_BODY    tfo_ip;
    MGB_DMS_RECEIVE_PACKET_PROTO_TFO_AE_BODY    tfo_ae;
} TNI_ATTR_PACK;

typedef struct {
  tuint  event_type;                        /* Control Bitmap */ 
#define MGB_RECEIVE_TFO_IP_EVENT         1   /* TFO_IP Event */
#define MGB_RECEIVE_TFO_AE_EVENT         2   /* TFO_AE Event */

  union MGB_DMS_RECEIVE_PACKET_PROTO_TFO_BODY   body;
} TNI_ATTR_PACK MGB_DMS_RECEIVE_PACKET_PROTO_TFO;


/******************************************************************************
 * TFO Violation Reports
 *
 * Description: DSP reprots TFO errors usign these violation reports. These
 *              errors could be state machine errors or generic errors.
 *****************************************************************************/
 typedef tuint MGB_TFO_VIOLATION_ERRORS_T;
 
 /* Codec mismatch detected */
 #define MGB_TFO_VIOLATION_CODEC_MIS    (MGB_TFO_VIOLATION_ERRORS_T)0x0001
 
typedef struct  {

  MGB_TFO_CODEC_DESC_T dis_codec;

}MGB_TFO_VIOLATION_CODEC_MIS_T;


typedef struct  {

  tuint rate;/* The invalid rate which caused the error */

}MGB_TFO_VIOLATION_INVALID_RATE_T;

/******************************************************************************
 * Name: MGB_DMS_VIOLATION_TFO_BODY
 * Description: Union of IU violation report types. For now only Error Events
 * that are filtered are reported. Filtered means consolodated and throttled
 * as opposed to reported individually.
 ******************************************************************************/
union MGB_DMS_VIOLATION_TFO_BODY {
    MGB_TFO_VIOLATION_ERRORS_T    errors;
    MGB_TFO_VIOLATION_CODEC_MIS_T codec_mismatch;
    MGB_TFO_VIOLATION_INVALID_RATE_T invalid_rate;
} TNI_ATTR_PACK;

struct MGB_DMS_VIOLATION_TFO {
  tuint    type;
#define MGB_TFO_VIOLATION_TYPE_CODEC_MISMATCH        0x0001
#define MGB_TFO_VIOLATION_TYPE_INVALID_RATE_FROM_NET 0x0002 /* If the packet received
                                                               from the network interface does
                                                               not belong to the allowed rates(ACS)
                                                             */

  union MGB_DMS_VIOLATION_TFO_BODY body;
} TNI_ATTR_PACK; 
 
 
 
/******************************************************************************
 * TFO statistics
 *****************************************************************************/
struct MGB_DMS_TFO_STAT {

 /*
  *  TFO State and Type
  *  
  *   15  14  13  12  11  10   9   8   7   6   5   4   3   2   1   0
  *  +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
  *  |                        Reserved               | STATE |  TYPE |
  *  +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
  *
  *  STATE, TTYPE : Fields have the same meaning as above.
  *
  */
  tuint  op_mode;
  /* valid bit masks */ 
#define MGB_TFO_STAT_TYPE_MASK           0x0003
#define MGB_TFO_STAT_STATE_MASK          0x000C
  /* Usefull Macro to extract above bit fields */
#define MGB_TFO_STAT_TYPE_SHIFT         0
#define MGB_TFO_STAT_STATE_SHIFT        2
  
 /*
  *  TFO Version details
  *  
  *   15  14  13  12  11  10   9   8   7   6   5   4   3   2   1   0
  *  +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
  *  |            CUR VER            |             CFG VER           |
  *  +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
  *
  *  CFG VER : Configured version
  *
  *  CUR VER : Current version
  */
  tuint   version; 
  
#define MGB_TFO_STAT_CFGVER_MASK         0x00FF
#define MGB_TFO_STAT_CURVER_MASK         0xFF00
  /* Usefull Macro to extract above bit fields */
#define MGB_TFO_STAT_CFGVER_SHIFT       0
#define MGB_TFO_STAT_CURVER_SHIFT       8
 
  tuint   rem_codec;        /* Distant codec type */
  
  /* TFO Frames count. This gives the number of IS messages if the TFO is in InPath 
   * equipment mode, and this gives the number of TFO frames while in Active mode
   */
  tuint   frames_from_net_MSW;  
  tuint   frames_from_net_LSW;  
  
  tuint   frames_to_net_MSW;
  tuint   frames_to_net_LSW;
  
  /* Error Count. This gives the number of errors received in IS messages
   * while operating in InPath Equipment mode
   */
  tuint   ismsg_err_cnt;
  
  /* Num of IS msgs "dropped" due to large number of errors in the 
   * message
   */
  tuint   ismsg_dropped;

  /* Frame Format used by the TFO to send code words 
   * Helps in identifying 32K or 16K format used for different wb-amr rates
   */
#define MGB_TFO_STAT_FFRMT_AMR16K    0
#define MGB_TFO_STAT_FFRMT_AMR8P8K   1
#define MGB_TFO_STAT_FFRMT_FREFR     2
#define MGB_TFO_STAT_FFRMT_EVRC      3
#define MGB_TFO_STAT_FFRMT_SMV       4
#define MGB_TFO_STAT_FFRMT_WBAMR16K  5  /* AMR_WB_TFO_16k  */
#define MGB_TFO_STAT_FFRMT_WBAMR32K  6  /* AMR_WB_TFO_32k */
#define MGB_TFO_STAT_FFRMT_EVRCB     7
  tuint   frame_format;

} TNI_ATTR_PACK;


#endif  /* __TFOAPI_H__ */

/* nothing past this point */
