#ifndef _DATAPKT_INCLUDED
#define _DATAPKT_INCLUDED
/*****************************************************************************
 * FILE PURPOSE:    Generic Data Packet definitions
 *****************************************************************************
 * FILE NAME:       datapkt.h
 *
 * DESCRIPTION:     This file contains definitions that deal with the data
 *                  packet type.
 *
 *                  The header fields of the data packet are described below.
 *
 *                      Header Byte 0:
 *                          Bit 7    : Start-of-Frame Bit
 *                          Bit 6    : DE Bit
 *                          Bit 5 - 0: Packet Data Type
 *
 *                      Header Byte 1:
 *                          Bit 7 - 5: End-of-Frame Status
 *                          Bit 4 - 0: Tx Sequence Number
 *
 *                      Header Byte 2 (reverse path info):
 *                          Bit 7 - 5: Number of Free Buffers
 *                          Bit 4 - 0: Last Played Out Sequence Number
 *
 *                      Header Byte 3 (reverse path info):
 *                          Bit 7 - 5: Reserved (zeros)
 *                          Bit 4 - 0: Last Received Sequence Number
 *
 * NOTE:            This file is processor independent, thus no assumption
 *                  of word size should be made. The data is interpreted
 *                  as a byte stream.
 *
 * (C) Copyright 2002, Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/****************************************************************************/
/*              General Packet Header Definitions                           */
/****************************************************************************/

#define DATAPKT_HDR_LEN_B           4       /* Header length in bytes       */
#define DATAPKT_MAX_PAYLOAD_B       278     /* Max payload length in bytes  */
#define DATAPKT_INIT_SEQ_NUM        1       /* Initial Tx/Rx seq num        */

/* Byte offsets */
#define DATAPKT_BO_SOF              0       /* Start of frame BO            */
#define DATAPKT_BO_DE               0       /* Discard eligible BO          */
#define DATAPKT_BO_TYPE             0       /* Packet data type BO          */
#define DATAPKT_BO_EOF              1       /* End of frame BO              */
#define DATAPKT_BO_TX_SN            1       /* Tx sequence number BO        */
#define DATAPKT_BO_FREE_BUF         2       /* Number of free buffers BO    */
#define DATAPKT_BO_LAST_PO_SN       2       /* Last played out seq num BO   */
#define DATAPKT_BO_LAST_RX_SN       3       /* Last received seq num BO     */

/* Byte masks */
#define DATAPKT_MASK_SOF            0x80    /* Start of frame mask          */
#define DATAPKT_MASK_DE             0x40    /* Discard eligible mask        */
#define DATAPKT_MASK_TYPE           0x3F    /* Packet data type mask        */
#define DATAPKT_MASK_EOF            0xE0    /* End of frame mask            */
#define DATAPKT_MASK_TX_SN          0x1F    /* Tx sequence number mask      */
#define DATAPKT_MASK_FREE_BUF       0xE0    /* Number of free buffers mask  */
#define DATAPKT_MASK_LAST_PO_SN     0x1F    /* Last played out seq num mask */
#define DATAPKT_MASK_LAST_RX_SN     0x1F    /* Last received seq num mask   */

/* Bit shifts within byte */
#define DATAPKT_SHIFT_SOF           7       /* Start of frame shift         */
#define DATAPKT_SHIFT_DE            6       /* Discard eligible shift       */
#define DATAPKT_SHIFT_TYPE          0       /* Packet data type shift       */
#define DATAPKT_SHIFT_EOF           5       /* End of frame shift           */
#define DATAPKT_SHIFT_TX_SN         0       /* Tx sequence number shift     */
#define DATAPKT_SHIFT_FREE_BUF      5       /* Number of free buffers shift */
#define DATAPKT_SHIFT_LAST_PO_SN    0       /* Last played out seq num shift*/
#define DATAPKT_SHIFT_LAST_RX_SN    0       /* Last received seq num shift  */

/* Data Types */
#define DATAPKT_DATATYPE_RAW        0       /* Raw data                     */
#define DATAPKT_DATATYPE_PPP_PKT    1       /* PPP packet                   */
#define DATAPKT_DATATYPE_SLIP_PKT   2       /* SLIP packet                  */
#define DATAPKT_DATATYPE_HDLC       3       /* HDLC frame                   */
#define DATAPKT_DATATYPE_RAW_ESC    4       /* Raw escape                   */
#define DATAPKT_DATATYPE_MTP        5       /* MTP                          */
#define DATAPKT_DATATYPE_FAX_CMD    10      /* Fax in-band command          */
#define DATAPKT_DATATYPE_FAX_STATUS 11      /* Fax in-band status           */
#define DATAPKT_DATATYPE_RES_31     31      /* Reserved value               */
#define DATAPKT_DATATYPE_CTRL_PKT   63      /* In-band signaling ctrl pkt   */

/* End-of-Frame Status */
#define DATAPKT_EOF_CONTINUE        0       /* Frame in progress            */
#define DATAPKT_EOF_END_GOOD        1       /* End of frame - good CRC      */
#define DATAPKT_EOF_END_BAD         2       /* End of frame - bad CRC       */
#define DATAPKT_EOF_ABORT           3       /* Abort frame                  */

/****************************************************************************/
/*              Control Packet Definitions                                  */
/****************************************************************************/

/* Control Packet IDs */
#define DATAPKT_CTRL_TX_MODE_CHANGE     0   /* Tx Mode change               */
#define DATAPKT_CTRL_FLOW_CTRL_INFO     1   /* Flow control info            */
#define DATAPKT_CTRL_FLUSH_Q            2   /* Flush transmit queue         */
#define DATAPKT_CTRL_HDLC_SYNC_INFO     3   /* HDLC synchronization info    */
#define DATAPKT_CTRL_TERMINATE_ACK      4   /* Terminate Acknowledgment     */

/* Control Packet Lengths */
#define DATAPKT_CTRL_PKT_MAX_LEN_B      4   /* Max ctrl pkt length (bytes)  */
#define DATAPKT_CTRL_PKT_ID_LEN_B       1   /* Length of ctl pkt ID (bytes) */
#define DATAPKT_CTRL_TXMODE_LEN_B       3   /* Tx Mode msg (bytes)          */
#define DATAPKT_CTRL_FCTRL_LEN_B        3   /* Flow Control Info msg (bytes)*/
#define DATAPKT_CTRL_FLUSHQ_LEN_B       3   /* Flush Queue msg (bytes)      */
#define DATAPKT_CTRL_HDLCSYNC_LEN_B     1   /* HDLC Sync msg length (bytes) */

/* Mode Change Parameters */
#define DATAPKT_CTRL_TXMODE_BO_CMD      0   /* Command byte offset          */
#define DATAPKT_CTRL_TXMODE_REQ         0   /* Tx mode change cmd: request  */
#define DATAPKT_CTRL_TXMODE_RESP        1   /* Tx mode change cmd: response */

#define DATAPKT_CTRL_TXMODE_BO_NEW_MODE 1   /* New mode byte offset         */
#define DATAPKT_CTRL_TXMODE_BO_MIN_FLGS 2   /* Min flags byte offset        */

/* Flow Control Info Parameters */
#define DATAPKT_CTRL_FCTRL_BO_MAXPKT_MS 0   /* Max pkt size byte offset, MS */
#define DATAPKT_CTRL_FCTRL_BO_MAXPKT_LS 1   /* Max pkt size byte offset, LS */
#define DATAPKT_CTRL_FCTRL_BO_MAX_BUF   2   /* Max buffers byte offset      */

/* Flush Queue Parameters */
#define DATAPKT_CTRL_FLUSHQ_BO_CMD      0   /* Command byte offset          */
#define DATAPKT_CTRL_FLUSHQ_ON          1   /* Command: flush on            */
#define DATAPKT_CTRL_FLUSHQ_OFF         0   /* Command: flush off           */

#define DATAPKT_CTRL_FLUSHQ_BO_TYPE     1   /* Drop type byte offset        */
#define DATAPKT_CTRL_FLUSHQ_DESTRUCTIVE 1   /* Type: destructive            */
#define DATAPKT_CTRL_FLUSHQ_GRACEFUL    2   /* Type: graceful               */

#define DATAPKT_CTRL_FLUSHQ_BO_DROP_OPT 2   /* Drop option byte offset      */
#define DATAPKT_CTRL_FLUSHQ_DROP_DE     1   /* Drop option: DE only         */
#define DATAPKT_CTRL_FLUSHQ_DROP_ALL    2   /* Drop option: all packets     */

/* HDLC Sync Info Parameters */
#define DATAPKT_CTRL_HDLCSYNC_BO_STATUS 0   /* Status byte offset           */
#define DATAPKT_CTRL_HDLCSYNC_DET       1   /* HDLC sync detected           */
#define DATAPKT_CTRL_HDLCSYNC_LOSS      2   /* HDLC sync loss detected      */

/* Terminate ACK Parameters */
#define DATAPKT_CTRL_TERMACK_BO_STATUS  0   /* Status byte offset           */

#endif  /* _DATAPKT_INCLUDED */
/* nothing after this point */
